"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableBlockVolume = exports.BlockVolumeFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * Block format options for formatting a blank/new BlockVolume.
 *
 * @stability stable
 */
var BlockVolumeFormat;
(function (BlockVolumeFormat) {
    BlockVolumeFormat["EXT3"] = "ext3";
    BlockVolumeFormat["EXT4"] = "ext4";
    BlockVolumeFormat["XFS"] = "xfs";
})(BlockVolumeFormat = exports.BlockVolumeFormat || (exports.BlockVolumeFormat = {}));
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an Amazon Elastic Block Store (EBS) Volume to itself when it is launched.
 *
 * The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *    **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
class MountableBlockVolume {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        var _b;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        this.grantRequiredPermissions(target);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScriptZip = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const volumeFormat = (_b = this.props.volumeFormat) !== null && _b !== void 0 ? _b : BlockVolumeFormat.XFS;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScriptZip}`, `bash ./mountEbsBlockVolume.sh ${this.props.blockVolume.volumeId} ${volumeFormat} ${mountDir} ${mountOptionsStr} ""`, 'popd', `rm -f ${mountScriptZip}`);
    }
    /**
     * Grant required permissions to the target.
     *
     * The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     *
     * @stability stable
     */
    grantRequiredPermissions(target) {
        // Volume.grantAttachVolumeByResourceTag() requires that the target be a construct; it adds a tag to the construct.
        // So, we fail if we're given something that is not compatible.
        if (!core_1.Construct.isConstruct(target)) {
            throw new Error('Target instance must be a construct. It cannot be constructed from attributes.');
        }
        // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
        // Accessed July 2020
        // ec2:DescribeVolumes does not support resource or condition constraints.
        target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ec2:DescribeVolumes',
            ],
            resources: ['*'],
        }));
        this.props.blockVolume.grantAttachVolumeByResourceTag(target.grantPrincipal, [target]);
    }
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton(scope) {
        var _b;
        const stack = core_1.Stack.of(scope);
        const uuid = '01ca4aa6-d440-4f83-84d8-80a5a21fd0e3';
        const uniqueId = 'MountableBlockVolumeAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEbsBlockVolume.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableBlockVolume = MountableBlockVolume;
_a = JSII_RTTI_SYMBOL_1;
MountableBlockVolume[_a] = { fqn: "aws-rfdk.MountableBlockVolume", version: "0.34.0" };
//# sourceMappingURL=data:application/json;base64,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