# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['alpaca',
 'alpaca.broker',
 'alpaca.broker.models',
 'alpaca.common',
 'alpaca.data',
 'alpaca.data.historical',
 'alpaca.data.live',
 'alpaca.data.models',
 'alpaca.trading']

package_data = \
{'': ['*']}

install_requires = \
['msgpack>=1.0.3,<2.0.0',
 'pandas==1.3.5',
 'pydantic>=1.9.0,<2.0.0',
 'requests>=2.27.1,<3.0.0',
 'websockets>=10.2,<11.0']

setup_kwargs = {
    'name': 'alpaca-py',
    'version': '0.4.2',
    'description': 'The Official Python SDK for Alpaca APIs',
    'long_description': '[![Alpaca-py](https://github.com/alpacahq/alpaca-py/blob/master/docs/images/alpaca-py-banner.png?raw=true)](https://alpaca.markets/docs/python-sdk)\n\n[![Downloads](https://pepy.tech/badge/alpaca-py/month)](https://pepy.tech/project/alpaca-py)\n[![Python Versions](https://img.shields.io/pypi/pyversions/alpaca-py.svg?logo=python&logoColor=white)](https://pypi.org/project/alpaca-py)\n[![GitHub](https://img.shields.io/github/license/alpacahq/alpaca-py?color=blue)](https://github.com/alpacahq/alpaca-py/blob/master/LICENSE.md)\n[![PyPI](https://img.shields.io/pypi/v/alpaca-py?color=blue)](https://pypi.org/project/alpaca-py/)\n\n## Table of Contents\n* [About](#about)\n* [Documentation](#documentation)\n* [Installation](#installation)\n* [What\'s New?](#whats-new)\n   1. [Broker API](#broker-api-new)\n   2. [OOP Design](#oop-design)\n   3. [Data Validation](#data-validation)\n   4. [Many Clients](#many-clients)\n* [API Keys](#api-keys)\n   1. [Trading and Market Data API Keys](#trading-api-keys)\n   2. [Broker API Keys](#trading-api-keys)\n* [Usage](#usage)\n   1. [Broker API Example](#broker-api-example)\n   2. [Trading API Example](#trading-api-example)\n   3. [Market Data API Example](#data-api-example)\n* [Contributing](https://github.com/alpacahq/alpaca-py/blob/master/CONTRIBUTING.md)\n* [License](https://github.com/alpacahq/alpaca-py/blob/master/LICENSE)\n\n## About <a name="about"></a>\n\nAlpaca-py provides an interface for interacting with the API products Alpaca offers. These API products are provided as various REST, WebSocket and SSE endpoints that allow you to do everything from streaming market data to creating your own investment apps. \n\nLearn more about the API products Alpaca offers at https://alpaca.markets.\n\n## Documentation <a name="documentation"></a>\n\nAlpaca-py has a supplementary documentation site which contains references for all clients, methods and models found in this codebase. The documentation\nalso contains examples to get started with alpaca-py.\n\nYou can find the documentation site here: https://alpaca.markets/docs/python-sdk\n\n## Installation <a name="installation"></a>\n\nAlpaca-py is supported on Python 3.7+.  You can install Alpaca-py using pip.\n\nRun the following command in your terminal.\n\n```shell\n  pip install alpaca-py\n```\n\n## What’s New? <a name="whats-new"></a>\nIf you’ve used the previous python SDK alpaca-trade-api, there are a few key differences to be aware of.\n\n### Broker API <a name="broker-api-new"></a>\nAlpaca-py lets you use Broker API to start building your investment apps! Learn more at the [Broker](https://alpaca.markets/docs/python-sdk/broker.html) page.\n\n### OOP Design <a name="oop-design"></a>\nAlpaca-py uses a more OOP approach to submitting requests compared to the previous SDK. To submit a request, you will most likely need to create a request object containing the desired request data. Generally, there is a unique request model for each method. \n\nSome examples of request models corresponding to methods: \n\n* ``GetOrdersRequest`` for ``TradingClient.get_orders()``\n* ``CryptoLatestOrderbookRequest`` for ``CryptoHistoricalDataClient.get_crypto_latest_orderbook()``\n\n**Usage Example**\n\nTo submit an order, you will need to provide a `MarketOrderRequest` object.\n\n```python\nfrom alpaca.trading.client import TradingClient\nfrom alpaca.trading.requests import MarketOrderRequest\nfrom alpaca.trading.enums import OrderSide, TimeInForce\n\nclient = TradingClient(\'api-key\', \'secret-key\')\n\nrequest_params = MarketOrderRequest(\n                        symbol="SPY",\n                        qty=3,\n                        side=OrderSide.BUY,\n                        time_in_force=TimeInForce.DAY\n                        )\n\nclient.submit_order(order_data=request_params)\n```\n\n### Data Validation <a name="data-validation"></a>\nAlpaca-py uses *pydantic* to validate data models at run-time. This means if you are receiving request data via JSON from a client. You can handle parsing and validation through Alpaca’s request models. All request models can be instantiated by passing in data in dictionary format.\n\nHere is a rough example of what is possible.\n\n```python\n\n @app.route(\'/post_json\', methods=[\'POST\'])\n def do_trade():\n     # ...\n\n     order_data_json = request.get_json()\n\n     # validate data\n     MarketOrderRequest(**order_data_json)\n\n     # ...\n```\n\n### Many Clients <a name="many-clients"></a>\nAlpaca-py has a lot of client classes. There is a client for each API and even asset class specific clients (``StockHistoricalDataClient``, ``CryptoDataStream``). This requires you to pick and choose clients based on your needs.\n\n**Broker API:** ``BrokerClient``\n\n**Trading API:** ``TradingClient``\n\n**Market Data API:**  ``StockHistoricalDataClient``, ``CryptoHistoricalDataClient``, ``CryptoDataStream``, ``StockDataStream``\n\n## API Keys <a name="api-keys"></a>\n\n### Trading and Market Data API <a name="trading-api-keys"></a>\nIn order to use Alpaca’s services you’ll need to sign up for an Alpaca account and retrieve your API keys. Signing up is completely free and takes only a few minutes. Sandbox environments are available to test out the API. To use the sandbox environment, you will need to provide sandbox/paper keys. API keys are passed into Alpaca-py through either TradingClient, StockHistoricalDataClient, CryptoHistoricalDataClient, StockDataStream, or CryptoDataStream.\n\n### Broker API <a name="broker-api-keys"></a>\nTo use the Broker API, you will need to sign up for a broker account and retrieve your Broker API keys. The API keys can be found on the dashboard once you’ve logged in. Alpaca also provides a sandbox environment to test out Broker API. To use the sandbox mode, provide your sandbox keys. Once you have your keys, you can pass them into BrokerClient to get started.\n\n## Usage <a name="usage"></a>\nAlpaca’s APIs allow you to do everything from building algorithmic trading strategies to building a full brokerage experience for your own end users. Here are some things you can do with Alpaca-py.\n\nTo view full descriptions and examples view the [documentation page](https://alpaca.markets/docs/python-sdk/index.html).\n\n**Market Data API**: Access live and historical market data for 5000+ stocks and 20+ crypto.\n\n**Trading API**: Trade stock and crypto with lightning fast execution speeds.\n\n**Broker API & Connect**: Build investment apps - from robo-advisors to brokerages.\n\n### Broker API Example <a name="broker-api-example"></a>\n\n**Listing All Accounts**\n\nThe BrokerClient::list_accounts method allows you to list all the brokerage accounts under your management. The method takes an optional parameter search_parameters which requires a ListAccountsRequest object. This parameter allows you filter the list of accounts returned.\n\n```python\nfrom alpaca.broker.client import BrokerClient\nfrom alpaca.broker.requests import ListAccountsRequest\nfrom alpaca.broker.enums import AccountEntities\n\nbroker_client = BrokerClient(\'api-key\', \'secret-key\')\n\n# search for accounts created after January 30th 2022.\n# Response should contain Contact and Identity fields for each account.\nfilter = ListAccountsRequest(\n                    created_after=datetime.datetime.strptime("2022-01-30", "%Y-%m-%d"),\n                    entities=[AccountEntities.CONTACT, AccountEntities.IDENTITY]\n                    )\n\naccounts = broker_client.list_accounts(search_parameters=filter)\n```\n\n### Trading API Example <a name="trading-api-example"></a>\n\n**Submitting an Order**\n\nTo create on order on Alpaca-py you must use an OrderRequest object. There are different OrderRequest objects based on the type of order you want to make. For market orders, there is MarketOrderRequest, limit orders have LimitOrderRequest, stop orders StopOrderRequest, and trailing stop orders have TrailingStopOrderRequest. Each order type have their own required parameters for a successful order.\n\n\n```python\nfrom alpaca.trading.client import TradingClient\nfrom alpaca.trading.requests import MarketOrderRequest\nfrom alpaca.trading.enums import OrderSide, TimeInForce\n\ntrading_client = TradingClient(\'api-key\', \'secret-key\')\n\n\n# preparing order data\nmarket_order_data = MarketOrderRequest(\n                      symbol="BTC/USD",\n                      qty=0.0001,\n                      side=OrderSide.BUY,\n                      time_in_force=TimeInForce.DAY\n                  )\n\n# Market order\nmarket_order = trading_client.submit_order(\n                order_data=market_order_data\n                )\n```\n\n\n### Market Data API Example <a name="data-api-example"></a>\n**Querying Historical Bar Data**\n\nYou can request bar data via the HistoricalDataClients. In this example, we query daily bar data for “BTC/USD” and “ETH/USD” since July 1st 2022. You can convert the response to a multi-index pandas dataframe using the .df property.\n\n```python\nfrom alpaca.data.historical import CryptoHistoricalDataClient\nfrom alpaca.data.requests import CryptoBarsRequest\nfrom alpaca.data.timeframe import TimeFrame\n\n# no keys required for crypto data\nclient = CryptoHistoricalDataClient()\n\nrequest_params = CryptoBarsRequest(\n                        symbol_or_symbols=["BTC/USD", "ETH/USD"],\n                        timeframe=TimeFrame.Day,\n                        start="2022-07-01"\n                        )\n\nbars = client.get_crypto_bars(request_params)\n\n# convert to dataframe\nbars.df\n\n```\n',
    'author': 'Rahul Chowdhury',
    'author_email': 'rahul.chowdhury@alpaca.markets',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/alpacahq/alpaca-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
