import click
import re
from click import Abort
from imagination import container
from typing import Optional, Any, Dict
from urllib.parse import urlparse

from dnastack.cli.data_connect.helper import handle_query
from dnastack.cli.helpers.client_factory import ConfigurationBasedClientFactory
from dnastack.cli.helpers.command.decorator import command
from dnastack.cli.helpers.command.group import AliasedGroup
from dnastack.cli.helpers.command.spec import ArgumentSpec, RESOURCE_OUTPUT_SPEC, DATA_OUTPUT_SPEC
from dnastack.cli.helpers.exporter import to_json
from dnastack.cli.helpers.iterator_printer import show_iterator
from dnastack.client.collections.client import CollectionServiceClient
from dnastack.client.data_connect import DataConnectClient
from dnastack.common.logger import get_logger
from dnastack.configuration.manager import ConfigurationManager
from dnastack.configuration.models import Context

_logger = get_logger('cli/collections')


def _get_context(context_name: Optional[str] = None) -> Context:
    config_manager: ConfigurationManager = container.get(ConfigurationManager)
    config = config_manager.load()
    config_context = config.contexts.get(context_name or config.current_context)

    assert config_context is not None, (
        f'The given context ({context_name} is not available.'
        if context_name
        else 'The default context is not initialized.'
    )

    return config_context


def _get(context: Optional[str] = None, id: Optional[str] = None) -> CollectionServiceClient:
    factory: ConfigurationBasedClientFactory = container.get(ConfigurationBasedClientFactory)
    return factory.get(CollectionServiceClient, context_name=context, endpoint_id=id)


def _switch_to_data_connect(context: Context,
                            collection_service_client: CollectionServiceClient,
                            collection_id_or_slug_name: Optional[str],
                            no_auth: bool) -> DataConnectClient:
    default_no_auth_properties = {'authentication': None, 'fallback_authentications': None}

    try:
        proposed_data_connect_endpoint = collection_service_client.data_connect_endpoint(collection_id_or_slug_name,
                                                                                         no_auth=no_auth)

        # Look up for any similar registered service endpoint.
        for endpoint in context.endpoints:
            if proposed_data_connect_endpoint.type == endpoint.type:
                proposed_data_connect_endpoint_url = proposed_data_connect_endpoint.url
                if not proposed_data_connect_endpoint_url.endswith('/'):
                    proposed_data_connect_endpoint_url += '/'

                reference_data_connect_endpoint_url = endpoint.url
                if not reference_data_connect_endpoint_url.endswith('/'):
                    reference_data_connect_endpoint_url += '/'

                if proposed_data_connect_endpoint_url == reference_data_connect_endpoint_url:
                    return DataConnectClient.make(
                        endpoint.copy(update=default_no_auth_properties)
                        if no_auth
                        else endpoint
                    )
            else:
                pass

        _logger.debug(
            f'Unable to find a registered {proposed_data_connect_endpoint.type} endpoint at {proposed_data_connect_endpoint_url}.'
        )

        # Fallback to the endpoint generated by the collection service client.
        return DataConnectClient.make(
            proposed_data_connect_endpoint.copy(update=default_no_auth_properties)
            if no_auth
            else proposed_data_connect_endpoint
        )
    except AssertionError:
        collection_service_client.list_collections(no_auth)
        _abort_with_collection_list(collection_service_client, collection_id_or_slug_name, no_auth=no_auth)


def _abort_with_collection_list(collection_service_client: CollectionServiceClient,
                                collection_id_or_slug_name: Optional[str],
                                no_auth: bool):
    available_identifiers = "\n - ".join(sorted([
        available_collection.slugName
        for available_collection in collection_service_client.list_collections(no_auth=no_auth)
    ]))

    error_message = f'The collection ID or slug name is not given via --collection or -c option or it is invalid. ' \
                    f'(Given: {collection_id_or_slug_name})'

    if available_identifiers:
        raise Abort(f'{error_message}\n\nHere is the list of available collection IDs:\n\n'
                    f' - {available_identifiers}\n')
    else:
        raise Abort(f'{error_message}\n\nHowever, you do not seem to have access to any collection at '
                    f'{collection_service_client.url} or there are no collections available at the moment.\n')


@click.group("collections", cls=AliasedGroup, aliases=['cs'])
def collection_command_group():
    """ Interact with Collection Service or Explorer Service (e.g., Viral AI) """


@command(collection_command_group, 'list', specs=[RESOURCE_OUTPUT_SPEC])
def list_collections(context: Optional[str], endpoint_id: Optional[str], no_auth: bool = False,
                     output: Optional[str] = None):
    """ List collections """
    show_iterator(output, _get(context, endpoint_id).list_collections(no_auth=no_auth))


@command(collection_command_group,
         specs=[
             ArgumentSpec(
                 name='collection',
                 arg_names=['--collection', '-c'],
                 as_option=True,
                 help='The ID or slug name of the target collection; required only by an explorer service',
             ),
             ArgumentSpec(
                 name='limit',
                 arg_names=['--limit', '-l'],
                 as_option=True,
                 help='The maximum number of items to display',
             ),
             RESOURCE_OUTPUT_SPEC
         ])
def list_items(context: Optional[str],
               endpoint_id: Optional[str],
               collection: Optional[str],
               limit: Optional[int] = 50,
               no_auth: bool = False,
               output: Optional[str] = None):
    """ List items of the given collection """
    logger = get_logger('CLI/list-items')
    limit_override = False

    limit = int(limit)  # This is for Python 3.7.

    click.secho(f'Retrieving upto {limit} item{"s" if limit == 1 else ""}...' if limit else 'Retrieving all items...',
                dim=True,
                err=True)

    collection_service_client = _get(context, endpoint_id)

    collection_id = collection.strip() if collection else None
    if not collection_id:
        _abort_with_collection_list(collection_service_client, collection, no_auth=no_auth)

    actual_collection = collection_service_client.get(collection_id, no_auth=no_auth)
    data_connect_client = _switch_to_data_connect(_get_context(context), collection_service_client,
                                                  actual_collection.slugName, no_auth)
    items_query = actual_collection.itemsQuery.strip()

    if re.search(r' limit\s*\d+$', items_query, re.IGNORECASE):
        logger.warning('The items query already has the limit defined and the CLI will not override that limit.')
    else:
        logger.debug(f'Only shows {limit} row(s)')
        limit_override = True
        items_query = f'{items_query} LIMIT {limit + 1}'  # We use +1 as an indicator whether there are more results.

    def __simplify_item(row: Dict[str, Any]) -> Dict[str, Any]:
        # NOTE: It is implemented this way to guarantee that "id" and "name" are more likely to show first.
        property_names = ['type', 'size', 'size_unit', 'version', 'item_updated_at']

        logger.debug(f'Item Simplifier: given: {to_json(row)}')

        item = dict(
            id=row['id'],
            name=row.get('qualified_table_name') or row.get('preferred_name') or row.get('display_name') or row['name'],
        )

        if row['type'] == 'blob':
            property_names.extend([
                'checksums',
                'metadata_url',
                'mime_type',
            ])
        elif row['type'] == 'table':
            property_names.extend([
                'json_schema',
            ])

        item.update({
            k: v
            for k, v in row.items()
            if k in property_names
        })

        # FIXME: Remove this logic when https://www.pivotaltracker.com/story/show/182309558 is resolved.
        if 'metadata_url' in item:
            parsed_url = urlparse(item['metadata_url'])
            item['metadata_url'] = f'{parsed_url.scheme}://{parsed_url.netloc}/{item["id"]}'

        return item

    items = [i for i in data_connect_client.query(items_query, no_auth=no_auth)]
    row_count = len(items)

    displayed_item_count = show_iterator(
        output or RESOURCE_OUTPUT_SPEC.default,
        items,
        __simplify_item,
        limit
    )

    click.secho(f'Displayed {displayed_item_count} item{"s" if displayed_item_count != 1 else ""} from this collection',
                fg='green',
                err=True)

    if limit_override and row_count > limit:
        click.secho(f'There exists more than {limit} item{"s" if limit != 1 else ""} in this collection. You may use '
                    '"--limit 0" to get all items in this collection.\n\n'
                    f'    dnastack collections list-items -c {actual_collection.slugName} --limit=0 '
                    f'{"--no-auth" if no_auth else ""}'
                    '\n',
                    fg='yellow',
                    err=True)


@command(collection_command_group,
         'query',
         [
             ArgumentSpec(
                 name='collection',
                 arg_names=['--collection', '-c'],
                 as_option=True,
                 help='The ID or slug name of the target collection; required only by an explorer service',
             ),
             ArgumentSpec(
                 name='decimal_as',
                 arg_names=['--decimal-as'],
                 as_option=True,
                 help='The format of the decimal value',
                 choices=["string", "float"],
             ),
             DATA_OUTPUT_SPEC
         ])
def query_collection(context: Optional[str],
                     endpoint_id: Optional[str],
                     collection: Optional[str],
                     query: str,
                     decimal_as: str = 'string',
                     no_auth: bool = False,
                     output: Optional[str] = None):
    """ Query data """
    client = _switch_to_data_connect(_get_context(context), _get(context, endpoint_id), collection, no_auth=no_auth)
    return handle_query(client, query, decimal_as=decimal_as, no_auth=no_auth, output_format=output)


@click.group("tables")
def table_command_group():
    """ Data Client API for Collections """


@command(table_command_group,
         'list',
         [
             ArgumentSpec(
                 name='collection',
                 arg_names=['--collection', '-c'],
                 as_option=True,
                 help='The ID or slug name of the target collection; required only by an explorer service',
             ),
             RESOURCE_OUTPUT_SPEC
         ])
def list_tables(context: Optional[str],
                endpoint_id: Optional[str],
                collection: Optional[str],
                no_auth: bool = False,
                output: Optional[str] = None):
    """ List all accessible tables """
    client = _switch_to_data_connect(_get_context(context), _get(context, endpoint_id), collection, no_auth=no_auth)
    show_iterator(output, client.iterate_tables(no_auth=no_auth))


# noinspection PyTypeChecker
collection_command_group.add_command(table_command_group)
