"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsSvcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * Class missing from `@aws-cdk/aws-apigatewayv2-alpha`.
 */
class HttpRouteIntegration extends apigwy.HttpRouteIntegration {
    constructor(id, opts) {
        super(id);
        this.httpIntegrationProps = opts.integrationProps;
        this.integration = opts.integration;
    }
    /**
     * (experimental) Bind this integration to the route.
     *
     * @experimental
     */
    bind(_options) {
        var _b;
        if (this.httpIntegrationProps === undefined) {
            throw new TypeError('bind called without IntegrationProps defined');
        }
        return {
            type: this.httpIntegrationProps.integrationType,
            payloadFormatVersion: (_b = this.httpIntegrationProps.payloadFormatVersion) !== null && _b !== void 0 ? _b : apigwy.PayloadFormatVersion.VERSION_2_0,
            connectionType: this.httpIntegrationProps.connectionType,
            connectionId: this.httpIntegrationProps.connectionId,
            credentials: this.httpIntegrationProps.credentials,
            method: this.httpIntegrationProps.method,
            parameterMapping: this.httpIntegrationProps.parameterMapping,
            secureServerName: this.httpIntegrationProps.secureServerName,
            subtype: this.httpIntegrationProps.integrationSubtype,
            uri: this.httpIntegrationProps.integrationUri,
        };
    }
}
/**
 * (experimental) Create a new MicroApps Services construct, including the Deployer and Router Lambda Functions, and the DynamoDB Table used by both.
 *
 * @experimental
 */
class MicroAppsSvcs extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { bucketApps, bucketAppsOAI, bucketAppsStaging, s3PolicyBypassAROAs = [], s3PolicyBypassPrincipalARNs = [], s3StrictBucketPolicy = false, appEnv, httpApi, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix = '', } = props;
        if (s3StrictBucketPolicy === true) {
            if (s3PolicyBypassAROAs.length === 0 && s3PolicyBypassPrincipalARNs.length === 0) {
                throw new Error('s3StrictBucketPolicy cannot be true without specifying at least one s3PolicyBypassAROAs or s3PolicyBypassPrincipalARNs');
            }
        }
        //
        // DynamoDB Table
        //
        this._table = new dynamodb.Table(this, 'table', {
            tableName: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : undefined,
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: 'PK',
                type: dynamodb.AttributeType.STRING,
            },
            sortKey: {
                name: 'SK',
                type: dynamodb.AttributeType.STRING,
            },
            removalPolicy,
        });
        //
        // Router Lambda Function
        //
        // Create Router Lambda Function
        const routerFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-router${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                DATABASE_TABLE_NAME: this._table.tableName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-router', 'dist', 'index.js'))) {
            // This is for local dev
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'dist')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-router', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-router')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else {
            // Create Router Lambda Layer
            const routerDataFiles = new lambda.LayerVersion(this, 'router-templates', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'templates')),
                removalPolicy,
            });
            this._routerFunc = new lambdaNodejs.NodejsFunction(this, 'router-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-router', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                layers: [routerDataFiles],
                ...routerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._routerFunc.applyRemovalPolicy(removalPolicy);
        }
        const policyReadTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [`${bucketApps.bucketArn}/*`],
        });
        for (const router of [this._routerFunc]) {
            router.addToRolePolicy(policyReadTarget);
            // Give the Router access to DynamoDB table
            this._table.grantReadData(router);
            this._table.grant(router, 'dynamodb:DescribeTable');
        }
        //
        // Deployer Lambda Function
        //
        // Create Deployer Lambda Function
        const iamRoleUploadName = assetNameRoot
            ? `${assetNameRoot}-deployer-upload${assetNameSuffix}`
            : undefined;
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                APIGWY_ID: httpApi.httpApiId,
                DATABASE_TABLE_NAME: this._table.tableName,
                FILESTORE_STAGING_BUCKET: bucketAppsStaging.bucketName,
                FILESTORE_DEST_BUCKET: bucketApps.bucketName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Give the Deployer access to DynamoDB table
        this._table.grantReadWriteData(this._deployerFunc);
        this._table.grant(this._deployerFunc, 'dynamodb:DescribeTable');
        //
        // Deloyer upload temp role
        // Deployer assumes this role with a limited policy to generate
        // an STS temp token to return to microapps-publish for the upload.
        //
        const iamRoleUpload = new iam.Role(this, 'deployer-upload-role', {
            roleName: iamRoleUploadName,
            inlinePolicies: {
                uploadPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['s3:ListBucket'],
                            resources: [bucketAppsStaging.bucketArn],
                        }),
                        new iam.PolicyStatement({
                            actions: ['s3:PutObject', 's3:GetObject', 's3:AbortMultipartUpload'],
                            resources: [`${bucketAppsStaging.bucketArn}/*`],
                        }),
                    ],
                }),
            },
            assumedBy: this._deployerFunc.grantPrincipal,
        });
        this._deployerFunc.addEnvironment('UPLOAD_ROLE_NAME', iamRoleUpload.roleName);
        //
        // Update S3 permissions
        //
        // Create PrincipalARN List
        const s3PolicyBypassArnPrincipals = [];
        for (const arnPrincipal of s3PolicyBypassPrincipalARNs) {
            s3PolicyBypassArnPrincipals.push(new iam.ArnPrincipal(arnPrincipal));
        }
        // Create AROA List that matches assumed sessions
        const s3PolicyBypassAROAMatches = [];
        for (const aroa of s3PolicyBypassAROAs) {
            s3PolicyBypassAROAMatches.push(`${aroa}:*`);
        }
        // Deny apps from reading:
        // - If they are missing the microapp-name tag
        // - Anything outside of the folder that matches their microapp-name tag
        const policyDenyPrefixOutsideTag = new iam.PolicyStatement({
            sid: 'deny-prefix-outside-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                ...s3PolicyBypassArnPrincipals,
                this._deployerFunc.grantPrincipal,
            ],
            notResources: [
                `${bucketApps.bucketArn}/\${aws:PrincipalTag/microapp-name}/*`,
                bucketApps.bucketArn,
            ],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'false' },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyPrefixOutsideTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyDenyMissingTag = new iam.PolicyStatement({
            sid: 'deny-missing-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                // Exclude the Deployer Function directly
                this._deployerFunc.grantPrincipal,
                // 2021-12-04 - Not 100% sure that this is actually needed...
                // Let's test this and remove if actually not necessary
                new iam.ArnPrincipal(`arn:aws:sts::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:assumed-role/${(_b = this._deployerFunc.role) === null || _b === void 0 ? void 0 : _b.roleName}/${this._deployerFunc.functionName}`),
                ...s3PolicyBypassArnPrincipals,
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'true' },
                // Note: This AROA must be specified to prevent this policy from locking
                // out non-root sessions that have assumed the admin role.
                // The notPrincipals will only match the role name exactly and will not match
                // any session that has assumed the role since notPrincipals does not allow
                // wildcard matches and does not do them implicitly either.
                // The AROA must be used because there are only 3 Principal variables:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
                //  aws:username, aws:userid, aws:PrincipalTag
                // For an assumed role, aws:username is blank, aws:userid is:
                //  [unique id AKA AROA for Role]:[session name]
                // Table of unique ID prefixes such as AROA:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
                // The name of the role is simply not available and if it was
                // we'd need to write a complicated comparison to make sure
                // that we didn't exclude the Deny tag from roles in other accounts.
                //
                // To get the AROA with the AWS CLI:
                //   aws iam get-role --role-name ROLE-NAME
                //   aws iam get-user -–user-name USER-NAME
                StringNotLike: { 'aws:userid': [aws_cdk_lib_1.Aws.ACCOUNT_ID, ...s3PolicyBypassAROAMatches] },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyMissingTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyCloudFrontAccess = new iam.PolicyStatement({
            sid: 'cloudfront-oai-access',
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            principals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        if (bucketApps.policy === undefined) {
            const document = new s3.BucketPolicy(this, 's3-policy', {
                bucket: bucketApps,
            }).document;
            document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                document.addStatements(policyDenyPrefixOutsideTag);
                document.addStatements(policyDenyMissingTag);
            }
        }
        else {
            bucketApps.policy.document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                bucketApps.policy.document.addStatements(policyDenyPrefixOutsideTag);
                bucketApps.policy.document.addStatements(policyDenyMissingTag);
            }
        }
        // Allow the Lambda to read from the staging bucket
        const policyReadListStaging = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            // FIXME: Allow Deployer to delete from Staging bucket
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:ListBucket'],
            resources: [`${bucketAppsStaging.bucketArn}/*`, bucketAppsStaging.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadListStaging);
        // Allow the Lambda to write to the target bucket and delete
        const policyReadWriteListTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:PutObject', 's3:ListBucket'],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadWriteListTarget);
        // Allow the deployer to get a temporary STS token
        const policyGetSTSToken = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:GetFederationToken'],
            resources: ['*'],
        });
        this._deployerFunc.addToRolePolicy(policyGetSTSToken);
        // Allow the deployer to assume the upload role
        const policyAssumeUpload = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: [iamRoleUpload.roleArn],
        });
        this._deployerFunc.addToRolePolicy(policyAssumeUpload);
        //
        // Give Deployer permissions to create routes and integrations
        // on the API Gateway API.
        //
        // Grant the ability to List all APIs (we have to find it)
        const policyAPIList = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:GET'],
            resources: [`arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis`],
        });
        this._deployerFunc.addToRolePolicy(policyAPIList);
        // Grant full control over the API we created
        const policyAPIManage = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:*'],
            resources: [
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${httpApi.httpApiId}/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes/*`,
            ],
        });
        this._deployerFunc.addToRolePolicy(policyAPIManage);
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEqualsIfExists: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
        // Create an integration for the Router
        // All traffic without another route goes to the Router
        const intRouter = new apigwy.HttpIntegration(this, 'router-integration', {
            integrationType: apigwy.HttpIntegrationType.AWS_PROXY,
            httpApi,
            integrationUri: this._routerFunc.functionArn,
            payloadFormatVersion: apigwy.PayloadFormatVersion.VERSION_2_0,
        });
        // new apigwycfn.CfnIntegration(this, 'router-integration', {
        //   apiId: httpApi.httpApiId,
        //   integrationType: 'AWS_PROXY',
        //   integrationUri: routerFunc.functionArn,
        // });
        // new apigwycfn.CfnRoute(this, 'route-default', {
        //   apiId: httpApi.httpApiId,
        //   routeKey: apigwy.HttpRouteKey.DEFAULT.key,
        //   target: `integrations/${intRouter.integrationId}`,
        // });
        // This creates an integration and a router
        const route = new apigwy.HttpRoute(this, 'route-default', {
            httpApi,
            routeKey: apigwy.HttpRouteKey.DEFAULT,
            // @ts-expect-error null is needed to prevent this.bind call
            authorizer: apigwy.Auth,
            integration: new HttpRouteIntegration('router-integration', {
                integration: intRouter,
            }),
        });
        let routeArn = route.produceRouteArn(apigwy.HttpMethod.ANY);
        // Remove the trailing `/` on the ARN, which is not correct
        routeArn = routeArn.slice(0, routeArn.length - 1);
        // Grant API Gateway permission to invoke the Lambda
        new lambda.CfnPermission(this, 'router-invoke', {
            action: 'lambda:InvokeFunction',
            functionName: this._routerFunc.functionName,
            principal: 'apigateway.amazonaws.com',
            sourceArn: routeArn,
        });
    }
    /**
     * (experimental) DynamoDB table used by Router, Deployer, and Release console app.
     *
     * @experimental
     */
    get table() {
        return this._table;
    }
    /**
     * (experimental) Lambda function for the Deployer.
     *
     * @experimental
     */
    get deployerFunc() {
        return this._deployerFunc;
    }
    /**
     * (experimental) Lambda function for the Router.
     *
     * @experimental
     */
    get routerFunc() {
        return this._routerFunc;
    }
}
exports.MicroAppsSvcs = MicroAppsSvcs;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsSvcs[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsSvcs", version: "0.2.7" };
//# sourceMappingURL=data:application/json;base64,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