"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsCF = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const cforigins = require("aws-cdk-lib/aws-cloudfront-origins");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const ReverseDomain_1 = require("./utils/ReverseDomain");
/**
 * (experimental) Create a new MicroApps CloudFront Distribution.
 *
 * @experimental
 */
class MicroAppsCF extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        const { domainNameEdge, domainNameOrigin, httpApi, removalPolicy, certEdge, assetNameRoot, assetNameSuffix, r53Zone, bucketLogs, bucketAppsOrigin, rootPathPrefix, createAPIPathRoute = true, } = props;
        const apigwyOriginRequestPolicy = MicroAppsCF.createAPIOriginPolicy(this, {
            assetNameRoot,
            assetNameSuffix,
            domainNameEdge,
        });
        //
        // Determine URL of the origin FQDN
        //
        let httpOriginFQDN = 'invalid.pwrdrvr.com';
        if (domainNameOrigin !== undefined) {
            httpOriginFQDN = domainNameOrigin;
        }
        else {
            httpOriginFQDN = `${httpApi.apiId}.execute-api.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`;
        }
        //
        // CloudFront Distro
        //
        const apiGwyOrigin = new cforigins.HttpOrigin(httpOriginFQDN, {
            protocolPolicy: cf.OriginProtocolPolicy.HTTPS_ONLY,
            originSslProtocols: [cf.OriginSslPolicy.TLS_V1_2],
        });
        this._cloudFrontDistro = new cf.Distribution(this, 'cft', {
            comment: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : domainNameEdge,
            domainNames: domainNameEdge !== undefined ? [domainNameEdge] : undefined,
            certificate: certEdge,
            httpVersion: cf.HttpVersion.HTTP2,
            defaultBehavior: {
                allowedMethods: cf.AllowedMethods.ALLOW_ALL,
                cachePolicy: cf.CachePolicy.CACHING_DISABLED,
                compress: true,
                originRequestPolicy: apigwyOriginRequestPolicy,
                origin: apiGwyOrigin,
                viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            enableIpv6: true,
            priceClass: cf.PriceClass.PRICE_CLASS_100,
            logBucket: bucketLogs,
            logFilePrefix: props.domainNameEdge
                ? `${ReverseDomain_1.reverseDomain(props.domainNameEdge)}/cloudfront-raw/`
                : undefined,
        });
        if (removalPolicy !== undefined) {
            this._cloudFrontDistro.applyRemovalPolicy(removalPolicy);
        }
        // Add routes to the CloudFront Distribution
        MicroAppsCF.addRoutes(scope, {
            apiGwyOrigin,
            bucketAppsOrigin,
            distro: this._cloudFrontDistro,
            apigwyOriginRequestPolicy: apigwyOriginRequestPolicy,
            rootPathPrefix,
            createAPIPathRoute,
        });
        //
        // Create the edge name for the CloudFront distro
        //
        if (r53Zone !== undefined) {
            const rrAppsEdge = new r53.RecordSet(this, 'edge-arecord', {
                recordName: domainNameEdge,
                recordType: r53.RecordType.A,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this._cloudFrontDistro)),
                zone: r53Zone,
            });
            if (removalPolicy !== undefined) {
                rrAppsEdge.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    /**
     * (experimental) Create or get the origin request policy.
     *
     * If a custom domain name is NOT used for the origin then a policy
     * will be created.
     *
     * If a custom domain name IS used for the origin then the ALL_VIEWER
     * policy will be returned.  This policy passes the Host header to the
     * origin, which is fine when using a custom domain name on the origin.
     *
     * @experimental
     */
    static createAPIOriginPolicy(scope, props) {
        const { assetNameRoot, assetNameSuffix, domainNameEdge } = props;
        let apigwyOriginRequestPolicy = cf.OriginRequestPolicy.ALL_VIEWER;
        if (domainNameEdge === undefined) {
            // When not using a custom domain name we must limit down the origin policy to
            // prevent it from passing the Host header (distribution_id.cloudfront.net) to
            // apigwy which will then reject it with a 403 because it does not match the
            // execute-api name that apigwy is expecting.
            //
            // 2021-12-28 - There is a bug in the name generation that causes the same asset
            // in different stacks to have the same generated name.  We have to make the id
            // in all cases to ensure the generated name is unique.
            apigwyOriginRequestPolicy = new cf.OriginRequestPolicy(scope, `apigwy-origin-policy-${aws_cdk_lib_1.Stack.of(scope).stackName}`, {
                comment: assetNameRoot ? `${assetNameRoot}-apigwy${assetNameSuffix}` : undefined,
                originRequestPolicyName: assetNameRoot
                    ? `${assetNameRoot}-apigwy${assetNameSuffix}`
                    : undefined,
                cookieBehavior: cf.OriginRequestCookieBehavior.all(),
                queryStringBehavior: cf.OriginRequestQueryStringBehavior.all(),
                headerBehavior: cf.OriginRequestHeaderBehavior.allowList('user-agent', 'referer'),
            });
        }
        return apigwyOriginRequestPolicy;
    }
    /**
     * (experimental) Add API Gateway and S3 routes to an existing CloudFront Distribution.
     *
     * @experimental
     */
    static addRoutes(_scope, props) {
        const { apiGwyOrigin, bucketAppsOrigin, distro, apigwyOriginRequestPolicy, rootPathPrefix = '', createAPIPathRoute = true, } = props;
        //
        // Add Behaviors
        //
        const s3BehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
            cachePolicy: cf.CachePolicy.CACHING_OPTIMIZED,
            compress: true,
            originRequestPolicy: cf.OriginRequestPolicy.CORS_S3_ORIGIN,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        const apiGwyBehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_ALL,
            // TODO: Caching needs to be set by the app response
            cachePolicy: cf.CachePolicy.CACHING_DISABLED,
            compress: true,
            originRequestPolicy: apigwyOriginRequestPolicy,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        //
        // If a route specifically has `/api/` in it, send it to API Gateway
        // This is needed to catch routes that have periods in the API path data,
        // such as: /release/0.0.0/api/update/default/release/0.0.0
        //
        if (createAPIPathRoute) {
            distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/api/*'), apiGwyOrigin, apiGwyBehaviorOptions);
        }
        //
        // All static assets are assumed to have a dot in them
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/*.*'), bucketAppsOrigin, s3BehaviorOptions);
        //
        // Everything that isn't a static asset is going to API Gateway
        // There is no trailing slash because Serverless Next.js wants
        // go load pages at /release/0.0.3 (with no trailing slash).
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*'), apiGwyOrigin, apiGwyBehaviorOptions);
    }
    /**
     * @experimental
     */
    get cloudFrontDistro() {
        return this._cloudFrontDistro;
    }
}
exports.MicroAppsCF = MicroAppsCF;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsCF[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsCF", version: "0.2.7" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiTWljcm9BcHBzQ0YuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvTWljcm9BcHBzQ0YudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwrQkFBMEM7QUFFMUMsNkNBQXdEO0FBR3hELGlEQUFpRDtBQUNqRCxnRUFBZ0U7QUFDaEUsK0NBQStDO0FBQy9DLDhEQUE4RDtBQUU5RCwyQ0FBdUM7QUFDdkMseURBQXNEOzs7Ozs7QUFnRnRELE1BQWEsV0FBWSxTQUFRLHNCQUFTOzs7O0lBc0d4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztTQUN0QztRQUVELElBQ0UsQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsY0FBYyxLQUFLLFNBQVMsQ0FBQztZQUNuRSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFDLEVBQ25FO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQzlGO1FBRUQsTUFBTSxFQUNKLGNBQWMsRUFDZCxnQkFBZ0IsRUFDaEIsT0FBTyxFQUNQLGFBQWEsRUFDYixRQUFRLEVBQ1IsYUFBYSxFQUNiLGVBQWUsRUFDZixPQUFPLEVBQ1AsVUFBVSxFQUNWLGdCQUFnQixFQUNoQixjQUFjLEVBQ2Qsa0JBQWtCLEdBQUcsSUFBSSxHQUMxQixHQUFHLEtBQUssQ0FBQztRQUVWLE1BQU0seUJBQXlCLEdBQUcsV0FBVyxDQUFDLHFCQUFxQixDQUFDLElBQUksRUFBRTtZQUN4RSxhQUFhO1lBQ2IsZUFBZTtZQUNmLGNBQWM7U0FDZixDQUFDLENBQUM7UUFFSCxFQUFFO1FBQ0YsbUNBQW1DO1FBQ25DLEVBQUU7UUFDRixJQUFJLGNBQWMsR0FBVyxxQkFBcUIsQ0FBQztRQUNuRCxJQUFJLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUNsQyxjQUFjLEdBQUcsZ0JBQWdCLENBQUM7U0FDbkM7YUFBTTtZQUNMLGNBQWMsR0FBRyxHQUFHLE9BQU8sQ0FBQyxLQUFLLGdCQUFnQixpQkFBRyxDQUFDLE1BQU0sZ0JBQWdCLENBQUM7U0FDN0U7UUFFRCxFQUFFO1FBQ0Ysb0JBQW9CO1FBQ3BCLEVBQUU7UUFDRixNQUFNLFlBQVksR0FBRyxJQUFJLFNBQVMsQ0FBQyxVQUFVLENBQUMsY0FBYyxFQUFFO1lBQzVELGNBQWMsRUFBRSxFQUFFLENBQUMsb0JBQW9CLENBQUMsVUFBVTtZQUNsRCxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDO1NBQ2xELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtZQUN4RCxPQUFPLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQyxHQUFHLGFBQWEsR0FBRyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYztZQUM5RSxXQUFXLEVBQUUsY0FBYyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN4RSxXQUFXLEVBQUUsUUFBUTtZQUNyQixXQUFXLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxLQUFLO1lBQ2pDLGVBQWUsRUFBRTtnQkFDZixjQUFjLEVBQUUsRUFBRSxDQUFDLGNBQWMsQ0FBQyxTQUFTO2dCQUMzQyxXQUFXLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0I7Z0JBQzVDLFFBQVEsRUFBRSxJQUFJO2dCQUNkLG1CQUFtQixFQUFFLHlCQUF5QjtnQkFDOUMsTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLG9CQUFvQixFQUFFLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUI7YUFDaEU7WUFDRCxVQUFVLEVBQUUsSUFBSTtZQUNoQixVQUFVLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxlQUFlO1lBQ3pDLFNBQVMsRUFBRSxVQUFVO1lBQ3JCLGFBQWEsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDakMsQ0FBQyxDQUFDLEdBQUcsNkJBQWEsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLGtCQUFrQjtnQkFDMUQsQ0FBQyxDQUFDLFNBQVM7U0FDZCxDQUFDLENBQUM7UUFDSCxJQUFJLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDL0IsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsNENBQTRDO1FBQzVDLFdBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFO1lBQzNCLFlBQVk7WUFDWixnQkFBZ0I7WUFDaEIsTUFBTSxFQUFFLElBQUksQ0FBQyxpQkFBaUI7WUFDOUIseUJBQXlCLEVBQUUseUJBQXlCO1lBQ3BELGNBQWM7WUFDZCxrQkFBa0I7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsRUFBRTtRQUNGLGlEQUFpRDtRQUNqRCxFQUFFO1FBRUYsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFO1lBQ3pCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUN6RCxVQUFVLEVBQUUsY0FBYztnQkFDMUIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDNUIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksVUFBVSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO2dCQUMzRixJQUFJLEVBQUUsT0FBTzthQUNkLENBQUMsQ0FBQztZQUNILElBQUksYUFBYSxLQUFLLFNBQVMsRUFBRTtnQkFDL0IsVUFBVSxDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO2FBQzlDO1NBQ0Y7SUFDSCxDQUFDOzs7Ozs7Ozs7Ozs7O0lBek1NLE1BQU0sQ0FBQyxxQkFBcUIsQ0FDakMsS0FBZ0IsRUFDaEIsS0FBbUM7UUFFbkMsTUFBTSxFQUFFLGFBQWEsRUFBRSxlQUFlLEVBQUUsY0FBYyxFQUFFLEdBQUcsS0FBSyxDQUFDO1FBRWpFLElBQUkseUJBQXlCLEdBQTRCLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUM7UUFDM0YsSUFBSSxjQUFjLEtBQUssU0FBUyxFQUFFO1lBQ2hDLDhFQUE4RTtZQUM5RSw4RUFBOEU7WUFDOUUsNEVBQTRFO1lBQzVFLDZDQUE2QztZQUM3QyxFQUFFO1lBQ0YsZ0ZBQWdGO1lBQ2hGLCtFQUErRTtZQUMvRSx1REFBdUQ7WUFDdkQseUJBQXlCLEdBQUcsSUFBSSxFQUFFLENBQUMsbUJBQW1CLENBQ3BELEtBQUssRUFDTCx3QkFBd0IsbUJBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQ25EO2dCQUNFLE9BQU8sRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDLEdBQUcsYUFBYSxVQUFVLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUVoRix1QkFBdUIsRUFBRSxhQUFhO29CQUNwQyxDQUFDLENBQUMsR0FBRyxhQUFhLFVBQVUsZUFBZSxFQUFFO29CQUM3QyxDQUFDLENBQUMsU0FBUztnQkFDYixjQUFjLEVBQUUsRUFBRSxDQUFDLDJCQUEyQixDQUFDLEdBQUcsRUFBRTtnQkFDcEQsbUJBQW1CLEVBQUUsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLEdBQUcsRUFBRTtnQkFDOUQsY0FBYyxFQUFFLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxTQUFTLENBQUMsWUFBWSxFQUFFLFNBQVMsQ0FBQzthQUNsRixDQUNGLENBQUM7U0FDSDtRQUVELE9BQU8seUJBQXlCLENBQUM7SUFDbkMsQ0FBQzs7Ozs7O0lBR00sTUFBTSxDQUFDLFNBQVMsQ0FBQyxNQUFpQixFQUFFLEtBQXVCO1FBQ2hFLE1BQU0sRUFDSixZQUFZLEVBQ1osZ0JBQWdCLEVBQ2hCLE1BQU0sRUFDTix5QkFBeUIsRUFDekIsY0FBYyxHQUFHLEVBQUUsRUFDbkIsa0JBQWtCLEdBQUcsSUFBSSxHQUMxQixHQUFHLEtBQUssQ0FBQztRQUVWLEVBQUU7UUFDRixnQkFBZ0I7UUFDaEIsRUFBRTtRQUNGLE1BQU0saUJBQWlCLEdBQTBCO1lBQy9DLGNBQWMsRUFBRSxFQUFFLENBQUMsY0FBYyxDQUFDLHNCQUFzQjtZQUN4RCxXQUFXLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxpQkFBaUI7WUFDN0MsUUFBUSxFQUFFLElBQUk7WUFDZCxtQkFBbUIsRUFBRSxFQUFFLENBQUMsbUJBQW1CLENBQUMsY0FBYztZQUMxRCxvQkFBb0IsRUFBRSxFQUFFLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCO1NBQ2hFLENBQUM7UUFDRixNQUFNLHFCQUFxQixHQUEwQjtZQUNuRCxjQUFjLEVBQUUsRUFBRSxDQUFDLGNBQWMsQ0FBQyxTQUFTO1lBQzNDLG9EQUFvRDtZQUNwRCxXQUFXLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0I7WUFDNUMsUUFBUSxFQUFFLElBQUk7WUFDZCxtQkFBbUIsRUFBRSx5QkFBeUI7WUFDOUMsb0JBQW9CLEVBQUUsRUFBRSxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQjtTQUNoRSxDQUFDO1FBRUYsRUFBRTtRQUNGLG9FQUFvRTtRQUNwRSx5RUFBeUU7UUFDekUsMkRBQTJEO1FBQzNELEVBQUU7UUFDRixJQUFJLGtCQUFrQixFQUFFO1lBQ3RCLE1BQU0sQ0FBQyxXQUFXLENBQ2hCLFlBQVMsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFLFlBQVksQ0FBQyxFQUM1QyxZQUFZLEVBQ1oscUJBQXFCLENBQ3RCLENBQUM7U0FDSDtRQUVELEVBQUU7UUFDRixzREFBc0Q7UUFDdEQsRUFBRTtRQUNGLE1BQU0sQ0FBQyxXQUFXLENBQ2hCLFlBQVMsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFLFVBQVUsQ0FBQyxFQUMxQyxnQkFBZ0IsRUFDaEIsaUJBQWlCLENBQ2xCLENBQUM7UUFFRixFQUFFO1FBQ0YsK0RBQStEO1FBQy9ELDhEQUE4RDtRQUM5RCw0REFBNEQ7UUFDNUQsRUFBRTtRQUNGLE1BQU0sQ0FBQyxXQUFXLENBQUMsWUFBUyxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsSUFBSSxDQUFDLEVBQUUsWUFBWSxFQUFFLHFCQUFxQixDQUFDLENBQUM7SUFDaEcsQ0FBQzs7OztJQUdELElBQVcsZ0JBQWdCO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDO0lBQ2hDLENBQUM7O0FBcEdILGtDQTRNQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHBvc2l4IGFzIHBvc2l4UGF0aCB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgYXBpZ3d5IGZyb20gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5djItYWxwaGEnO1xuaW1wb3J0IHsgQXdzLCBSZW1vdmFsUG9saWN5LCBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbi8vIGltcG9ydCAqIGFzIGFwaWd3eWNmbiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheXYyJztcbmltcG9ydCAqIGFzIGFjbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCAqIGFzIGNmIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZGZyb250JztcbmltcG9ydCAqIGFzIGNmb3JpZ2lucyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWRmcm9udC1vcmlnaW5zJztcbmltcG9ydCAqIGFzIHI1MyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtcm91dGU1Myc7XG5pbXBvcnQgKiBhcyByNTN0YXJnZXRzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzLXRhcmdldHMnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgcmV2ZXJzZURvbWFpbiB9IGZyb20gJy4vdXRpbHMvUmV2ZXJzZURvbWFpbic7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElNaWNyb0FwcHNDRiB7XG4gIHJlYWRvbmx5IGNsb3VkRnJvbnREaXN0cm86IGNmLkRpc3RyaWJ1dGlvbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTWljcm9BcHBzQ0ZQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXRBcHBzT3JpZ2luOiBjZm9yaWdpbnMuUzNPcmlnaW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVja2V0TG9ncz86IHMzLklCdWNrZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVFZGdlPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVPcmlnaW4/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaHR0cEFwaTogYXBpZ3d5Lkh0dHBBcGk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXNzZXROYW1lUm9vdD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXNzZXROYW1lU3VmZml4Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2VydEVkZ2U/OiBhY20uSUNlcnRpZmljYXRlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcjUzWm9uZT86IHI1My5JSG9zdGVkWm9uZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcm9vdFBhdGhQcmVmaXg/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNyZWF0ZUFQSVBhdGhSb3V0ZT86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIENyZWF0ZUFQSU9yaWdpblBvbGljeU9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhc3NldE5hbWVSb290Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhc3NldE5hbWVTdWZmaXg/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVFZGdlPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFkZFJvdXRlc09wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwaUd3eU9yaWdpbjogY2YuSU9yaWdpbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVja2V0QXBwc09yaWdpbjogY2ZvcmlnaW5zLlMzT3JpZ2luO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRpc3RybzogY2YuRGlzdHJpYnV0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwaWd3eU9yaWdpblJlcXVlc3RQb2xpY3k6IGNmLklPcmlnaW5SZXF1ZXN0UG9saWN5O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb290UGF0aFByZWZpeD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY3JlYXRlQVBJUGF0aFJvdXRlPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE1pY3JvQXBwc0NGIGV4dGVuZHMgQ29uc3RydWN0IGltcGxlbWVudHMgSU1pY3JvQXBwc0NGIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjcmVhdGVBUElPcmlnaW5Qb2xpY3koXG4gICAgc2NvcGU6IENvbnN0cnVjdCxcbiAgICBwcm9wczogQ3JlYXRlQVBJT3JpZ2luUG9saWN5T3B0aW9ucyxcbiAgKTogY2YuSU9yaWdpblJlcXVlc3RQb2xpY3kge1xuICAgIGNvbnN0IHsgYXNzZXROYW1lUm9vdCwgYXNzZXROYW1lU3VmZml4LCBkb21haW5OYW1lRWRnZSB9ID0gcHJvcHM7XG5cbiAgICBsZXQgYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeTogY2YuSU9yaWdpblJlcXVlc3RQb2xpY3kgPSBjZi5PcmlnaW5SZXF1ZXN0UG9saWN5LkFMTF9WSUVXRVI7XG4gICAgaWYgKGRvbWFpbk5hbWVFZGdlID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIFdoZW4gbm90IHVzaW5nIGEgY3VzdG9tIGRvbWFpbiBuYW1lIHdlIG11c3QgbGltaXQgZG93biB0aGUgb3JpZ2luIHBvbGljeSB0b1xuICAgICAgLy8gcHJldmVudCBpdCBmcm9tIHBhc3NpbmcgdGhlIEhvc3QgaGVhZGVyIChkaXN0cmlidXRpb25faWQuY2xvdWRmcm9udC5uZXQpIHRvXG4gICAgICAvLyBhcGlnd3kgd2hpY2ggd2lsbCB0aGVuIHJlamVjdCBpdCB3aXRoIGEgNDAzIGJlY2F1c2UgaXQgZG9lcyBub3QgbWF0Y2ggdGhlXG4gICAgICAvLyBleGVjdXRlLWFwaSBuYW1lIHRoYXQgYXBpZ3d5IGlzIGV4cGVjdGluZy5cbiAgICAgIC8vXG4gICAgICAvLyAyMDIxLTEyLTI4IC0gVGhlcmUgaXMgYSBidWcgaW4gdGhlIG5hbWUgZ2VuZXJhdGlvbiB0aGF0IGNhdXNlcyB0aGUgc2FtZSBhc3NldFxuICAgICAgLy8gaW4gZGlmZmVyZW50IHN0YWNrcyB0byBoYXZlIHRoZSBzYW1lIGdlbmVyYXRlZCBuYW1lLiAgV2UgaGF2ZSB0byBtYWtlIHRoZSBpZFxuICAgICAgLy8gaW4gYWxsIGNhc2VzIHRvIGVuc3VyZSB0aGUgZ2VuZXJhdGVkIG5hbWUgaXMgdW5pcXVlLlxuICAgICAgYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSA9IG5ldyBjZi5PcmlnaW5SZXF1ZXN0UG9saWN5KFxuICAgICAgICBzY29wZSxcbiAgICAgICAgYGFwaWd3eS1vcmlnaW4tcG9saWN5LSR7U3RhY2sub2Yoc2NvcGUpLnN0YWNrTmFtZX1gLFxuICAgICAgICB7XG4gICAgICAgICAgY29tbWVudDogYXNzZXROYW1lUm9vdCA/IGAke2Fzc2V0TmFtZVJvb3R9LWFwaWd3eSR7YXNzZXROYW1lU3VmZml4fWAgOiB1bmRlZmluZWQsXG5cbiAgICAgICAgICBvcmlnaW5SZXF1ZXN0UG9saWN5TmFtZTogYXNzZXROYW1lUm9vdFxuICAgICAgICAgICAgPyBgJHthc3NldE5hbWVSb290fS1hcGlnd3kke2Fzc2V0TmFtZVN1ZmZpeH1gXG4gICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICBjb29raWVCZWhhdmlvcjogY2YuT3JpZ2luUmVxdWVzdENvb2tpZUJlaGF2aW9yLmFsbCgpLFxuICAgICAgICAgIHF1ZXJ5U3RyaW5nQmVoYXZpb3I6IGNmLk9yaWdpblJlcXVlc3RRdWVyeVN0cmluZ0JlaGF2aW9yLmFsbCgpLFxuICAgICAgICAgIGhlYWRlckJlaGF2aW9yOiBjZi5PcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IuYWxsb3dMaXN0KCd1c2VyLWFnZW50JywgJ3JlZmVyZXInKSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGFwaWd3eU9yaWdpblJlcXVlc3RQb2xpY3k7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhZGRSb3V0ZXMoX3Njb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBBZGRSb3V0ZXNPcHRpb25zKSB7XG4gICAgY29uc3Qge1xuICAgICAgYXBpR3d5T3JpZ2luLFxuICAgICAgYnVja2V0QXBwc09yaWdpbixcbiAgICAgIGRpc3RybyxcbiAgICAgIGFwaWd3eU9yaWdpblJlcXVlc3RQb2xpY3ksXG4gICAgICByb290UGF0aFByZWZpeCA9ICcnLFxuICAgICAgY3JlYXRlQVBJUGF0aFJvdXRlID0gdHJ1ZSxcbiAgICB9ID0gcHJvcHM7XG5cbiAgICAvL1xuICAgIC8vIEFkZCBCZWhhdmlvcnNcbiAgICAvL1xuICAgIGNvbnN0IHMzQmVoYXZpb3JPcHRpb25zOiBjZi5BZGRCZWhhdmlvck9wdGlvbnMgPSB7XG4gICAgICBhbGxvd2VkTWV0aG9kczogY2YuQWxsb3dlZE1ldGhvZHMuQUxMT1dfR0VUX0hFQURfT1BUSU9OUyxcbiAgICAgIGNhY2hlUG9saWN5OiBjZi5DYWNoZVBvbGljeS5DQUNISU5HX09QVElNSVpFRCxcbiAgICAgIGNvbXByZXNzOiB0cnVlLFxuICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeTogY2YuT3JpZ2luUmVxdWVzdFBvbGljeS5DT1JTX1MzX09SSUdJTixcbiAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBjZi5WaWV3ZXJQcm90b2NvbFBvbGljeS5SRURJUkVDVF9UT19IVFRQUyxcbiAgICB9O1xuICAgIGNvbnN0IGFwaUd3eUJlaGF2aW9yT3B0aW9uczogY2YuQWRkQmVoYXZpb3JPcHRpb25zID0ge1xuICAgICAgYWxsb3dlZE1ldGhvZHM6IGNmLkFsbG93ZWRNZXRob2RzLkFMTE9XX0FMTCxcbiAgICAgIC8vIFRPRE86IENhY2hpbmcgbmVlZHMgdG8gYmUgc2V0IGJ5IHRoZSBhcHAgcmVzcG9uc2VcbiAgICAgIGNhY2hlUG9saWN5OiBjZi5DYWNoZVBvbGljeS5DQUNISU5HX0RJU0FCTEVELFxuICAgICAgY29tcHJlc3M6IHRydWUsXG4gICAgICBvcmlnaW5SZXF1ZXN0UG9saWN5OiBhcGlnd3lPcmlnaW5SZXF1ZXN0UG9saWN5LFxuICAgICAgdmlld2VyUHJvdG9jb2xQb2xpY3k6IGNmLlZpZXdlclByb3RvY29sUG9saWN5LlJFRElSRUNUX1RPX0hUVFBTLFxuICAgIH07XG5cbiAgICAvL1xuICAgIC8vIElmIGEgcm91dGUgc3BlY2lmaWNhbGx5IGhhcyBgL2FwaS9gIGluIGl0LCBzZW5kIGl0IHRvIEFQSSBHYXRld2F5XG4gICAgLy8gVGhpcyBpcyBuZWVkZWQgdG8gY2F0Y2ggcm91dGVzIHRoYXQgaGF2ZSBwZXJpb2RzIGluIHRoZSBBUEkgcGF0aCBkYXRhLFxuICAgIC8vIHN1Y2ggYXM6IC9yZWxlYXNlLzAuMC4wL2FwaS91cGRhdGUvZGVmYXVsdC9yZWxlYXNlLzAuMC4wXG4gICAgLy9cbiAgICBpZiAoY3JlYXRlQVBJUGF0aFJvdXRlKSB7XG4gICAgICBkaXN0cm8uYWRkQmVoYXZpb3IoXG4gICAgICAgIHBvc2l4UGF0aC5qb2luKHJvb3RQYXRoUHJlZml4LCAnLyovKi9hcGkvKicpLFxuICAgICAgICBhcGlHd3lPcmlnaW4sXG4gICAgICAgIGFwaUd3eUJlaGF2aW9yT3B0aW9ucyxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgLy9cbiAgICAvLyBBbGwgc3RhdGljIGFzc2V0cyBhcmUgYXNzdW1lZCB0byBoYXZlIGEgZG90IGluIHRoZW1cbiAgICAvL1xuICAgIGRpc3Ryby5hZGRCZWhhdmlvcihcbiAgICAgIHBvc2l4UGF0aC5qb2luKHJvb3RQYXRoUHJlZml4LCAnLyovKi8qLionKSxcbiAgICAgIGJ1Y2tldEFwcHNPcmlnaW4sXG4gICAgICBzM0JlaGF2aW9yT3B0aW9ucyxcbiAgICApO1xuXG4gICAgLy9cbiAgICAvLyBFdmVyeXRoaW5nIHRoYXQgaXNuJ3QgYSBzdGF0aWMgYXNzZXQgaXMgZ29pbmcgdG8gQVBJIEdhdGV3YXlcbiAgICAvLyBUaGVyZSBpcyBubyB0cmFpbGluZyBzbGFzaCBiZWNhdXNlIFNlcnZlcmxlc3MgTmV4dC5qcyB3YW50c1xuICAgIC8vIGdvIGxvYWQgcGFnZXMgYXQgL3JlbGVhc2UvMC4wLjMgKHdpdGggbm8gdHJhaWxpbmcgc2xhc2gpLlxuICAgIC8vXG4gICAgZGlzdHJvLmFkZEJlaGF2aW9yKHBvc2l4UGF0aC5qb2luKHJvb3RQYXRoUHJlZml4LCAnLyonKSwgYXBpR3d5T3JpZ2luLCBhcGlHd3lCZWhhdmlvck9wdGlvbnMpO1xuICB9XG5cbiAgcHJpdmF0ZSBfY2xvdWRGcm9udERpc3RybzogY2YuRGlzdHJpYnV0aW9uO1xuICBwdWJsaWMgZ2V0IGNsb3VkRnJvbnREaXN0cm8oKTogY2YuRGlzdHJpYnV0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5fY2xvdWRGcm9udERpc3RybztcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBNaWNyb0FwcHNDRlByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChwcm9wcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3Byb3BzIG11c3QgYmUgc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgKHByb3BzLnI1M1pvbmUgPT09IHVuZGVmaW5lZCAmJiBwcm9wcy5kb21haW5OYW1lRWRnZSAhPT0gdW5kZWZpbmVkKSB8fFxuICAgICAgKHByb3BzLnI1M1pvbmUgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5kb21haW5OYW1lRWRnZSA9PT0gdW5kZWZpbmVkKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJZiBlaXRoZXIgb2YgcjUzWm9uZSBvciBkb21haW5OYW1lRWRnZSBhcmUgc2V0IHRoZW4gdGhlIG90aGVyIG11c3QgYmUgc2V0Jyk7XG4gICAgfVxuXG4gICAgY29uc3Qge1xuICAgICAgZG9tYWluTmFtZUVkZ2UsXG4gICAgICBkb21haW5OYW1lT3JpZ2luLFxuICAgICAgaHR0cEFwaSxcbiAgICAgIHJlbW92YWxQb2xpY3ksXG4gICAgICBjZXJ0RWRnZSxcbiAgICAgIGFzc2V0TmFtZVJvb3QsXG4gICAgICBhc3NldE5hbWVTdWZmaXgsXG4gICAgICByNTNab25lLFxuICAgICAgYnVja2V0TG9ncyxcbiAgICAgIGJ1Y2tldEFwcHNPcmlnaW4sXG4gICAgICByb290UGF0aFByZWZpeCxcbiAgICAgIGNyZWF0ZUFQSVBhdGhSb3V0ZSA9IHRydWUsXG4gICAgfSA9IHByb3BzO1xuXG4gICAgY29uc3QgYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSA9IE1pY3JvQXBwc0NGLmNyZWF0ZUFQSU9yaWdpblBvbGljeSh0aGlzLCB7XG4gICAgICBhc3NldE5hbWVSb290LFxuICAgICAgYXNzZXROYW1lU3VmZml4LFxuICAgICAgZG9tYWluTmFtZUVkZ2UsXG4gICAgfSk7XG5cbiAgICAvL1xuICAgIC8vIERldGVybWluZSBVUkwgb2YgdGhlIG9yaWdpbiBGUUROXG4gICAgLy9cbiAgICBsZXQgaHR0cE9yaWdpbkZRRE46IHN0cmluZyA9ICdpbnZhbGlkLnB3cmRydnIuY29tJztcbiAgICBpZiAoZG9tYWluTmFtZU9yaWdpbiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBodHRwT3JpZ2luRlFETiA9IGRvbWFpbk5hbWVPcmlnaW47XG4gICAgfSBlbHNlIHtcbiAgICAgIGh0dHBPcmlnaW5GUUROID0gYCR7aHR0cEFwaS5hcGlJZH0uZXhlY3V0ZS1hcGkuJHtBd3MuUkVHSU9OfS5hbWF6b25hd3MuY29tYDtcbiAgICB9XG5cbiAgICAvL1xuICAgIC8vIENsb3VkRnJvbnQgRGlzdHJvXG4gICAgLy9cbiAgICBjb25zdCBhcGlHd3lPcmlnaW4gPSBuZXcgY2ZvcmlnaW5zLkh0dHBPcmlnaW4oaHR0cE9yaWdpbkZRRE4sIHtcbiAgICAgIHByb3RvY29sUG9saWN5OiBjZi5PcmlnaW5Qcm90b2NvbFBvbGljeS5IVFRQU19PTkxZLFxuICAgICAgb3JpZ2luU3NsUHJvdG9jb2xzOiBbY2YuT3JpZ2luU3NsUG9saWN5LlRMU19WMV8yXSxcbiAgICB9KTtcbiAgICB0aGlzLl9jbG91ZEZyb250RGlzdHJvID0gbmV3IGNmLkRpc3RyaWJ1dGlvbih0aGlzLCAnY2Z0Jywge1xuICAgICAgY29tbWVudDogYXNzZXROYW1lUm9vdCA/IGAke2Fzc2V0TmFtZVJvb3R9JHthc3NldE5hbWVTdWZmaXh9YCA6IGRvbWFpbk5hbWVFZGdlLFxuICAgICAgZG9tYWluTmFtZXM6IGRvbWFpbk5hbWVFZGdlICE9PSB1bmRlZmluZWQgPyBbZG9tYWluTmFtZUVkZ2VdIDogdW5kZWZpbmVkLFxuICAgICAgY2VydGlmaWNhdGU6IGNlcnRFZGdlLFxuICAgICAgaHR0cFZlcnNpb246IGNmLkh0dHBWZXJzaW9uLkhUVFAyLFxuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIGFsbG93ZWRNZXRob2RzOiBjZi5BbGxvd2VkTWV0aG9kcy5BTExPV19BTEwsXG4gICAgICAgIGNhY2hlUG9saWN5OiBjZi5DYWNoZVBvbGljeS5DQUNISU5HX0RJU0FCTEVELFxuICAgICAgICBjb21wcmVzczogdHJ1ZSxcbiAgICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeTogYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSxcbiAgICAgICAgb3JpZ2luOiBhcGlHd3lPcmlnaW4sXG4gICAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBjZi5WaWV3ZXJQcm90b2NvbFBvbGljeS5SRURJUkVDVF9UT19IVFRQUyxcbiAgICAgIH0sXG4gICAgICBlbmFibGVJcHY2OiB0cnVlLFxuICAgICAgcHJpY2VDbGFzczogY2YuUHJpY2VDbGFzcy5QUklDRV9DTEFTU18xMDAsXG4gICAgICBsb2dCdWNrZXQ6IGJ1Y2tldExvZ3MsXG4gICAgICBsb2dGaWxlUHJlZml4OiBwcm9wcy5kb21haW5OYW1lRWRnZVxuICAgICAgICA/IGAke3JldmVyc2VEb21haW4ocHJvcHMuZG9tYWluTmFtZUVkZ2UpfS9jbG91ZGZyb250LXJhdy9gXG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgIH0pO1xuICAgIGlmIChyZW1vdmFsUG9saWN5ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRoaXMuX2Nsb3VkRnJvbnREaXN0cm8uYXBwbHlSZW1vdmFsUG9saWN5KHJlbW92YWxQb2xpY3kpO1xuICAgIH1cblxuICAgIC8vIEFkZCByb3V0ZXMgdG8gdGhlIENsb3VkRnJvbnQgRGlzdHJpYnV0aW9uXG4gICAgTWljcm9BcHBzQ0YuYWRkUm91dGVzKHNjb3BlLCB7XG4gICAgICBhcGlHd3lPcmlnaW4sXG4gICAgICBidWNrZXRBcHBzT3JpZ2luLFxuICAgICAgZGlzdHJvOiB0aGlzLl9jbG91ZEZyb250RGlzdHJvLFxuICAgICAgYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeTogYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSxcbiAgICAgIHJvb3RQYXRoUHJlZml4LFxuICAgICAgY3JlYXRlQVBJUGF0aFJvdXRlLFxuICAgIH0pO1xuXG4gICAgLy9cbiAgICAvLyBDcmVhdGUgdGhlIGVkZ2UgbmFtZSBmb3IgdGhlIENsb3VkRnJvbnQgZGlzdHJvXG4gICAgLy9cblxuICAgIGlmIChyNTNab25lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGNvbnN0IHJyQXBwc0VkZ2UgPSBuZXcgcjUzLlJlY29yZFNldCh0aGlzLCAnZWRnZS1hcmVjb3JkJywge1xuICAgICAgICByZWNvcmROYW1lOiBkb21haW5OYW1lRWRnZSxcbiAgICAgICAgcmVjb3JkVHlwZTogcjUzLlJlY29yZFR5cGUuQSxcbiAgICAgICAgdGFyZ2V0OiByNTMuUmVjb3JkVGFyZ2V0LmZyb21BbGlhcyhuZXcgcjUzdGFyZ2V0cy5DbG91ZEZyb250VGFyZ2V0KHRoaXMuX2Nsb3VkRnJvbnREaXN0cm8pKSxcbiAgICAgICAgem9uZTogcjUzWm9uZSxcbiAgICAgIH0pO1xuICAgICAgaWYgKHJlbW92YWxQb2xpY3kgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICByckFwcHNFZGdlLmFwcGx5UmVtb3ZhbFBvbGljeShyZW1vdmFsUG9saWN5KTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cbiJdfQ==