"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsAPIGwy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
/**
 * (experimental) Create a new MicroApps API Gateway HTTP API endpoint.
 *
 * @experimental
 */
class MicroAppsAPIGwy extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        if ((props.domainNameOrigin === undefined && props.certOrigin !== undefined) ||
            (props.domainNameOrigin !== undefined && props.certOrigin === undefined)) {
            throw new Error('If either of domainNameOrigin or certOrigin are set then the other must be set');
        }
        if ((props.domainNameEdge === undefined && props.certOrigin !== undefined) ||
            (props.domainNameEdge !== undefined && props.certOrigin === undefined)) {
            throw new Error('If either of domainNameOrigin or certOrigin are set then the other must be set');
        }
        const { r53Zone, domainNameEdge, domainNameOrigin, certOrigin, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix, } = props;
        // API Gateway uses the `id` string as the gateway name without
        // any randomization... we have to make sure the name is unique-ish
        const apigatewayName = assetNameRoot
            ? `${assetNameRoot}${assetNameSuffix}`
            : `${aws_cdk_lib_1.Stack.of(this).stackName}-microapps`;
        //
        // APIGateway domain names for CloudFront and origin
        //
        this._httpApi = new apigwy.HttpApi(this, 'gwy', {
            apiName: apigatewayName,
            createDefaultStage: false,
        });
        if (removalPolicy !== undefined) {
            this._httpApi.applyRemovalPolicy(removalPolicy);
        }
        // Create the stage
        const stage = new apigwy.HttpStage(this, 'stage', {
            httpApi: this._httpApi,
            autoDeploy: true,
            // If rootPathPrefix is not defined this will be the $default stage
            stageName: rootPathPrefix,
        });
        if (domainNameEdge !== undefined && certOrigin !== undefined) {
            // Create Custom Domains for API Gateway
            const dnAppsEdge = new apigwy.DomainName(this, 'microapps-apps-edge-dn', {
                domainName: domainNameEdge,
                certificate: certOrigin,
            });
            if (removalPolicy !== undefined) {
                dnAppsEdge.applyRemovalPolicy(removalPolicy);
            }
            // Create the edge domain name mapping for the API
            const apiMapping = new apigwy.ApiMapping(this, 'mapping', {
                api: this._httpApi,
                domainName: dnAppsEdge,
                stage,
            });
            // 2022-01-16 - CDK is still generating CloudFormation with no dependency
            // between the R53 RecordSet and the Mapping
            apiMapping.node.addDependency(dnAppsEdge);
        }
        if (domainNameOrigin !== undefined && certOrigin !== undefined) {
            this._dnAppsOrigin = new apigwy.DomainName(this, 'origin-dn', {
                domainName: domainNameOrigin,
                certificate: certOrigin,
            });
            if (removalPolicy !== undefined) {
                this._dnAppsOrigin.applyRemovalPolicy(removalPolicy);
            }
        }
        // Enable access logs on API Gateway
        const apiAccessLogs = new logs.LogGroup(this, 'logs', {
            logGroupName: apigatewayName
                ? `/aws/apigwy/${apigatewayName}`
                : `/aws/apigwy/${this.httpApi.httpApiName}`,
            retention: logs.RetentionDays.TWO_WEEKS,
        });
        if (removalPolicy !== undefined) {
            apiAccessLogs.applyRemovalPolicy(removalPolicy);
        }
        // const stage = this._httpApi.defaultStage?.node.defaultChild as apigwy.CfnStage;
        stage.accessLogSettings = {
            destinationArn: apiAccessLogs.logGroupArn,
            format: JSON.stringify({
                requestId: '$context.requestId',
                userAgent: '$context.identity.userAgent',
                sourceIp: '$context.identity.sourceIp',
                requestTime: '$context.requestTime',
                requestTimeEpoch: '$context.requestTimeEpoch',
                httpMethod: '$context.httpMethod',
                path: '$context.path',
                status: '$context.status',
                protocol: '$context.protocol',
                responseLength: '$context.responseLength',
                domainName: '$context.domainName',
            }),
        };
        // Create a logging role
        // Tips: https://github.com/aws/aws-cdk/issues/11100
        const apiGwyLogRole = new iam.Role(this, 'logs-role', {
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs'),
            ],
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        });
        apiAccessLogs.grantWrite(apiGwyLogRole);
        //
        // Let API Gateway accept requests using domainNameOrigin
        // That is the origin URI that CloudFront uses for this gateway.
        // The gateway will refuse the traffic if it doesn't have the
        // domain name registered.
        //
        if (this._dnAppsOrigin !== undefined) {
            const mappingAppsApis = new apigwy.ApiMapping(this, 'api-map-origin', {
                api: this._httpApi,
                domainName: this._dnAppsOrigin,
                stage,
            });
            // 2022-01-16 - CDK is still generating CloudFormation with no dependency
            // between the R53 RecordSet and the Mapping
            mappingAppsApis.node.addDependency(this._dnAppsOrigin);
            if (removalPolicy !== undefined) {
                mappingAppsApis.applyRemovalPolicy(removalPolicy);
            }
        }
        //
        // Create the origin name for API Gateway
        //
        if (r53Zone !== undefined && this._dnAppsOrigin) {
            const rrAppsOrigin = new r53.ARecord(this, 'origin-arecord', {
                zone: r53Zone,
                recordName: domainNameOrigin,
                target: r53.RecordTarget.fromAlias(new r53targets.ApiGatewayv2DomainProperties(this._dnAppsOrigin.regionalDomainName, this._dnAppsOrigin.regionalHostedZoneId)),
            });
            if (removalPolicy !== undefined) {
                rrAppsOrigin.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    /**
     * (experimental) Domain Name applied to API Gateway origin.
     *
     * @experimental
     */
    get dnAppsOrigin() {
        return this._dnAppsOrigin;
    }
    /**
     * (experimental) API Gateway.
     *
     * @experimental
     */
    get httpApi() {
        return this._httpApi;
    }
}
exports.MicroAppsAPIGwy = MicroAppsAPIGwy;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsAPIGwy[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsAPIGwy", version: "0.2.7" };
//# sourceMappingURL=data:application/json;base64,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