import json
import datetime
import time
import os
import requests
from bilibili_api.src import exception
import base64
import urllib3
import logging

urllib3.disable_warnings()

__path = os.path.dirname(__file__)
default_headers = {
    "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) "
                  "Chrome/79.0.3945.130 Safari/537.36",
    "Referer": "https://www.bilibili.com"
}
logger = logging.getLogger("bilibili_api")
logger.setLevel(logging.WARNING)
logger.addHandler(logging.StreamHandler())


def get_apis():
    with open(os.path.join(__path, "api.json"), "r", encoding="utf-8") as f:
        apis = json.loads(f.read())
        f.close()
    return apis


class Danmaku:
    def __init__(self, text: str, dm_time: float = 0.0, send_time: float = time.time(), id: str = "False"
                 , color: int = int("FFFFFF", 16),
                 mode: int = 1, font_size: int = 25, is_sub: bool = False):
        self.dm_time = datetime.timedelta(seconds=dm_time)
        self.send_time = datetime.datetime.fromtimestamp(send_time)
        self.id = id
        self.color = color
        self.mode = mode
        self.font_size = font_size
        self.is_sub = is_sub
        self.text = text
        self.__font_size_map = {
            "small": 18,
            "normal": 25,
            "big": 36
        }
        self.__mode_map = {
            "fly": 1,
            "top": 5,
            "bottom": 4
        }

    def __str__(self):
        ret = "%s, %s, %s" % (self.send_time, self.dm_time, self.text)
        return ret

    def __len__(self):
        return len(self.text)

    def set_hex_color(self, hex_color):
        dec = int(hex_color, 16)
        self.color = dec

    def set_mode(self, mode):
        self.mode = self.__mode_map[mode]

    def set_font_size(self, font_size):
        self.font_size = self.__font_size_map[font_size]

    def get_hex_color(self):
        return hex(self.color)


class Verify:
    def __init__(self, sessdata: str = "False", csrf: str = "False"):
        self.sessdata = sessdata
        self.csrf = csrf

    def get_cookies(self):
        cookies = {
            "SESSDATA": self.sessdata
        }
        return cookies

    def has_sess(self):
        if self.sessdata != "False":
            return True
        else:
            return False

    def has_csrf(self):
        if self.csrf != "False":
            return True
        else:
            return False

    def check(self):
        ret = {
            "code": -2,
            "message": ""
        }
        if not self.has_sess():
            ret["code"] = -3
            ret["message"] = "未提供SESSDATA"
        else:
            api = "https://api.bilibili.com/x/web-interface/archive/like"
            data = {"aid": "83175485", "like": 1, "csrf": self.csrf}
            req = requests.post(url=api, data=data, cookies=self.get_cookies())
            if req.ok:
                con = req.json()
                if con["code"] == -111:
                    ret["code"] = -1
                    ret["message"] = "csrf 校验失败"
                elif con["code"] == -101 or con["code"] == -400:
                    ret["code"] = -2
                    ret["message"] = "SESSDATA值有误"
                else:
                    ret["code"] = 0
                    ret["message"] = "0"
            else:
                raise exception.NetworkException(req.status_code)
        return ret


class Get:
    def __init__(self, url, params=None, cookies=None, headers=None):
        if headers is None:
            self.headers = default_headers
        else:
            self.headers = headers
        if cookies is None:
            self.cookies = {}
        else:
            self.cookies = cookies
        if params is None:
            self.params = {}
        else:
            self.params = params
        self.url = url

    def __call__(self):
        req = requests.get(url=self.url, params=self.params, headers=self.headers, cookies=self.cookies, verify=False)
        if req.ok:
            con = json.loads(req.text)
            if con["code"] != 0:
                raise exception.BiliException(con["code"], con["message"])
            else:
                #take care of new api
                if 'data' in con.keys():  
                    return con['data']
                else :
                    if 'result' in con.keys() :
                        return con["result"]
                    else :
                        raise exception.BiliException('data key error',con)
        else:
            raise exception.NetworkException(req.status_code)


class Post:
    def __init__(self, url, cookies, data=None, headers=None):
        if headers is None:
            self.headers = default_headers
        else:
            self.headers = headers
        if cookies is None:
            self.cookies = {}
        else:
            self.cookies = cookies
        if data is None:
            self.data = {}
        else:
            self.data = data
        self.url = url

    def __call__(self):

        req = requests.post(url=self.url, data=self.data, headers=self.headers, cookies=self.cookies, verify=False)
        if req.ok:
            con = json.loads(req.text)
            if con["code"] != 0:
                raise exception.BiliException(con["code"], con["message"])
            else:
                return con
        else:
            raise exception.NetworkException(req.status_code)


def nyanpass(这是一个毫无用处的参数="nyanpass"):
    audio = base64.b64decode("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")
    file = open("tmp.mp3", "wb")
    file.write(audio)
    file.close()
    os.startfile("tmp.mp3")
    print("喵帕斯(=・ω・=)\nhttp://space.bilibili.com/12440199")
    time.sleep(5)
    os.remove("tmp.mp3")


# 代码来源：https://www.zhihu.com/question/381784377/answer/1099438784
def bvid2aid(bv: str):
    table = 'fZodR9XQDSUm21yCkr6zBqiveYah8bt4xsWpHnJE7jL5VG3guMTKNPAwcF'
    tr = {}
    for i in range(58):
        tr[table[i]] = i
    s = [11, 10, 3, 8, 4, 6]
    xor = 177451812
    add = 8728348608

    def dec(x):
        r = 0
        for i in range(6):
            r += tr[x[s[i]]] * 58 ** i
        return (r - add) ^ xor

    return dec(bv)


# 代码来源：https://www.zhihu.com/question/381784377/answer/1099438784
def aid2bvid(aid: int):
    table = 'fZodR9XQDSUm21yCkr6zBqiveYah8bt4xsWpHnJE7jL5VG3guMTKNPAwcF'
    tr = {}
    for i in range(58):
        tr[table[i]] = i
    s = [11, 10, 3, 8, 4, 6]
    xor = 177451812
    add = 8728348608

    def enc(x):
        x = (x ^ xor) + add
        r = list('BV1  4 1 7  ')
        for i in range(6):
            r[s[i]] = table[x // 58 ** i % 58]
        return ''.join(r)

    return enc(aid)
