# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytest_mock_generator']

package_data = \
{'': ['*']}

install_requires = \
['mock-generator>=2.2.1,<3.0.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=4.5.0,<5.0.0']}

entry_points = \
{'pytest11': ['pytest_mock_generator = pytest_mock_generator']}

setup_kwargs = {
    'name': 'pytest-mock-generator',
    'version': '1.0.0',
    'description': 'A pytest fixture wrapper for https://pypi.org/project/mock-generator',
    'long_description': '# pytest-mock-generator\n\n<div align="center">\n\n[![Build status](https://github.com/pksol/pytest-mock-generator/workflows/build/badge.svg?branch=master&event=push)](https://github.com/pksol/pytest-mock-generator/actions?query=workflow%3Abuild)\n[![Python Version](https://img.shields.io/pypi/pyversions/pytest-mock-generator.svg)](https://pypi.org/project/pytest-mock-generator/)\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pksol/pytest-mock-generator/blob/master/.pre-commit-config.yaml)\n[![Semantic Versions](https://img.shields.io/badge/%20%20%F0%9F%93%A6%F0%9F%9A%80-semantic--versions-e10079.svg)](https://github.com/pksol/pytest-mock-generator/releases)\n[![License](https://img.shields.io/github/license/pksol/pytest-mock-generator)](https://github.com/pksol/pytest-mock-generator/blob/master/LICENSE)\n\nA pytest fixture wrapper for https://pypi.org/project/mock-generator\n\n</div>\n\n## Installation\n\n```bash\npip install pytest-mock-generator\n```\n\nor install with [poetry](https://github.com/python-poetry/poetry):\n\n```bash\npoetry add pytest-mock-generator\n```\n\n## Usage\nThis [pytest plugin](https://docs.pytest.org/en/latest/how-to/writing_plugins.html) \nadds the `mg` [fixture](https://docs.pytest.org/en/latest/reference/fixtures.html#fixture)\nwhich helps when writing tests that use [python mocks](https://docs.python.org/3.7/library/unittest.mock.html).\n\nLet\'s start with an easy example. Assume you have a module named `tested_module.py` which holds a function\nto process a string sent to it and then add it to a zip file:\n```python\nimport zipfile\n\ndef process_and_zip(zip_path, file_name, contents):\n    processed_contents = "processed " + contents  # some complex logic\n    with zipfile.ZipFile(zip_path, \'w\') as zip_container:\n        zip_container.writestr(file_name, processed_contents)\n```\nThis is the unit under test, or UUT.\n\nAlthough this is a very short function, \nwriting the test code takes a lot of time. It\'s the fact that the function uses\na context manager makes the testing more complex than it should be.\n*If you don\'t believe me, try to manually write mocks and asserts which verify\nthat `zip_container.writestr` was called with the right parameters.*\n\nIn any case, you start with a test skeleton:\n\n```python\nfrom tests.sample.code.tested_module import process_and_zip\n\ndef test_process_and_zip(mocker, mg):\n    # Arrange: todo  \n    \n    # Act: invoking the tested code\n    process_and_zip(\'/path/to.zip\', \'in_zip.txt\', \'foo bar\')\n    \n    # Assert: todo\n```\nNow it\'s time to use Mock Generator instead of manually writing the \'Arrange\' \nand \'Assert\' sections.\n\n### Generating the \'Arrange\' section\nTo generate the \'Arrange\' section, simply put this code at the beginning of \nyour test function skeleton and run it (make sure to add the `mg` fixture to \nyour test function):\n```python\nmg.generate_uut_mocks(process_and_zip)\n```\nThis will generate the \'Arrange\' section for you:\n```python\n# mocked dependencies\nmock_ZipFile = mocker.MagicMock(name=\'ZipFile\')\nmocker.patch(\'tests.sample.code.tested_module.zipfile.ZipFile\', new=mock_ZipFile)\n```\n<b>The generated code is returned, printed to the console and also copied to the\nclipboard for your convenience. \nJust paste it (as simple as ctrl+V) at the start of your test function:</b>\n```python\nfrom tests.sample.code.tested_module import process_and_zip\n\ndef test_process_and_zip(mocker, mg):\n    # mocked dependencies\n    mock_ZipFile = mocker.MagicMock(name=\'ZipFile\')\n    mocker.patch(\'tests.sample.code.tested_module.zipfile.ZipFile\', new=mock_ZipFile)\n    \n    # Act: invoking the tested code\n    process_and_zip(\'/path/to.zip\', \'in_zip.txt\', \'foo bar\')\n    \n    # Assert: todo\n```\n\nExcellent! Arrange section is ready.\n\n### Generating the Assert section\nNow it\'s time to add the asserts. Add the following code\n**at the \'Assert\'** step:\n```python\nmg.generate_asserts(mock_ZipFile)\n```\nThe `mock_ZipFile` is the mock object you generated earlier.\nNow execute the test function to get the assert section: \n```python\nassert 1 == mock_ZipFile.call_count\nmock_ZipFile.assert_called_once_with(\'/path/to.zip\', \'w\')\nmock_ZipFile.return_value.__enter__.assert_called_once_with()\nmock_ZipFile.return_value.__enter__.return_value.writestr.assert_called_once_with(\'in_zip.txt\', \'processed foo bar\')\nmock_ZipFile.return_value.__exit__.assert_called_once_with(None, None, None)\n```\nWow, that\'s a handful of asserts! Some are very useful: \n* Checking that we opened the zip file with the right parameters.\n* Checking that we wrote the correct data to the proper file.\n* Finally, ensuring that `__enter__` and `__exit__` are called, so there \nare no open file handles which could cause problems.\n\nYou can remove any generated line which you find unnecessary.   \n\nPaste that code right after the act phase, and you\'re done!\n\nThe complete test function:\n```python\nfrom tests.sample.code.tested_module import process_and_zip\n\ndef test_process_and_zip(mocker):\n    # mocked dependencies\n    mock_ZipFile = mocker.MagicMock(name=\'ZipFile\')\n    mocker.patch(\'tests.sample.code.tested_module.zipfile.ZipFile\', new=mock_ZipFile)\n    \n    # Act: invoking the tested code\n    process_and_zip(\'/path/to.zip\', \'in_zip.txt\', \'foo bar\')\n    \n    assert 1 == mock_ZipFile.call_count\n    mock_ZipFile.assert_called_once_with(\'/path/to.zip\', \'w\')\n    mock_ZipFile.return_value.__enter__.assert_called_once_with()\n    mock_ZipFile.return_value.__enter__.return_value.writestr.assert_called_once_with(\'in_zip.txt\', \'processed foo bar\')\n    mock_ZipFile.return_value.__exit__.assert_called_once_with(None, None, None)\n```\nCan you imagine the time it would have taken you to code this on your own?\n\n### What\'s Next\n#### Asserting Existing Mocks\nAt times, you may be editing a test code already containing mocks, or\nyou choose to write the mocks yourself, to gain some extra control.\n\nMock Generator can generate the assert section for standard \nPython mocks, even if they were not created using the Mock Generator. \n\nPut this after the \'Act\' (replace `mock_obj` with your mock object name): \n```python\nmg.generate_asserts(mock_obj)\n```\nTake the generated code and paste it at the \'Assert\' section. \n\n#### Generating the \'Arrange\' and \'Assert\' sections in one call\nYou can make the `generate_uut_mocks_with_asserts` call create the \n`generate_asserts` code for you (instead of having to call \n`generate_uut_mocks`):\n```python\nmg.generate_uut_mocks_with_asserts(function_under_test)\n```\n\n## More information\nAdditional documentation can be found in the [mock-generator pypi](https://pypi.org/project/mock-generator).\n\n## 📈 Releases\n\nYou can see the list of available releases on the [GitHub Releases](https://github.com/pksol/pytest-mock-generator/releases) page.\n\nWe follow [Semantic Versions](https://semver.org/) specification.\n\n## 🛡 License\n\n[![License](https://img.shields.io/github/license/pksol/pytest-mock-generator)](https://github.com/pksol/pytest-mock-generator/blob/master/LICENSE)\n\nThis project is licensed under the terms of the `MIT` license. See [LICENSE](https://github.com/pksol/pytest-mock-generator/blob/master/LICENSE) for more details.\n\n## 📃 Citation\n\n```bibtex\n@misc{pytest-mock-generator,\n  author = {Peter Kogan},\n  title = {A pytest fixture wrapper for https://pypi.org/project/mock-generator},\n  year = {2021},\n  publisher = {GitHub},\n  journal = {GitHub repository},\n  howpublished = {\\url{https://github.com/pksol/pytest-mock-generator}}\n}\n```\n\n## Credits [![🚀 Your next Python package needs a bleeding-edge project structure.](https://img.shields.io/badge/python--package--template-%F0%9F%9A%80-brightgreen)](https://github.com/TezRomacH/python-package-template)\n\nThis project was generated with [`python-package-template`](https://github.com/TezRomacH/python-package-template)\n',
    'author': 'Peter Kogan',
    'author_email': 'kogan.peter@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pksol/pytest-mock-generator',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
