"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWebhookRedelivery = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const utils_1 = require("./utils");
const webhook_redelivery_function_1 = require("./webhook-redelivery-function");
/**
 * Create a Lambda that runs every 5 minutes to check for Github webhook delivery failures and retry them.
 *
 * @internal
 */
class GithubWebhookRedelivery extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.handler = new webhook_redelivery_function_1.WebhookRedeliveryFunction(this, 'Lambda', {
            description: 'Check for GitHub webhook delivery failures and redeliver them',
            environment: {
                GITHUB_SECRET_ARN: props.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: props.secrets.githubPrivateKey.secretArn,
            },
            reservedConcurrentExecutions: 1, // avoid concurrent executions
            timeout: cdk.Duration.seconds(4.5 * 60), // 4.5 minutes
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.ORCHESTRATOR),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            // applicationLogLevelV2: ApplicationLogLevel.DEBUG,
        });
        props.secrets.github.grantRead(this.handler);
        props.secrets.githubPrivateKey.grantRead(this.handler);
        new aws_cdk_lib_1.aws_events.Rule(this, 'Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.minutes(5)),
            description: 'Schedule to run the webhook redelivery lambda every 5 minutes',
            targets: [
                new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler, {
                    retryAttempts: 0,
                }),
            ],
        });
    }
}
exports.GithubWebhookRedelivery = GithubWebhookRedelivery;
//# sourceMappingURL=data:application/json;base64,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