"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyBody = verifyBody;
exports.generateExecutionName = generateExecutionName;
exports.handler = handler;
const crypto = require("crypto");
const client_sfn_1 = require("@aws-sdk/client-sfn");
const lambda_github_1 = require("./lambda-github");
const lambda_helpers_1 = require("./lambda-helpers");
const sf = new client_sfn_1.SFNClient();
// TODO use @octokit/webhooks?
function getHeader(event, header) {
    // API Gateway doesn't lowercase headers (V1 event) but Lambda URLs do (V2 event) :(
    for (const headerName of Object.keys(event.headers)) {
        if (headerName.toLowerCase() === header.toLowerCase()) {
            return event.headers[headerName];
        }
    }
    return undefined;
}
/**
 * Exported for unit testing.
 * @internal
 */
function verifyBody(event, secret) {
    const sig = Buffer.from(getHeader(event, 'x-hub-signature-256') || '', 'utf8');
    if (!event.body) {
        throw new Error('No body');
    }
    let body;
    if (event.isBase64Encoded) {
        body = Buffer.from(event.body, 'base64');
    }
    else {
        body = Buffer.from(event.body || '', 'utf8');
    }
    const hmac = crypto.createHmac('sha256', secret);
    hmac.update(body);
    const expectedSig = Buffer.from(`sha256=${hmac.digest('hex')}`, 'utf8');
    console.log('Calculated signature: ', expectedSig.toString());
    if (sig.length !== expectedSig.length || !crypto.timingSafeEqual(sig, expectedSig)) {
        throw new Error(`Signature mismatch. Expected ${expectedSig.toString()} but got ${sig.toString()}`);
    }
    return body.toString();
}
async function isDeploymentPending(payload) {
    const statusesUrl = payload.deployment?.statuses_url;
    if (statusesUrl === undefined) {
        return false;
    }
    try {
        const { octokit } = await (0, lambda_github_1.getOctokit)(payload.installation?.id);
        const statuses = await octokit.request(statusesUrl);
        return statuses.data[0]?.state === 'waiting';
    }
    catch (e) {
        console.error('Unable to check deployment. Try adding deployment read permission.', e);
        return false;
    }
}
function matchLabelsToProvider(labels) {
    const jobLabelSet = labels.map((label) => label.toLowerCase());
    const supportedLabels = JSON.parse(process.env.SUPPORTED_LABELS);
    // is every label the job requires available in the runner provider?
    for (const supportedLabelSet of supportedLabels) {
        const lowerCasedSupportedLabelSet = supportedLabelSet.labels.map((label) => label.toLowerCase());
        if (jobLabelSet.every(label => label == 'self-hosted' || lowerCasedSupportedLabelSet.includes(label))) {
            return supportedLabelSet.provider;
        }
    }
    return undefined;
}
/**
 * Generate a unique execution name which is limited to 64 characters (also used as runner name).
 *
 * Exported for unit testing.
 *
 * @internal
 */
function generateExecutionName(event, payload) {
    const deliveryId = getHeader(event, 'x-github-delivery') ?? `${Math.random()}`;
    const repoNameTruncated = payload.repository.name.slice(0, 64 - deliveryId.length - 1);
    return `${repoNameTruncated}-${deliveryId}`;
}
async function handler(event) {
    if (!process.env.WEBHOOK_SECRET_ARN || !process.env.STEP_FUNCTION_ARN || !process.env.SUPPORTED_LABELS || !process.env.REQUIRE_SELF_HOSTED_LABEL) {
        throw new Error('Missing environment variables');
    }
    const webhookSecret = (await (0, lambda_helpers_1.getSecretJsonValue)(process.env.WEBHOOK_SECRET_ARN)).webhookSecret;
    let body;
    try {
        body = verifyBody(event, webhookSecret);
    }
    catch (e) {
        console.error(e);
        return {
            statusCode: 403,
            body: 'Bad signature',
        };
    }
    if (getHeader(event, 'content-type') !== 'application/json') {
        console.error(`This webhook only accepts JSON payloads, got ${getHeader(event, 'content-type')}`);
        return {
            statusCode: 400,
            body: 'Expecting JSON payload',
        };
    }
    if (getHeader(event, 'x-github-event') === 'ping') {
        return {
            statusCode: 200,
            body: 'Pong',
        };
    }
    // if (getHeader(event, 'x-github-event') !== 'workflow_job' && getHeader(event, 'x-github-event') !== 'workflow_run') {
    //     console.error(`This webhook only accepts workflow_job and workflow_run, got ${getHeader(event, 'x-github-event')}`);
    if (getHeader(event, 'x-github-event') !== 'workflow_job') {
        console.error(`This webhook only accepts workflow_job, got ${getHeader(event, 'x-github-event')}`);
        return {
            statusCode: 200,
            body: 'Expecting workflow_job',
        };
    }
    const payload = JSON.parse(body);
    if (payload.action !== 'queued') {
        console.log({
            notice: `Ignoring action "${payload.action}", expecting "queued"`,
            job: payload.workflow_job,
        });
        return {
            statusCode: 200,
            body: 'OK. No runner started (action is not "queued").',
        };
    }
    if (process.env.REQUIRE_SELF_HOSTED_LABEL === '1' && !payload.workflow_job.labels.includes('self-hosted')) {
        console.log({
            notice: `Ignoring labels "${payload.workflow_job.labels}", expecting "self-hosted"`,
            job: payload.workflow_job,
        });
        return {
            statusCode: 200,
            body: 'OK. No runner started (no "self-hosted" label).',
        };
    }
    // don't start step function unless labels match a runner provider
    const provider = matchLabelsToProvider(payload.workflow_job.labels);
    if (!provider) {
        console.log({
            notice: `Ignoring labels "${payload.workflow_job.labels}", as they don't match a supported runner provider`,
            job: payload.workflow_job,
        });
        return {
            statusCode: 200,
            body: 'OK. No runner started (no provider with matching labels).',
        };
    }
    // don't start runners for a deployment that's still pending as GitHub will send another event when it's ready
    if (await isDeploymentPending(payload)) {
        console.log({
            notice: 'Ignoring job as its deployment is still pending',
            job: payload.workflow_job,
        });
        return {
            statusCode: 200,
            body: 'OK. No runner started (deployment pending).',
        };
    }
    // start execution
    const executionName = generateExecutionName(event, payload);
    const input = {
        owner: payload.repository.owner.login,
        repo: payload.repository.name,
        jobId: payload.workflow_job.id,
        jobUrl: payload.workflow_job.html_url,
        installationId: payload.installation?.id ?? -1, // always pass value because step function can't handle missing input
        labels: payload.workflow_job.labels.join(','),
        provider: provider,
    };
    const execution = await sf.send(new client_sfn_1.StartExecutionCommand({
        stateMachineArn: process.env.STEP_FUNCTION_ARN,
        input: JSON.stringify(input),
        // name is not random so multiple execution of this webhook won't cause multiple builders to start
        name: executionName,
    }));
    console.log({
        notice: 'Started orchestrator',
        execution: execution.executionArn,
        sfnInput: input,
        job: payload.workflow_job,
    });
    return {
        statusCode: 202,
        body: executionName,
    };
}
//# sourceMappingURL=data:application/json;base64,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