"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunnerProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const autoscaling = require("aws-cdk-lib/aws-autoscaling");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const fargate_1 = require("./fargate");
const image_builders_1 = require("../image-builders");
const utils_1 = require("../utils");
/**
 * Custom ECS EC2 launch target that allows specifying capacity provider strategy and propagating tags.
 */
class CustomEcsEc2LaunchTarget extends aws_cdk_lib_1.aws_stepfunctions_tasks.EcsEc2LaunchTarget {
    constructor(options) {
        super(options);
        this.capacityProvider = options.capacityProvider;
    }
    /**
     * Called when the ECS launch type configured on RunTask
     */
    bind(_task, _launchTargetOptions) {
        const base = super.bind(_task, _launchTargetOptions);
        return {
            ...base,
            parameters: {
                ...(base.parameters ?? {}),
                PropagateTags: aws_cdk_lib_1.aws_ecs.PropagatedTagSource.TASK_DEFINITION,
                CapacityProviderStrategy: [
                    {
                        CapacityProvider: this.capacityProvider,
                    },
                ],
                LaunchType: undefined, // You may choose a capacity provider or a launch type but not both.
            },
        };
    }
}
/**
 * GitHub Actions runner provider using ECS on EC2 to execute jobs.
 *
 * ECS can be useful when you want more control of the infrastructure running the GitHub Actions Docker containers. You can control the autoscaling
 * group to scale down to zero during the night and scale up during work hours. This way you can still save money, but have to wait less for
 * infrastructure to spin up.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class EcsRunnerProvider extends common_1.BaseProvider {
    /**
     * Create new image builder that builds ECS specific runner images.
     *
     * You can customize the OS, architecture, VPC, subnet, security groups, etc. by passing in props.
     *
     * You can add components to the image builder by calling `imageBuilder.addComponent()`.
     *
     * The default OS is Ubuntu running on x64 architecture.
     *
     * Included components:
     *  * `RunnerImageComponent.requiredPackages()`
     *  * `RunnerImageComponent.runnerUser()`
     *  * `RunnerImageComponent.git()`
     *  * `RunnerImageComponent.githubCli()`
     *  * `RunnerImageComponent.awsCli()`
     *  * `RunnerImageComponent.docker()`
     *  * `RunnerImageComponent.githubRunner()`
     */
    static imageBuilder(scope, id, props) {
        return image_builders_1.RunnerImageBuilder.new(scope, id, {
            os: common_1.Os.LINUX_UBUNTU,
            architecture: common_1.Architecture.X86_64,
            components: [
                image_builders_1.RunnerImageComponent.requiredPackages(),
                image_builders_1.RunnerImageComponent.runnerUser(),
                image_builders_1.RunnerImageComponent.git(),
                image_builders_1.RunnerImageComponent.githubCli(),
                image_builders_1.RunnerImageComponent.awsCli(),
                image_builders_1.RunnerImageComponent.docker(),
                image_builders_1.RunnerImageComponent.githubRunner(props?.runnerVersion ?? common_1.RunnerVersion.latest()),
            ],
            ...props,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.retryableErrors = [
            'Ecs.EcsException',
            'ECS.AmazonECSException',
            'Ecs.LimitExceededException',
            'Ecs.UpdateInProgressException',
        ];
        this.labels = props?.labels ?? ['ecs'];
        this.group = props?.group;
        this.defaultLabels = props?.defaultLabels ?? true;
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'default vpc', { isDefault: true });
        this.subnetSelection = props?.subnetSelection;
        this.securityGroups = props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'security group', { vpc: this.vpc })];
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
        this.assignPublicIp = props?.assignPublicIp ?? true;
        this.placementStrategies = props?.placementStrategies;
        this.placementConstraints = props?.placementConstraints;
        this.cluster = props?.cluster ? props.cluster : new aws_cdk_lib_1.aws_ecs.Cluster(this, 'cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: false,
        });
        if (props?.storageOptions && !props?.storageSize) {
            throw new Error('storageSize is required when storageOptions are specified');
        }
        const imageBuilder = props?.imageBuilder ?? EcsRunnerProvider.imageBuilder(this, 'Image Builder');
        const image = this.image = imageBuilder.bindDockerImage();
        if (props?.capacityProvider) {
            if (props?.minInstances || props?.maxInstances || props?.instanceType || props?.storageSize || props?.spot || props?.spotMaxPrice) {
                cdk.Annotations.of(this).addWarning('When using a custom capacity provider, minInstances, maxInstances, instanceType, storageSize, spot, and spotMaxPrice will be ignored.');
            }
            this.capacityProvider = props.capacityProvider;
        }
        else {
            const spot = props?.spot ?? props?.spotMaxPrice !== undefined;
            const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch Template', {
                machineImage: this.defaultClusterInstanceAmi(),
                instanceType: props?.instanceType ?? this.defaultClusterInstanceType(),
                blockDevices: props?.storageSize ? [
                    {
                        deviceName: (0, common_1.amiRootDevice)(this, this.defaultClusterInstanceAmi().getImage(this).imageId).ref,
                        volume: {
                            ebsDevice: {
                                deleteOnTermination: true,
                                volumeSize: props.storageSize.toGibibytes(),
                                volumeType: props.storageOptions?.volumeType,
                                iops: props.storageOptions?.iops,
                                throughput: props.storageOptions?.throughput,
                            },
                        },
                    },
                ] : undefined,
                spotOptions: spot ? {
                    requestType: aws_cdk_lib_1.aws_ec2.SpotRequestType.ONE_TIME,
                    maxPrice: props?.spotMaxPrice ? parseFloat(props?.spotMaxPrice) : undefined,
                } : undefined,
                requireImdsv2: true,
                securityGroup: this.securityGroups[0],
                role: new aws_cdk_lib_1.aws_iam.Role(this, 'Launch Template Role', {
                    assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
                }),
                userData: aws_cdk_lib_1.aws_ec2.UserData.forOperatingSystem(image.os.is(common_1.Os.WINDOWS) ? aws_cdk_lib_1.aws_ec2.OperatingSystemType.WINDOWS : aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX),
            });
            this.securityGroups.slice(1).map(sg => launchTemplate.connections.addSecurityGroup(sg));
            const autoScalingGroup = new autoscaling.AutoScalingGroup(this, 'Auto Scaling Group', {
                vpc: this.vpc,
                launchTemplate,
                vpcSubnets: this.subnetSelection,
                minCapacity: props?.minInstances ?? 0,
                maxCapacity: props?.maxInstances ?? 5,
            });
            this.capacityProvider = props?.capacityProvider ?? new aws_cdk_lib_1.aws_ecs.AsgCapacityProvider(this, 'Capacity Provider', {
                autoScalingGroup,
                spotInstanceDraining: false, // waste of money to restart jobs as the restarted job won't have a token
            });
        }
        this.capacityProvider.autoScalingGroup.addUserData(
        // we don't exit on errors because all of these commands are optional
        ...this.loginCommands(), this.pullCommand(), ...this.ecsSettingsCommands());
        this.capacityProvider.autoScalingGroup.role.addToPrincipalPolicy(utils_1.MINIMAL_EC2_SSM_SESSION_MANAGER_POLICY_STATEMENT);
        image.imageRepository.grantPull(this.capacityProvider.autoScalingGroup);
        this.cluster.addAsgCapacityProvider(this.capacityProvider, {
            spotInstanceDraining: false,
            machineImageType: aws_ecs_1.MachineImageType.AMAZON_LINUX_2,
        });
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'logs', {
            retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.dind = (props?.dockerInDocker ?? true) && !image.os.is(common_1.Os.WINDOWS);
        this.task = new aws_cdk_lib_1.aws_ecs.Ec2TaskDefinition(this, 'task');
        this.container = this.task.addContainer('runner', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromEcrRepository(image.imageRepository, image.imageTag),
            cpu: props?.cpu ?? 1024,
            memoryLimitMiB: props?.memoryLimitMiB ?? (props?.memoryReservationMiB ? undefined : 3500),
            memoryReservationMiB: props?.memoryReservationMiB,
            logging: aws_cdk_lib_1.aws_ecs.AwsLogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'runner',
            }),
            command: (0, fargate_1.ecsRunCommand)(this.image.os, this.dind),
            user: image.os.is(common_1.Os.WINDOWS) ? undefined : 'runner',
            privileged: this.dind,
        });
        this.grantPrincipal = this.task.taskRole;
        // permissions for SSM Session Manager
        this.task.taskRole.addToPrincipalPolicy(utils_1.MINIMAL_ECS_SSM_SESSION_MANAGER_POLICY_STATEMENT);
    }
    defaultClusterInstanceType() {
        if (this.image.architecture.is(common_1.Architecture.X86_64)) {
            return aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M6I, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        }
        if (this.image.architecture.is(common_1.Architecture.ARM64)) {
            return aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M6G, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        }
        throw new Error(`Unable to find instance type for ECS instances for ${this.image.architecture.name}`);
    }
    defaultClusterInstanceAmi() {
        let baseImage;
        let ssmPath;
        let found = false;
        if (this.image.os.isIn(common_1.Os._ALL_LINUX_VERSIONS)) {
            if (this.image.architecture.is(common_1.Architecture.X86_64)) {
                baseImage = aws_cdk_lib_1.aws_ecs.EcsOptimizedImage.amazonLinux2(aws_cdk_lib_1.aws_ecs.AmiHardwareType.STANDARD);
                ssmPath = '/aws/service/ecs/optimized-ami/amazon-linux-2023/recommended/image_id';
                found = true;
            }
            if (this.image.architecture.is(common_1.Architecture.ARM64)) {
                baseImage = aws_cdk_lib_1.aws_ecs.EcsOptimizedImage.amazonLinux2(aws_cdk_lib_1.aws_ecs.AmiHardwareType.ARM);
                ssmPath = '/aws/service/ecs/optimized-ami/amazon-linux-2023/arm64/recommended/image_id';
                found = true;
            }
        }
        if (this.image.os.is(common_1.Os.WINDOWS)) {
            baseImage = aws_cdk_lib_1.aws_ecs.EcsOptimizedImage.windows(aws_cdk_lib_1.aws_ecs.WindowsOptimizedVersion.SERVER_2019);
            ssmPath = '/aws/service/ami-windows-latest/Windows_Server-2019-English-Full-ECS_Optimized/image_id';
            found = true;
        }
        if (!found) {
            throw new Error(`Unable to find AMI for ECS instances for ${this.image.os.name}/${this.image.architecture.name}`);
        }
        const image = {
            getImage(scope) {
                const baseImageRes = baseImage.getImage(scope);
                return {
                    imageId: `resolve:ssm:${ssmPath}`,
                    userData: baseImageRes.userData,
                    osType: baseImageRes.osType,
                };
            },
        };
        return image;
    }
    pullCommand() {
        if (this.image.os.is(common_1.Os.WINDOWS)) {
            return `Start-Job -ScriptBlock { docker pull ${this.image.imageRepository.repositoryUri}:${this.image.imageTag} }`;
        }
        return `docker pull ${this.image.imageRepository.repositoryUri}:${this.image.imageTag} &`;
    }
    loginCommands() {
        const thisStack = aws_cdk_lib_1.Stack.of(this);
        if (this.image.os.is(common_1.Os.WINDOWS)) {
            return [`(Get-ECRLoginCommand).Password | docker login --username AWS --password-stdin ${thisStack.account}.dkr.ecr.${thisStack.region}.amazonaws.com`];
        }
        return [
            'yum install -y awscli || dnf install -y awscli',
            `aws ecr get-login-password --region ${thisStack.region} | docker login --username AWS --password-stdin ${thisStack.account}.dkr.ecr.${thisStack.region}.amazonaws.com`,
        ];
    }
    ecsSettingsCommands() {
        // don't let ECS accumulate too many stopped tasks that can end up very big in our case
        // the default is 10m duration with 1h jitter which can end up with 1h10m delay for cleaning up stopped tasks
        if (this.image.os.is(common_1.Os.WINDOWS)) {
            return [
                '[Environment]::SetEnvironmentVariable("ECS_ENGINE_TASK_CLEANUP_WAIT_DURATION", "5s", "Machine")',
                '[Environment]::SetEnvironmentVariable("ECS_ENGINE_TASK_CLEANUP_WAIT_DURATION_JITTER", "5s", "Machine")',
            ];
        }
        return [
            'echo ECS_ENGINE_TASK_CLEANUP_WAIT_DURATION=5s >> /etc/ecs/ecs.config',
            'echo ECS_ENGINE_TASK_CLEANUP_WAIT_DURATION_JITTER=5s >> /etc/ecs/ecs.config',
        ];
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsRunTask(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB, // sync
            taskDefinition: this.task,
            cluster: this.cluster,
            launchTarget: new CustomEcsEc2LaunchTarget({
                capacityProvider: this.capacityProvider.capacityProviderName,
                placementStrategies: this.placementStrategies,
                placementConstraints: this.placementConstraints,
            }),
            enableExecuteCommand: this.image.os.isIn(common_1.Os._ALL_LINUX_VERSIONS),
            assignPublicIp: this.assignPublicIp,
            containerOverrides: [
                {
                    containerDefinition: this.container,
                    environment: [
                        {
                            name: 'RUNNER_TOKEN',
                            value: parameters.runnerTokenPath,
                        },
                        {
                            name: 'RUNNER_NAME',
                            value: parameters.runnerNamePath,
                        },
                        {
                            name: 'RUNNER_LABEL',
                            value: this.labels.join(','),
                        },
                        {
                            name: 'RUNNER_GROUP',
                            value: this.group ? `--runnergroup ${this.group}` : '',
                        },
                        {
                            name: 'DEFAULT_LABELS',
                            value: this.defaultLabels ? '' : '--no-default-labels',
                        },
                        {
                            name: 'GITHUB_DOMAIN',
                            value: parameters.githubDomainPath,
                        },
                        {
                            name: 'OWNER',
                            value: parameters.ownerPath,
                        },
                        {
                            name: 'REPO',
                            value: parameters.repoPath,
                        },
                        {
                            name: 'REGISTRATION_URL',
                            value: parameters.registrationUrl,
                        },
                    ],
                },
            ],
        });
    }
    grantStateMachine(_) {
    }
    status(statusFunctionRole) {
        this.image.imageRepository.grant(statusFunctionRole, 'ecr:DescribeImages');
        return {
            type: this.constructor.name,
            labels: this.labels,
            vpcArn: this.vpc?.vpcArn,
            securityGroups: this.securityGroups.map(sg => sg.securityGroupId),
            roleArn: this.task.taskRole.roleArn,
            logGroup: this.logGroup.logGroupName,
            image: {
                imageRepository: this.image.imageRepository.repositoryUri,
                imageTag: this.image.imageTag,
                imageBuilderLogGroup: this.image.logGroup?.logGroupName,
            },
        };
    }
}
exports.EcsRunnerProvider = EcsRunnerProvider;
_a = JSII_RTTI_SYMBOL_1;
EcsRunnerProvider[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.EcsRunnerProvider", version: "0.14.15" };
//# sourceMappingURL=data:application/json;base64,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