"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Runner = exports.Ec2RunnerProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const image_builders_1 = require("../image-builders");
const utils_1 = require("../utils");
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below
const linuxUserDataTemplate = `#!/bin/bash -x
TASK_TOKEN="{}"
logGroupName="{}"
runnerNamePath="{}"
githubDomainPath="{}"
ownerPath="{}"
repoPath="{}"
runnerTokenPath="{}"
labels="{}"
registrationURL="{}"
runnerGroup1="{}"
runnerGroup2="{}"
defaultLabels="{}"

heartbeat () {
  while true; do
    aws stepfunctions send-task-heartbeat --task-token "$TASK_TOKEN"
    sleep 60
  done
}
setup_logs () {
  cat <<EOF > /tmp/log.conf || exit 1
  {
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
          "collect_list": [
            {
              "file_path": "/var/log/runner.log",
              "log_group_name": "$logGroupName",
              "log_stream_name": "$runnerNamePath",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }
EOF
  /opt/aws/amazon-cloudwatch-agent/bin/amazon-cloudwatch-agent-ctl -a fetch-config -m ec2 -s -c file:/tmp/log.conf || exit 2
}
action () {
  # Determine the value of RUNNER_FLAGS
  if [ "$(< /home/runner/RUNNER_VERSION)" = "latest" ]; then
    RUNNER_FLAGS=""
  else
    RUNNER_FLAGS="--disableupdate"
  fi

  labelsTemplate="$labels,cdkghr:started:$(date +%s)"

  # Execute the configuration command for runner registration
  sudo -Hu runner /home/runner/config.sh --unattended --url "$registrationURL" --token "$runnerTokenPath" --ephemeral --work _work --labels "$labelsTemplate" $RUNNER_FLAGS --name "$runnerNamePath" $runnerGroup1 $runnerGroup2 $defaultLabels || exit 1

  # Execute the run command
  sudo --preserve-env=AWS_REGION -Hu runner /home/runner/run.sh || exit 2

  # Retrieve the status
  STATUS=$(grep -Phors "finish job request for job [0-9a-f\\-]+ with result: \K.*" /home/runner/_diag/ | tail -n1)

  # Check and print the job status
  [ -n "$STATUS" ] && echo CDKGHA JOB DONE "$labels" "$STATUS"
}
heartbeat &
if setup_logs && action | tee /var/log/runner.log 2>&1; then
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{"ok": true}'
else
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
fi
sleep 10  # give cloudwatch agent its default 5 seconds buffer duration to upload logs
poweroff
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below and their order should match the linux script
const windowsUserDataTemplate = `<powershell>
$TASK_TOKEN = "{}"
$logGroupName="{}"
$runnerNamePath="{}"
$githubDomainPath="{}"
$ownerPath="{}"
$repoPath="{}"
$runnerTokenPath="{}"
$labels="{}"
$registrationURL="{}"
$runnerGroup1="{}"
$runnerGroup2="{}"
$defaultLabels="{}"

# EC2Launch only starts ssm agent after user data is done, so we need to start it ourselves (it is disabled by default)
Set-Service -StartupType Manual AmazonSSMAgent
Start-Service AmazonSSMAgent

Start-Job -ScriptBlock {
  while (1) {
    aws stepfunctions send-task-heartbeat --task-token "$using:TASK_TOKEN"
    sleep 60
  }
}
function setup_logs () {
  echo "{
    \`"logs\`": {
      \`"log_stream_name\`": \`"unknown\`",
      \`"logs_collected\`": {
        \`"files\`": {
         \`"collect_list\`": [
            {
              \`"file_path\`": \`"/actions/runner.log\`",
              \`"log_group_name\`": \`"$logGroupName\`",
              \`"log_stream_name\`": \`"$runnerNamePath\`",
              \`"timezone\`": \`"UTC\`"
            }
          ]
        }
      }
    }
  }" | Out-File -Encoding ASCII $Env:TEMP/log.conf
  & "C:/Program Files/Amazon/AmazonCloudWatchAgent/amazon-cloudwatch-agent-ctl.ps1" -a fetch-config -m ec2 -s -c file:$Env:TEMP/log.conf
}
function action () {
  cd /actions
  $RunnerVersion = Get-Content /actions/RUNNER_VERSION -Raw
  if ($RunnerVersion -eq "latest") { $RunnerFlags = "" } else { $RunnerFlags = "--disableupdate" }
  ./config.cmd --unattended --url "\${registrationUrl}" --token "\${runnerTokenPath}" --ephemeral --work _work --labels "\${labels},cdkghr:started:$(Get-Date -UFormat +%s)" $RunnerFlags --name "\${runnerNamePath}" \${runnerGroup1} \${runnerGroup2} \${defaultLabels} 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log

  if ($LASTEXITCODE -ne 0) { return 1 }
  ./run.cmd 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 2 }

  $STATUS = Select-String -Path './_diag/*.log' -Pattern 'finish job request for job [0-9a-f\\-]+ with result: (.*)' | %{$_.Matches.Groups[1].Value} | Select-Object -Last 1

  if ($STATUS) {
      echo "CDKGHA JOB DONE \${labels} $STATUS" | Out-File -Encoding ASCII -Append /actions/runner.log
  }

  return 0
}
setup_logs
$r = action
if ($r -eq 0) {
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{ }'
} else {
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
}
Start-Sleep -Seconds 10  # give cloudwatch agent its default 5 seconds buffer duration to upload logs
Stop-Computer -ComputerName localhost -Force
</powershell>
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
/**
 * GitHub Actions runner provider using EC2 to execute jobs.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class Ec2RunnerProvider extends common_1.BaseProvider {
    /**
     * Create new image builder that builds EC2 specific runner images.
     *
     * You can customize the OS, architecture, VPC, subnet, security groups, etc. by passing in props.
     *
     * You can add components to the image builder by calling `imageBuilder.addComponent()`.
     *
     * The default OS is Ubuntu running on x64 architecture.
     *
     * Included components:
     *  * `RunnerImageComponent.requiredPackages()`
     *  * `RunnerImageComponent.cloudWatchAgent()`
     *  * `RunnerImageComponent.runnerUser()`
     *  * `RunnerImageComponent.git()`
     *  * `RunnerImageComponent.githubCli()`
     *  * `RunnerImageComponent.awsCli()`
     *  * `RunnerImageComponent.docker()`
     *  * `RunnerImageComponent.githubRunner()`
     */
    static imageBuilder(scope, id, props) {
        return image_builders_1.RunnerImageBuilder.new(scope, id, {
            os: common_1.Os.LINUX_UBUNTU,
            architecture: common_1.Architecture.X86_64,
            builderType: image_builders_1.RunnerImageBuilderType.AWS_IMAGE_BUILDER,
            components: [
                image_builders_1.RunnerImageComponent.requiredPackages(),
                image_builders_1.RunnerImageComponent.cloudWatchAgent(),
                image_builders_1.RunnerImageComponent.runnerUser(),
                image_builders_1.RunnerImageComponent.git(),
                image_builders_1.RunnerImageComponent.githubCli(),
                image_builders_1.RunnerImageComponent.awsCli(),
                image_builders_1.RunnerImageComponent.docker(),
                image_builders_1.RunnerImageComponent.githubRunner(props?.runnerVersion ?? common_1.RunnerVersion.latest()),
            ],
            ...props,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.retryableErrors = [
            'Ec2.Ec2Exception',
            'States.Timeout',
        ];
        this.labels = props?.labels ?? ['ec2'];
        this.group = props?.group;
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'Default VPC', { isDefault: true });
        this.securityGroups = props?.securityGroup ? [props.securityGroup] : (props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })]);
        this.subnets = props?.subnet ? [props.subnet] : this.vpc.selectSubnets(props?.subnetSelection).subnets;
        this.instanceType = props?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M6I, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        this.storageSize = props?.storageSize ?? cdk.Size.gibibytes(30); // 30 is the minimum for Windows
        this.storageOptions = props?.storageOptions;
        this.spot = props?.spot ?? false;
        this.spotMaxPrice = props?.spotMaxPrice;
        this.defaultLabels = props?.defaultLabels ?? true;
        this.amiBuilder = props?.imageBuilder ?? props?.amiBuilder ?? Ec2RunnerProvider.imageBuilder(this, 'Ami Builder', {
            vpc: props?.vpc,
            subnetSelection: props?.subnetSelection,
            securityGroups: this.securityGroups,
        });
        this.ami = this.amiBuilder.bindAmi();
        if (this.amiBuilder instanceof image_builders_1.AwsImageBuilderRunnerImageBuilder) {
            if (this.amiBuilder.storageSize && this.storageSize.toBytes() < this.amiBuilder.storageSize.toBytes()) {
                throw new Error(`Runner storage size (${this.storageSize.toGibibytes()} GiB) must be at least the same as the image builder storage size (${this.amiBuilder.storageSize.toGibibytes()} GiB)`);
            }
        }
        if (!this.ami.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`AMI architecture (${this.ami.architecture.name}) doesn't match runner instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        this.grantPrincipal = this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['states:SendTaskFailure', 'states:SendTaskSuccess', 'states:SendTaskHeartbeat'],
            resources: ['*'], // no support for stateMachine.stateMachineArn :(
            conditions: {
                StringEquals: {
                    'aws:ResourceTag/aws:cloudformation:stack-id': cdk.Stack.of(this).stackId,
                },
            },
        }));
        this.grantPrincipal.addToPrincipalPolicy(utils_1.MINIMAL_EC2_SSM_SESSION_MANAGER_POLICY_STATEMENT);
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.logGroup.grantWrite(this);
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        // we need to build user data in two steps because passing the template as the first parameter to stepfunctions.JsonPath.format fails on syntax
        const params = [
            aws_cdk_lib_1.aws_stepfunctions.JsonPath.taskToken,
            this.logGroup.logGroupName,
            parameters.runnerNamePath,
            parameters.githubDomainPath,
            parameters.ownerPath,
            parameters.repoPath,
            parameters.runnerTokenPath,
            this.labels.join(','),
            parameters.registrationUrl,
            this.group ? '--runnergroup' : '',
            // this is split into 2 for powershell otherwise it will pass "--runnergroup name" as a single argument and config.sh will fail
            this.group ? this.group : '',
            this.defaultLabels ? '' : '--no-default-labels',
        ];
        const passUserData = new aws_cdk_lib_1.aws_stepfunctions.Pass(this, `${this.labels.join(', ')} data`, {
            parameters: {
                userdataTemplate: this.ami.os.is(common_1.Os.WINDOWS) ? windowsUserDataTemplate : linuxUserDataTemplate,
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2'),
        });
        // we use ec2:RunInstances because we must
        // we can't use fleets because they don't let us override user data, security groups or even disk size
        // we can't use requestSpotInstances because it doesn't support launch templates, and it's deprecated
        // ec2:RunInstances also seemed like the only one to immediately return an error when spot capacity is not available
        // we build a complicated chain of states here because ec2:RunInstances can only try one subnet at a time
        // if someone can figure out a good way to use Map for this, please open a PR
        // build a state for each subnet we want to try
        const instanceProfile = new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
            roles: [this.role.roleName],
        });
        const rootDeviceResource = (0, common_1.amiRootDevice)(this, this.ami.launchTemplate.launchTemplateId);
        rootDeviceResource.node.addDependency(this.amiBuilder);
        const subnetRunners = this.subnets.map((subnet, index) => {
            return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, `${this.labels.join(', ')} subnet${index + 1}`, {
                comment: subnet.subnetId,
                integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                service: 'ec2',
                action: 'runInstances',
                heartbeatTimeout: aws_cdk_lib_1.aws_stepfunctions.Timeout.duration(aws_cdk_lib_1.Duration.minutes(10)),
                parameters: {
                    LaunchTemplate: {
                        LaunchTemplateId: this.ami.launchTemplate.launchTemplateId,
                    },
                    MinCount: 1,
                    MaxCount: 1,
                    InstanceType: this.instanceType.toString(),
                    UserData: aws_cdk_lib_1.aws_stepfunctions.JsonPath.base64Encode(aws_cdk_lib_1.aws_stepfunctions.JsonPath.format(aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2.userdataTemplate'), ...params)),
                    InstanceInitiatedShutdownBehavior: aws_cdk_lib_1.aws_ec2.InstanceInitiatedShutdownBehavior.TERMINATE,
                    IamInstanceProfile: {
                        Arn: instanceProfile.attrArn,
                    },
                    MetadataOptions: {
                        HttpTokens: 'required',
                    },
                    SecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
                    SubnetId: subnet.subnetId,
                    BlockDeviceMappings: [{
                            DeviceName: rootDeviceResource.ref,
                            Ebs: {
                                DeleteOnTermination: true,
                                VolumeSize: this.storageSize.toGibibytes(),
                                VolumeType: this.storageOptions?.volumeType,
                                Iops: this.storageOptions?.iops,
                                Throughput: this.storageOptions?.throughput,
                            },
                        }],
                    InstanceMarketOptions: this.spot ? {
                        MarketType: 'spot',
                        SpotOptions: {
                            MaxPrice: this.spotMaxPrice,
                            SpotInstanceType: 'one-time',
                        },
                    } : undefined,
                    TagSpecifications: [
                        {
                            ResourceType: 'instance',
                            Tags: [{
                                    Key: 'GitHubRunners:Provider',
                                    Value: this.node.path,
                                }],
                        },
                        {
                            ResourceType: 'volume',
                            Tags: [{
                                    Key: 'GitHubRunners:Provider',
                                    Value: this.node.path,
                                }],
                        },
                    ],
                },
                iamResources: ['*'],
            });
        });
        // start with the first subnet
        passUserData.next(subnetRunners[0]);
        // chain up the rest of the subnets
        for (let i = 1; i < subnetRunners.length; i++) {
            subnetRunners[i - 1].addCatch(subnetRunners[i], {
                errors: ['Ec2.Ec2Exception', 'States.Timeout'],
                resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.lastSubnetError'),
            });
        }
        return passUserData;
    }
    grantStateMachine(stateMachineRole) {
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [this.role.roleArn],
            conditions: {
                StringEquals: {
                    'iam:PassedToService': 'ec2.amazonaws.com',
                },
            },
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:createTags'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'ec2',
                    resource: '*',
                })],
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
            conditions: {
                StringEquals: {
                    'iam:AWSServiceName': 'spot.amazonaws.com',
                },
            },
        }));
    }
    status(statusFunctionRole) {
        statusFunctionRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:DescribeLaunchTemplateVersions'],
            resources: ['*'],
        }));
        return {
            type: this.constructor.name,
            labels: this.labels,
            securityGroups: this.securityGroups.map(sg => sg.securityGroupId),
            roleArn: this.role.roleArn,
            logGroup: this.logGroup.logGroupName,
            ami: {
                launchTemplate: this.ami.launchTemplate.launchTemplateId || 'unknown',
                amiBuilderLogGroup: this.ami.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
}
exports.Ec2RunnerProvider = Ec2RunnerProvider;
_a = JSII_RTTI_SYMBOL_1;
Ec2RunnerProvider[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.Ec2RunnerProvider", version: "0.14.15" };
/**
 * @deprecated use {@link Ec2RunnerProvider}
 */
class Ec2Runner extends Ec2RunnerProvider {
}
exports.Ec2Runner = Ec2Runner;
_b = JSII_RTTI_SYMBOL_1;
Ec2Runner[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Ec2Runner", version: "0.14.15" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWMyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3Byb3ZpZGVycy9lYzIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxtQ0FBbUM7QUFDbkMsNkNBU3FCO0FBQ3JCLG1EQUFxRDtBQUNyRCxxRUFBbUU7QUFFbkUscUNBWWtCO0FBQ2xCLHNEQU8yQjtBQUMzQixvQ0FBNEU7QUFFNUUsNkVBQTZFO0FBQzdFLHFEQUFxRDtBQUNyRCxNQUFNLHFCQUFxQixHQUFHOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Q0F3RTdCLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFFckUsNkVBQTZFO0FBQzdFLG1HQUFtRztBQUNuRyxNQUFNLHVCQUF1QixHQUFHOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Q0F3RS9CLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLENBQUM7QUF1SHJFOzs7O0dBSUc7QUFDSCxNQUFhLGlCQUFrQixTQUFRLHFCQUFZO0lBQ2pEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCO1FBQ3RGLE9BQU8sbUNBQWtCLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDdkMsRUFBRSxFQUFFLFdBQUUsQ0FBQyxZQUFZO1lBQ25CLFlBQVksRUFBRSxxQkFBWSxDQUFDLE1BQU07WUFDakMsV0FBVyxFQUFFLHVDQUFzQixDQUFDLGlCQUFpQjtZQUNyRCxVQUFVLEVBQUU7Z0JBQ1YscUNBQW9CLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQ3ZDLHFDQUFvQixDQUFDLGVBQWUsRUFBRTtnQkFDdEMscUNBQW9CLENBQUMsVUFBVSxFQUFFO2dCQUNqQyxxQ0FBb0IsQ0FBQyxHQUFHLEVBQUU7Z0JBQzFCLHFDQUFvQixDQUFDLFNBQVMsRUFBRTtnQkFDaEMscUNBQW9CLENBQUMsTUFBTSxFQUFFO2dCQUM3QixxQ0FBb0IsQ0FBQyxNQUFNLEVBQUU7Z0JBQzdCLHFDQUFvQixDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsYUFBYSxJQUFJLHNCQUFhLENBQUMsTUFBTSxFQUFFLENBQUM7YUFDbEY7WUFDRCxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBc0NELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBOEI7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFwQmpCLG9CQUFlLEdBQUc7WUFDekIsa0JBQWtCO1lBQ2xCLGdCQUFnQjtTQUNqQixDQUFDO1FBbUJBLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxFQUFFLE1BQU0sSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxFQUFFLEtBQUssQ0FBQztRQUMxQixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssRUFBRSxHQUFHLElBQUkscUJBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUN0RixJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxjQUFjLElBQUksQ0FBQyxJQUFJLHFCQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZKLElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDdkcsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLEVBQUUsWUFBWSxJQUFJLHFCQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxxQkFBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUscUJBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUcsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLEVBQUUsV0FBVyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsZ0NBQWdDO1FBQ2pHLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxFQUFFLGNBQWMsQ0FBQztRQUM1QyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssRUFBRSxJQUFJLElBQUksS0FBSyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxFQUFFLFlBQVksQ0FBQztRQUN4QyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssRUFBRSxhQUFhLElBQUksSUFBSSxDQUFDO1FBRWxELElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxFQUFFLFlBQVksSUFBSSxLQUFLLEVBQUUsVUFBVSxJQUFJLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ2hILEdBQUcsRUFBRSxLQUFLLEVBQUUsR0FBRztZQUNmLGVBQWUsRUFBRSxLQUFLLEVBQUUsZUFBZTtZQUN2QyxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7U0FDcEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRXJDLElBQUksSUFBSSxDQUFDLFVBQVUsWUFBWSxrREFBaUMsRUFBRSxDQUFDO1lBQ2pFLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUUsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO2dCQUN0RyxNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixJQUFJLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRSxzRUFBc0UsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2hNLENBQUM7UUFDSCxDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDO1lBQ2hFLE1BQU0sSUFBSSxLQUFLLENBQUMscUJBQXFCLElBQUksQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLElBQUkseUNBQXlDLElBQUksQ0FBQyxZQUFZLE1BQU0sSUFBSSxDQUFDLFlBQVksQ0FBQyxZQUFZLEdBQUcsQ0FBQyxDQUFDO1FBQ3BLLENBQUM7UUFFRCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxxQkFBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQzNELFNBQVMsRUFBRSxJQUFJLHFCQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDekQsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO1lBQy9ELE9BQU8sRUFBRSxDQUFDLHdCQUF3QixFQUFFLHdCQUF3QixFQUFFLDBCQUEwQixDQUFDO1lBQ3pGLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLGlEQUFpRDtZQUNuRSxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFO29CQUNaLDZDQUE2QyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU87aUJBQzFFO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsd0RBQWdELENBQUMsQ0FBQztRQUUzRixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksc0JBQUksQ0FBQyxRQUFRLENBQy9CLElBQUksRUFDSixNQUFNLEVBQ047WUFDRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFlBQVksSUFBSSx3QkFBYSxDQUFDLFNBQVM7WUFDekQsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTztTQUNyQyxDQUNGLENBQUM7UUFDRixJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsbUJBQW1CLENBQUMsVUFBbUM7UUFDckQsK0lBQStJO1FBRS9JLE1BQU0sTUFBTSxHQUFHO1lBQ2IsK0JBQWEsQ0FBQyxRQUFRLENBQUMsU0FBUztZQUNoQyxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVk7WUFDMUIsVUFBVSxDQUFDLGNBQWM7WUFDekIsVUFBVSxDQUFDLGdCQUFnQjtZQUMzQixVQUFVLENBQUMsU0FBUztZQUNwQixVQUFVLENBQUMsUUFBUTtZQUNuQixVQUFVLENBQUMsZUFBZTtZQUMxQixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7WUFDckIsVUFBVSxDQUFDLGVBQWU7WUFDMUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2pDLCtIQUErSDtZQUMvSCxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzVCLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMscUJBQXFCO1NBQ2hELENBQUM7UUFFRixNQUFNLFlBQVksR0FBRyxJQUFJLCtCQUFhLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDbEYsVUFBVSxFQUFFO2dCQUNWLGdCQUFnQixFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxXQUFFLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQyxxQkFBcUI7YUFDL0Y7WUFDRCxVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQztTQUNyRCxDQUFDLENBQUM7UUFFSCwwQ0FBMEM7UUFDMUMsc0dBQXNHO1FBQ3RHLHFHQUFxRztRQUNyRyxvSEFBb0g7UUFFcEgseUdBQXlHO1FBQ3pHLDZFQUE2RTtRQUU3RSwrQ0FBK0M7UUFDL0MsTUFBTSxlQUFlLEdBQUcsSUFBSSxxQkFBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUMzRSxLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztTQUM1QixDQUFDLENBQUM7UUFDSCxNQUFNLGtCQUFrQixHQUFHLElBQUEsc0JBQWEsRUFBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUN6RixrQkFBa0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN2RCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsRUFBRTtZQUN2RCxPQUFPLElBQUkscUNBQW1CLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEtBQUssR0FBQyxDQUFDLEVBQUUsRUFBRTtnQkFDaEcsT0FBTyxFQUFFLE1BQU0sQ0FBQyxRQUFRO2dCQUN4QixrQkFBa0IsRUFBRSxzQ0FBa0IsQ0FBQyxtQkFBbUI7Z0JBQzFELE9BQU8sRUFBRSxLQUFLO2dCQUNkLE1BQU0sRUFBRSxjQUFjO2dCQUN0QixnQkFBZ0IsRUFBRSwrQkFBYSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQ3RFLFVBQVUsRUFBRTtvQkFDVixjQUFjLEVBQUU7d0JBQ2QsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCO3FCQUMzRDtvQkFDRCxRQUFRLEVBQUUsQ0FBQztvQkFDWCxRQUFRLEVBQUUsQ0FBQztvQkFDWCxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUU7b0JBQzFDLFFBQVEsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQzNDLCtCQUFhLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FDM0IsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLHdCQUF3QixDQUFDLEVBQ3pELEdBQUcsTUFBTSxDQUNWLENBQ0Y7b0JBQ0QsaUNBQWlDLEVBQUUscUJBQUcsQ0FBQyxpQ0FBaUMsQ0FBQyxTQUFTO29CQUNsRixrQkFBa0IsRUFBRTt3QkFDbEIsR0FBRyxFQUFFLGVBQWUsQ0FBQyxPQUFPO3FCQUM3QjtvQkFDRCxlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLFVBQVU7cUJBQ3ZCO29CQUNELGdCQUFnQixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQztvQkFDbkUsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO29CQUN6QixtQkFBbUIsRUFBRSxDQUFDOzRCQUNwQixVQUFVLEVBQUUsa0JBQWtCLENBQUMsR0FBRzs0QkFDbEMsR0FBRyxFQUFFO2dDQUNILG1CQUFtQixFQUFFLElBQUk7Z0NBQ3pCLFVBQVUsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRTtnQ0FDMUMsVUFBVSxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsVUFBVTtnQ0FDM0MsSUFBSSxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsSUFBSTtnQ0FDL0IsVUFBVSxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsVUFBVTs2QkFDNUM7eUJBQ0YsQ0FBQztvQkFDRixxQkFBcUIsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQzt3QkFDakMsVUFBVSxFQUFFLE1BQU07d0JBQ2xCLFdBQVcsRUFBRTs0QkFDWCxRQUFRLEVBQUUsSUFBSSxDQUFDLFlBQVk7NEJBQzNCLGdCQUFnQixFQUFFLFVBQVU7eUJBQzdCO3FCQUNGLENBQUMsQ0FBQyxDQUFDLFNBQVM7b0JBQ2IsaUJBQWlCLEVBQUU7d0JBQ2pCOzRCQUNFLFlBQVksRUFBRSxVQUFVOzRCQUN4QixJQUFJLEVBQUUsQ0FBQztvQ0FDTCxHQUFHLEVBQUUsd0JBQXdCO29DQUM3QixLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJO2lDQUN0QixDQUFDO3lCQUNIO3dCQUNEOzRCQUNFLFlBQVksRUFBRSxRQUFROzRCQUN0QixJQUFJLEVBQUUsQ0FBQztvQ0FDTCxHQUFHLEVBQUUsd0JBQXdCO29DQUM3QixLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJO2lDQUN0QixDQUFDO3lCQUNIO3FCQUNGO2lCQUNGO2dCQUNELFlBQVksRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNwQixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILDhCQUE4QjtRQUM5QixZQUFZLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXBDLG1DQUFtQztRQUNuQyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsYUFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQzlDLGFBQWEsQ0FBQyxDQUFDLEdBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDNUMsTUFBTSxFQUFFLENBQUMsa0JBQWtCLEVBQUUsZ0JBQWdCLENBQUM7Z0JBQzlDLFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUM7YUFDakUsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxnQkFBZ0M7UUFDaEQsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDM0UsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3pCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO1lBQzlCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUU7b0JBQ1oscUJBQXFCLEVBQUUsbUJBQW1CO2lCQUMzQzthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUMzRSxPQUFPLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQztZQUMzQixTQUFTLEVBQUUsQ0FBQyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ25DLE9BQU8sRUFBRSxLQUFLO29CQUNkLFFBQVEsRUFBRSxHQUFHO2lCQUNkLENBQUMsQ0FBQztTQUNKLENBQUMsQ0FBQyxDQUFDO1FBRUosZ0JBQWdCLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDM0UsT0FBTyxFQUFFLENBQUMsNkJBQTZCLENBQUM7WUFDeEMsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUU7b0JBQ1osb0JBQW9CLEVBQUUsb0JBQW9CO2lCQUMzQzthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDO0lBRUQsTUFBTSxDQUFDLGtCQUFrQztRQUN2QyxrQkFBa0IsQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUM3RSxPQUFPLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQztZQUMvQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1lBQ0wsSUFBSSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSTtZQUMzQixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07WUFDbkIsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQztZQUNqRSxPQUFPLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQzFCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVk7WUFDcEMsR0FBRyxFQUFFO2dCQUNILGNBQWMsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsSUFBSSxTQUFTO2dCQUNyRSxrQkFBa0IsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxZQUFZO2FBQ3BEO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsV0FBVztRQUNwQixPQUFPLElBQUkscUJBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7SUFDdEUsQ0FBQzs7QUExVEgsOENBMlRDOzs7QUFFRDs7R0FFRztBQUNILE1BQWEsU0FBVSxTQUFRLGlCQUFpQjs7QUFBaEQsOEJBQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHtcbiAgYXdzX2VjMiBhcyBlYzIsXG4gIGF3c19pYW0gYXMgaWFtLFxuICBhd3NfbG9ncyBhcyBsb2dzLFxuICBhd3Nfc3RlcGZ1bmN0aW9ucyBhcyBzdGVwZnVuY3Rpb25zLFxuICBhd3Nfc3RlcGZ1bmN0aW9uc190YXNrcyBhcyBzdGVwZnVuY3Rpb25zX3Rhc2tzLFxuICBEdXJhdGlvbixcbiAgUmVtb3ZhbFBvbGljeSxcbiAgU3RhY2ssXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgeyBJbnRlZ3JhdGlvblBhdHRlcm4gfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7XG4gIGFtaVJvb3REZXZpY2UsXG4gIEFyY2hpdGVjdHVyZSxcbiAgQmFzZVByb3ZpZGVyLFxuICBJUnVubmVyUHJvdmlkZXIsXG4gIElSdW5uZXJQcm92aWRlclN0YXR1cyxcbiAgT3MsXG4gIFJ1bm5lckFtaSxcbiAgUnVubmVyUHJvdmlkZXJQcm9wcyxcbiAgUnVubmVyUnVudGltZVBhcmFtZXRlcnMsXG4gIFJ1bm5lclZlcnNpb24sXG4gIFN0b3JhZ2VPcHRpb25zLFxufSBmcm9tICcuL2NvbW1vbic7XG5pbXBvcnQge1xuICBBd3NJbWFnZUJ1aWxkZXJSdW5uZXJJbWFnZUJ1aWxkZXIsXG4gIElSdW5uZXJJbWFnZUJ1aWxkZXIsXG4gIFJ1bm5lckltYWdlQnVpbGRlcixcbiAgUnVubmVySW1hZ2VCdWlsZGVyUHJvcHMsXG4gIFJ1bm5lckltYWdlQnVpbGRlclR5cGUsXG4gIFJ1bm5lckltYWdlQ29tcG9uZW50LFxufSBmcm9tICcuLi9pbWFnZS1idWlsZGVycyc7XG5pbXBvcnQgeyBNSU5JTUFMX0VDMl9TU01fU0VTU0lPTl9NQU5BR0VSX1BPTElDWV9TVEFURU1FTlQgfSBmcm9tICcuLi91dGlscyc7XG5cbi8vIHRoaXMgc2NyaXB0IGlzIHNwZWNpZmljYWxseSBtYWRlIHNvIGBwb3dlcm9mZmAgaXMgYWJzb2x1dGVseSBhbHdheXMgY2FsbGVkXG4vLyBlYWNoIGB7fWAgaXMgYSB2YXJpYWJsZSBjb21pbmcgZnJvbSBgcGFyYW1zYCBiZWxvd1xuY29uc3QgbGludXhVc2VyRGF0YVRlbXBsYXRlID0gYCMhL2Jpbi9iYXNoIC14XG5UQVNLX1RPS0VOPVwie31cIlxubG9nR3JvdXBOYW1lPVwie31cIlxucnVubmVyTmFtZVBhdGg9XCJ7fVwiXG5naXRodWJEb21haW5QYXRoPVwie31cIlxub3duZXJQYXRoPVwie31cIlxucmVwb1BhdGg9XCJ7fVwiXG5ydW5uZXJUb2tlblBhdGg9XCJ7fVwiXG5sYWJlbHM9XCJ7fVwiXG5yZWdpc3RyYXRpb25VUkw9XCJ7fVwiXG5ydW5uZXJHcm91cDE9XCJ7fVwiXG5ydW5uZXJHcm91cDI9XCJ7fVwiXG5kZWZhdWx0TGFiZWxzPVwie31cIlxuXG5oZWFydGJlYXQgKCkge1xuICB3aGlsZSB0cnVlOyBkb1xuICAgIGF3cyBzdGVwZnVuY3Rpb25zIHNlbmQtdGFzay1oZWFydGJlYXQgLS10YXNrLXRva2VuIFwiJFRBU0tfVE9LRU5cIlxuICAgIHNsZWVwIDYwXG4gIGRvbmVcbn1cbnNldHVwX2xvZ3MgKCkge1xuICBjYXQgPDxFT0YgPiAvdG1wL2xvZy5jb25mIHx8IGV4aXQgMVxuICB7XG4gICAgXCJsb2dzXCI6IHtcbiAgICAgIFwibG9nX3N0cmVhbV9uYW1lXCI6IFwidW5rbm93blwiLFxuICAgICAgXCJsb2dzX2NvbGxlY3RlZFwiOiB7XG4gICAgICAgIFwiZmlsZXNcIjoge1xuICAgICAgICAgIFwiY29sbGVjdF9saXN0XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJmaWxlX3BhdGhcIjogXCIvdmFyL2xvZy9ydW5uZXIubG9nXCIsXG4gICAgICAgICAgICAgIFwibG9nX2dyb3VwX25hbWVcIjogXCIkbG9nR3JvdXBOYW1lXCIsXG4gICAgICAgICAgICAgIFwibG9nX3N0cmVhbV9uYW1lXCI6IFwiJHJ1bm5lck5hbWVQYXRoXCIsXG4gICAgICAgICAgICAgIFwidGltZXpvbmVcIjogXCJVVENcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfVxuRU9GXG4gIC9vcHQvYXdzL2FtYXpvbi1jbG91ZHdhdGNoLWFnZW50L2Jpbi9hbWF6b24tY2xvdWR3YXRjaC1hZ2VudC1jdGwgLWEgZmV0Y2gtY29uZmlnIC1tIGVjMiAtcyAtYyBmaWxlOi90bXAvbG9nLmNvbmYgfHwgZXhpdCAyXG59XG5hY3Rpb24gKCkge1xuICAjIERldGVybWluZSB0aGUgdmFsdWUgb2YgUlVOTkVSX0ZMQUdTXG4gIGlmIFsgXCIkKDwgL2hvbWUvcnVubmVyL1JVTk5FUl9WRVJTSU9OKVwiID0gXCJsYXRlc3RcIiBdOyB0aGVuXG4gICAgUlVOTkVSX0ZMQUdTPVwiXCJcbiAgZWxzZVxuICAgIFJVTk5FUl9GTEFHUz1cIi0tZGlzYWJsZXVwZGF0ZVwiXG4gIGZpXG5cbiAgbGFiZWxzVGVtcGxhdGU9XCIkbGFiZWxzLGNka2docjpzdGFydGVkOiQoZGF0ZSArJXMpXCJcblxuICAjIEV4ZWN1dGUgdGhlIGNvbmZpZ3VyYXRpb24gY29tbWFuZCBmb3IgcnVubmVyIHJlZ2lzdHJhdGlvblxuICBzdWRvIC1IdSBydW5uZXIgL2hvbWUvcnVubmVyL2NvbmZpZy5zaCAtLXVuYXR0ZW5kZWQgLS11cmwgXCIkcmVnaXN0cmF0aW9uVVJMXCIgLS10b2tlbiBcIiRydW5uZXJUb2tlblBhdGhcIiAtLWVwaGVtZXJhbCAtLXdvcmsgX3dvcmsgLS1sYWJlbHMgXCIkbGFiZWxzVGVtcGxhdGVcIiAkUlVOTkVSX0ZMQUdTIC0tbmFtZSBcIiRydW5uZXJOYW1lUGF0aFwiICRydW5uZXJHcm91cDEgJHJ1bm5lckdyb3VwMiAkZGVmYXVsdExhYmVscyB8fCBleGl0IDFcblxuICAjIEV4ZWN1dGUgdGhlIHJ1biBjb21tYW5kXG4gIHN1ZG8gLS1wcmVzZXJ2ZS1lbnY9QVdTX1JFR0lPTiAtSHUgcnVubmVyIC9ob21lL3J1bm5lci9ydW4uc2ggfHwgZXhpdCAyXG5cbiAgIyBSZXRyaWV2ZSB0aGUgc3RhdHVzXG4gIFNUQVRVUz0kKGdyZXAgLVBob3JzIFwiZmluaXNoIGpvYiByZXF1ZXN0IGZvciBqb2IgWzAtOWEtZlxcXFwtXSsgd2l0aCByZXN1bHQ6IFxcSy4qXCIgL2hvbWUvcnVubmVyL19kaWFnLyB8IHRhaWwgLW4xKVxuXG4gICMgQ2hlY2sgYW5kIHByaW50IHRoZSBqb2Igc3RhdHVzXG4gIFsgLW4gXCIkU1RBVFVTXCIgXSAmJiBlY2hvIENES0dIQSBKT0IgRE9ORSBcIiRsYWJlbHNcIiBcIiRTVEFUVVNcIlxufVxuaGVhcnRiZWF0ICZcbmlmIHNldHVwX2xvZ3MgJiYgYWN0aW9uIHwgdGVlIC92YXIvbG9nL3J1bm5lci5sb2cgMj4mMTsgdGhlblxuICBhd3Mgc3RlcGZ1bmN0aW9ucyBzZW5kLXRhc2stc3VjY2VzcyAtLXRhc2stdG9rZW4gXCIkVEFTS19UT0tFTlwiIC0tdGFzay1vdXRwdXQgJ3tcIm9rXCI6IHRydWV9J1xuZWxzZVxuICBhd3Mgc3RlcGZ1bmN0aW9ucyBzZW5kLXRhc2stZmFpbHVyZSAtLXRhc2stdG9rZW4gXCIkVEFTS19UT0tFTlwiXG5maVxuc2xlZXAgMTAgICMgZ2l2ZSBjbG91ZHdhdGNoIGFnZW50IGl0cyBkZWZhdWx0IDUgc2Vjb25kcyBidWZmZXIgZHVyYXRpb24gdG8gdXBsb2FkIGxvZ3NcbnBvd2Vyb2ZmXG5gLnJlcGxhY2UoL3svZywgJ1xcXFx7JykucmVwbGFjZSgvfS9nLCAnXFxcXH0nKS5yZXBsYWNlKC9cXFxce1xcXFx9L2csICd7fScpO1xuXG4vLyB0aGlzIHNjcmlwdCBpcyBzcGVjaWZpY2FsbHkgbWFkZSBzbyBgcG93ZXJvZmZgIGlzIGFic29sdXRlbHkgYWx3YXlzIGNhbGxlZFxuLy8gZWFjaCBge31gIGlzIGEgdmFyaWFibGUgY29taW5nIGZyb20gYHBhcmFtc2AgYmVsb3cgYW5kIHRoZWlyIG9yZGVyIHNob3VsZCBtYXRjaCB0aGUgbGludXggc2NyaXB0XG5jb25zdCB3aW5kb3dzVXNlckRhdGFUZW1wbGF0ZSA9IGA8cG93ZXJzaGVsbD5cbiRUQVNLX1RPS0VOID0gXCJ7fVwiXG4kbG9nR3JvdXBOYW1lPVwie31cIlxuJHJ1bm5lck5hbWVQYXRoPVwie31cIlxuJGdpdGh1YkRvbWFpblBhdGg9XCJ7fVwiXG4kb3duZXJQYXRoPVwie31cIlxuJHJlcG9QYXRoPVwie31cIlxuJHJ1bm5lclRva2VuUGF0aD1cInt9XCJcbiRsYWJlbHM9XCJ7fVwiXG4kcmVnaXN0cmF0aW9uVVJMPVwie31cIlxuJHJ1bm5lckdyb3VwMT1cInt9XCJcbiRydW5uZXJHcm91cDI9XCJ7fVwiXG4kZGVmYXVsdExhYmVscz1cInt9XCJcblxuIyBFQzJMYXVuY2ggb25seSBzdGFydHMgc3NtIGFnZW50IGFmdGVyIHVzZXIgZGF0YSBpcyBkb25lLCBzbyB3ZSBuZWVkIHRvIHN0YXJ0IGl0IG91cnNlbHZlcyAoaXQgaXMgZGlzYWJsZWQgYnkgZGVmYXVsdClcblNldC1TZXJ2aWNlIC1TdGFydHVwVHlwZSBNYW51YWwgQW1hem9uU1NNQWdlbnRcblN0YXJ0LVNlcnZpY2UgQW1hem9uU1NNQWdlbnRcblxuU3RhcnQtSm9iIC1TY3JpcHRCbG9jayB7XG4gIHdoaWxlICgxKSB7XG4gICAgYXdzIHN0ZXBmdW5jdGlvbnMgc2VuZC10YXNrLWhlYXJ0YmVhdCAtLXRhc2stdG9rZW4gXCIkdXNpbmc6VEFTS19UT0tFTlwiXG4gICAgc2xlZXAgNjBcbiAgfVxufVxuZnVuY3Rpb24gc2V0dXBfbG9ncyAoKSB7XG4gIGVjaG8gXCJ7XG4gICAgXFxgXCJsb2dzXFxgXCI6IHtcbiAgICAgIFxcYFwibG9nX3N0cmVhbV9uYW1lXFxgXCI6IFxcYFwidW5rbm93blxcYFwiLFxuICAgICAgXFxgXCJsb2dzX2NvbGxlY3RlZFxcYFwiOiB7XG4gICAgICAgIFxcYFwiZmlsZXNcXGBcIjoge1xuICAgICAgICAgXFxgXCJjb2xsZWN0X2xpc3RcXGBcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcXGBcImZpbGVfcGF0aFxcYFwiOiBcXGBcIi9hY3Rpb25zL3J1bm5lci5sb2dcXGBcIixcbiAgICAgICAgICAgICAgXFxgXCJsb2dfZ3JvdXBfbmFtZVxcYFwiOiBcXGBcIiRsb2dHcm91cE5hbWVcXGBcIixcbiAgICAgICAgICAgICAgXFxgXCJsb2dfc3RyZWFtX25hbWVcXGBcIjogXFxgXCIkcnVubmVyTmFtZVBhdGhcXGBcIixcbiAgICAgICAgICAgICAgXFxgXCJ0aW1lem9uZVxcYFwiOiBcXGBcIlVUQ1xcYFwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XCIgfCBPdXQtRmlsZSAtRW5jb2RpbmcgQVNDSUkgJEVudjpURU1QL2xvZy5jb25mXG4gICYgXCJDOi9Qcm9ncmFtIEZpbGVzL0FtYXpvbi9BbWF6b25DbG91ZFdhdGNoQWdlbnQvYW1hem9uLWNsb3Vkd2F0Y2gtYWdlbnQtY3RsLnBzMVwiIC1hIGZldGNoLWNvbmZpZyAtbSBlYzIgLXMgLWMgZmlsZTokRW52OlRFTVAvbG9nLmNvbmZcbn1cbmZ1bmN0aW9uIGFjdGlvbiAoKSB7XG4gIGNkIC9hY3Rpb25zXG4gICRSdW5uZXJWZXJzaW9uID0gR2V0LUNvbnRlbnQgL2FjdGlvbnMvUlVOTkVSX1ZFUlNJT04gLVJhd1xuICBpZiAoJFJ1bm5lclZlcnNpb24gLWVxIFwibGF0ZXN0XCIpIHsgJFJ1bm5lckZsYWdzID0gXCJcIiB9IGVsc2UgeyAkUnVubmVyRmxhZ3MgPSBcIi0tZGlzYWJsZXVwZGF0ZVwiIH1cbiAgLi9jb25maWcuY21kIC0tdW5hdHRlbmRlZCAtLXVybCBcIlxcJHtyZWdpc3RyYXRpb25Vcmx9XCIgLS10b2tlbiBcIlxcJHtydW5uZXJUb2tlblBhdGh9XCIgLS1lcGhlbWVyYWwgLS13b3JrIF93b3JrIC0tbGFiZWxzIFwiXFwke2xhYmVsc30sY2RrZ2hyOnN0YXJ0ZWQ6JChHZXQtRGF0ZSAtVUZvcm1hdCArJXMpXCIgJFJ1bm5lckZsYWdzIC0tbmFtZSBcIlxcJHtydW5uZXJOYW1lUGF0aH1cIiBcXCR7cnVubmVyR3JvdXAxfSBcXCR7cnVubmVyR3JvdXAyfSBcXCR7ZGVmYXVsdExhYmVsc30gMj4mMSB8IE91dC1GaWxlIC1FbmNvZGluZyBBU0NJSSAtQXBwZW5kIC9hY3Rpb25zL3J1bm5lci5sb2dcblxuICBpZiAoJExBU1RFWElUQ09ERSAtbmUgMCkgeyByZXR1cm4gMSB9XG4gIC4vcnVuLmNtZCAyPiYxIHwgT3V0LUZpbGUgLUVuY29kaW5nIEFTQ0lJIC1BcHBlbmQgL2FjdGlvbnMvcnVubmVyLmxvZ1xuICBpZiAoJExBU1RFWElUQ09ERSAtbmUgMCkgeyByZXR1cm4gMiB9XG5cbiAgJFNUQVRVUyA9IFNlbGVjdC1TdHJpbmcgLVBhdGggJy4vX2RpYWcvKi5sb2cnIC1QYXR0ZXJuICdmaW5pc2ggam9iIHJlcXVlc3QgZm9yIGpvYiBbMC05YS1mXFxcXC1dKyB3aXRoIHJlc3VsdDogKC4qKScgfCAleyRfLk1hdGNoZXMuR3JvdXBzWzFdLlZhbHVlfSB8IFNlbGVjdC1PYmplY3QgLUxhc3QgMVxuXG4gIGlmICgkU1RBVFVTKSB7XG4gICAgICBlY2hvIFwiQ0RLR0hBIEpPQiBET05FIFxcJHtsYWJlbHN9ICRTVEFUVVNcIiB8IE91dC1GaWxlIC1FbmNvZGluZyBBU0NJSSAtQXBwZW5kIC9hY3Rpb25zL3J1bm5lci5sb2dcbiAgfVxuXG4gIHJldHVybiAwXG59XG5zZXR1cF9sb2dzXG4kciA9IGFjdGlvblxuaWYgKCRyIC1lcSAwKSB7XG4gIGF3cyBzdGVwZnVuY3Rpb25zIHNlbmQtdGFzay1zdWNjZXNzIC0tdGFzay10b2tlbiBcIiRUQVNLX1RPS0VOXCIgLS10YXNrLW91dHB1dCAneyB9J1xufSBlbHNlIHtcbiAgYXdzIHN0ZXBmdW5jdGlvbnMgc2VuZC10YXNrLWZhaWx1cmUgLS10YXNrLXRva2VuIFwiJFRBU0tfVE9LRU5cIlxufVxuU3RhcnQtU2xlZXAgLVNlY29uZHMgMTAgICMgZ2l2ZSBjbG91ZHdhdGNoIGFnZW50IGl0cyBkZWZhdWx0IDUgc2Vjb25kcyBidWZmZXIgZHVyYXRpb24gdG8gdXBsb2FkIGxvZ3NcblN0b3AtQ29tcHV0ZXIgLUNvbXB1dGVyTmFtZSBsb2NhbGhvc3QgLUZvcmNlXG48L3Bvd2Vyc2hlbGw+XG5gLnJlcGxhY2UoL3svZywgJ1xcXFx7JykucmVwbGFjZSgvfS9nLCAnXFxcXH0nKS5yZXBsYWNlKC9cXFxce1xcXFx9L2csICd7fScpO1xuXG5cbi8qKlxuICogUHJvcGVydGllcyBmb3Ige0BsaW5rIEVjMlJ1bm5lclByb3ZpZGVyfSBjb25zdHJ1Y3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWMyUnVubmVyUHJvdmlkZXJQcm9wcyBleHRlbmRzIFJ1bm5lclByb3ZpZGVyUHJvcHMge1xuICAvKipcbiAgICogUnVubmVyIGltYWdlIGJ1aWxkZXIgdXNlZCB0byBidWlsZCBBTUkgY29udGFpbmluZyBHaXRIdWIgUnVubmVyIGFuZCBhbGwgcmVxdWlyZW1lbnRzLlxuICAgKlxuICAgKiBUaGUgaW1hZ2UgYnVpbGRlciBkZXRlcm1pbmVzIHRoZSBPUyBhbmQgYXJjaGl0ZWN0dXJlIG9mIHRoZSBydW5uZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IEVjMlJ1bm5lclByb3ZpZGVyLmltYWdlQnVpbGRlcigpXG4gICAqL1xuICByZWFkb25seSBpbWFnZUJ1aWxkZXI/OiBJUnVubmVySW1hZ2VCdWlsZGVyO1xuXG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgaW1hZ2VCdWlsZGVyXG4gICAqL1xuICByZWFkb25seSBhbWlCdWlsZGVyPzogSVJ1bm5lckltYWdlQnVpbGRlcjtcblxuICAvKipcbiAgICogR2l0SHViIEFjdGlvbnMgbGFiZWxzIHVzZWQgZm9yIHRoaXMgcHJvdmlkZXIuXG4gICAqXG4gICAqIFRoZXNlIGxhYmVscyBhcmUgdXNlZCB0byBpZGVudGlmeSB3aGljaCBwcm92aWRlciBzaG91bGQgc3Bhd24gYSBuZXcgb24tZGVtYW5kIHJ1bm5lci4gRXZlcnkgam9iIHNlbmRzIGEgd2ViaG9vayB3aXRoIHRoZSBsYWJlbHMgaXQncyBsb29raW5nIGZvclxuICAgKiBiYXNlZCBvbiBydW5zLW9uLiBXZSBtYXRjaCB0aGUgbGFiZWxzIGZyb20gdGhlIHdlYmhvb2sgd2l0aCB0aGUgbGFiZWxzIHNwZWNpZmllZCBoZXJlLiBJZiBhbGwgdGhlIGxhYmVscyBzcGVjaWZpZWQgaGVyZSBhcmUgcHJlc2VudCBpbiB0aGVcbiAgICogam9iJ3MgbGFiZWxzLCB0aGlzIHByb3ZpZGVyIHdpbGwgYmUgY2hvc2VuIGFuZCBzcGF3biBhIG5ldyBydW5uZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFsnZWMyJ11cbiAgICovXG4gIHJlYWRvbmx5IGxhYmVscz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBHaXRIdWIgQWN0aW9ucyBydW5uZXIgZ3JvdXAgbmFtZS5cbiAgICpcbiAgICogSWYgc3BlY2lmaWVkLCB0aGUgcnVubmVyIHdpbGwgYmUgcmVnaXN0ZXJlZCB3aXRoIHRoaXMgZ3JvdXAgbmFtZS4gU2V0dGluZyBhIHJ1bm5lciBncm91cCBjYW4gaGVscCBtYW5hZ2luZyBhY2Nlc3MgdG8gc2VsZi1ob3N0ZWQgcnVubmVycy4gSXRcbiAgICogcmVxdWlyZXMgYSBwYWlkIEdpdEh1YiBhY2NvdW50LlxuICAgKlxuICAgKiBUaGUgZ3JvdXAgbXVzdCBleGlzdCBvciB0aGUgcnVubmVyIHdpbGwgbm90IHN0YXJ0LlxuICAgKlxuICAgKiBVc2VycyB3aWxsIHN0aWxsIGJlIGFibGUgdG8gdHJpZ2dlciB0aGlzIHJ1bm5lciB3aXRoIHRoZSBjb3JyZWN0IGxhYmVscy4gQnV0IHRoZSBydW5uZXIgd2lsbCBvbmx5IGJlIGFibGUgdG8gcnVuIGpvYnMgZnJvbSByZXBvcyBhbGxvd2VkIHRvIHVzZSB0aGUgZ3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgZ3JvdXA/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluc3RhbmNlIHR5cGUgZm9yIGxhdW5jaGVkIHJ1bm5lciBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG02aS5sYXJnZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogU2l6ZSBvZiB2b2x1bWUgYXZhaWxhYmxlIGZvciBsYXVuY2hlZCBydW5uZXIgaW5zdGFuY2VzLiBUaGlzIG1vZGlmaWVzIHRoZSBib290IHZvbHVtZSBzaXplIGFuZCBkb2Vzbid0IGFkZCBhbnkgYWRkaXRpb25hbCB2b2x1bWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAzMEdCXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlU2l6ZT86IGNkay5TaXplO1xuXG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciBydW5uZXIgaW5zdGFuY2Ugc3RvcmFnZSB2b2x1bWUuXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlT3B0aW9ucz86IFN0b3JhZ2VPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBTZWN1cml0eSBHcm91cCB0byBhc3NpZ24gdG8gbGF1bmNoZWQgcnVubmVyIGluc3RhbmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgYSBuZXcgc2VjdXJpdHkgZ3JvdXBcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIHtAbGluayBzZWN1cml0eUdyb3Vwc31cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBlYzIuSVNlY3VyaXR5R3JvdXA7XG5cbiAgLyoqXG4gICAqIFNlY3VyaXR5IGdyb3VwcyB0byBhc3NpZ24gdG8gbGF1bmNoZWQgcnVubmVyIGluc3RhbmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgYSBuZXcgc2VjdXJpdHkgZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogZWMyLklTZWN1cml0eUdyb3VwW107XG5cbiAgLyoqXG4gICAqIFN1Ym5ldCB3aGVyZSB0aGUgcnVubmVyIGluc3RhbmNlcyB3aWxsIGJlIGxhdW5jaGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBkZWZhdWx0IHN1Ym5ldCBvZiBhY2NvdW50J3MgZGVmYXVsdCBWUENcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIHtAbGluayB2cGN9IGFuZCB7QGxpbmsgc3VibmV0U2VsZWN0aW9ufVxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0PzogZWMyLklTdWJuZXQ7XG5cbiAgLyoqXG4gICAqIFZQQyB3aGVyZSBydW5uZXIgaW5zdGFuY2VzIHdpbGwgYmUgbGF1bmNoZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGRlZmF1bHQgYWNjb3VudCBWUENcbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBXaGVyZSB0byBwbGFjZSB0aGUgbmV0d29yayBpbnRlcmZhY2VzIHdpdGhpbiB0aGUgVlBDLiBPbmx5IHRoZSBmaXJzdCBtYXRjaGVkIHN1Ym5ldCB3aWxsIGJlIHVzZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGRlZmF1bHQgVlBDIHN1Ym5ldFxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVXNlIHNwb3QgaW5zdGFuY2VzIHRvIHNhdmUgbW9uZXkuIFNwb3QgaW5zdGFuY2VzIGFyZSBjaGVhcGVyIGJ1dCBub3QgYWx3YXlzIGF2YWlsYWJsZSBhbmQgY2FuIGJlIHN0b3BwZWQgcHJlbWF0dXJlbHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBzcG90PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogU2V0IGEgbWF4aW11bSBwcmljZSBmb3Igc3BvdCBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vIG1heCBwcmljZSAoeW91IHdpbGwgcGF5IGN1cnJlbnQgc3BvdCBwcmljZSlcbiAgICovXG4gIHJlYWRvbmx5IHNwb3RNYXhQcmljZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBHaXRIdWIgQWN0aW9ucyBydW5uZXIgcHJvdmlkZXIgdXNpbmcgRUMyIHRvIGV4ZWN1dGUgam9icy5cbiAqXG4gKiBUaGlzIGNvbnN0cnVjdCBpcyBub3QgbWVhbnQgdG8gYmUgdXNlZCBieSBpdHNlbGYuIEl0IHNob3VsZCBiZSBwYXNzZWQgaW4gdGhlIHByb3ZpZGVycyBwcm9wZXJ0eSBmb3IgR2l0SHViUnVubmVycy5cbiAqL1xuZXhwb3J0IGNsYXNzIEVjMlJ1bm5lclByb3ZpZGVyIGV4dGVuZHMgQmFzZVByb3ZpZGVyIGltcGxlbWVudHMgSVJ1bm5lclByb3ZpZGVyIHtcbiAgLyoqXG4gICAqIENyZWF0ZSBuZXcgaW1hZ2UgYnVpbGRlciB0aGF0IGJ1aWxkcyBFQzIgc3BlY2lmaWMgcnVubmVyIGltYWdlcy5cbiAgICpcbiAgICogWW91IGNhbiBjdXN0b21pemUgdGhlIE9TLCBhcmNoaXRlY3R1cmUsIFZQQywgc3VibmV0LCBzZWN1cml0eSBncm91cHMsIGV0Yy4gYnkgcGFzc2luZyBpbiBwcm9wcy5cbiAgICpcbiAgICogWW91IGNhbiBhZGQgY29tcG9uZW50cyB0byB0aGUgaW1hZ2UgYnVpbGRlciBieSBjYWxsaW5nIGBpbWFnZUJ1aWxkZXIuYWRkQ29tcG9uZW50KClgLlxuICAgKlxuICAgKiBUaGUgZGVmYXVsdCBPUyBpcyBVYnVudHUgcnVubmluZyBvbiB4NjQgYXJjaGl0ZWN0dXJlLlxuICAgKlxuICAgKiBJbmNsdWRlZCBjb21wb25lbnRzOlxuICAgKiAgKiBgUnVubmVySW1hZ2VDb21wb25lbnQucmVxdWlyZWRQYWNrYWdlcygpYFxuICAgKiAgKiBgUnVubmVySW1hZ2VDb21wb25lbnQuY2xvdWRXYXRjaEFnZW50KClgXG4gICAqICAqIGBSdW5uZXJJbWFnZUNvbXBvbmVudC5ydW5uZXJVc2VyKClgXG4gICAqICAqIGBSdW5uZXJJbWFnZUNvbXBvbmVudC5naXQoKWBcbiAgICogICogYFJ1bm5lckltYWdlQ29tcG9uZW50LmdpdGh1YkNsaSgpYFxuICAgKiAgKiBgUnVubmVySW1hZ2VDb21wb25lbnQuYXdzQ2xpKClgXG4gICAqICAqIGBSdW5uZXJJbWFnZUNvbXBvbmVudC5kb2NrZXIoKWBcbiAgICogICogYFJ1bm5lckltYWdlQ29tcG9uZW50LmdpdGh1YlJ1bm5lcigpYFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpbWFnZUJ1aWxkZXIoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBSdW5uZXJJbWFnZUJ1aWxkZXJQcm9wcykge1xuICAgIHJldHVybiBSdW5uZXJJbWFnZUJ1aWxkZXIubmV3KHNjb3BlLCBpZCwge1xuICAgICAgb3M6IE9zLkxJTlVYX1VCVU5UVSxcbiAgICAgIGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlLlg4Nl82NCxcbiAgICAgIGJ1aWxkZXJUeXBlOiBSdW5uZXJJbWFnZUJ1aWxkZXJUeXBlLkFXU19JTUFHRV9CVUlMREVSLFxuICAgICAgY29tcG9uZW50czogW1xuICAgICAgICBSdW5uZXJJbWFnZUNvbXBvbmVudC5yZXF1aXJlZFBhY2thZ2VzKCksXG4gICAgICAgIFJ1bm5lckltYWdlQ29tcG9uZW50LmNsb3VkV2F0Y2hBZ2VudCgpLFxuICAgICAgICBSdW5uZXJJbWFnZUNvbXBvbmVudC5ydW5uZXJVc2VyKCksXG4gICAgICAgIFJ1bm5lckltYWdlQ29tcG9uZW50LmdpdCgpLFxuICAgICAgICBSdW5uZXJJbWFnZUNvbXBvbmVudC5naXRodWJDbGkoKSxcbiAgICAgICAgUnVubmVySW1hZ2VDb21wb25lbnQuYXdzQ2xpKCksXG4gICAgICAgIFJ1bm5lckltYWdlQ29tcG9uZW50LmRvY2tlcigpLFxuICAgICAgICBSdW5uZXJJbWFnZUNvbXBvbmVudC5naXRodWJSdW5uZXIocHJvcHM/LnJ1bm5lclZlcnNpb24gPz8gUnVubmVyVmVyc2lvbi5sYXRlc3QoKSksXG4gICAgICBdLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTGFiZWxzIGFzc29jaWF0ZWQgd2l0aCB0aGlzIHByb3ZpZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgbGFiZWxzOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogR3JhbnQgcHJpbmNpcGFsIHVzZWQgdG8gYWRkIHBlcm1pc3Npb25zIHRvIHRoZSBydW5uZXIgcm9sZS5cbiAgICovXG4gIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICAvKipcbiAgICogTG9nIGdyb3VwIHdoZXJlIHByb3ZpZGVkIHJ1bm5lcnMgd2lsbCBzYXZlIHRoZWlyIGxvZ3MuXG4gICAqXG4gICAqIE5vdGUgdGhhdCB0aGlzIGlzIG5vdCB0aGUgam9iIGxvZywgYnV0IHRoZSBydW5uZXIgaXRzZWxmLiBJdCB3aWxsIG5vdCBjb250YWluIG91dHB1dCBmcm9tIHRoZSBHaXRIdWIgQWN0aW9uIGJ1dCBvbmx5IG1ldGFkYXRhIG9uIGl0cyBleGVjdXRpb24uXG4gICAqL1xuICByZWFkb25seSBsb2dHcm91cDogbG9ncy5JTG9nR3JvdXA7XG5cbiAgcmVhZG9ubHkgcmV0cnlhYmxlRXJyb3JzID0gW1xuICAgICdFYzIuRWMyRXhjZXB0aW9uJyxcbiAgICAnU3RhdGVzLlRpbWVvdXQnLFxuICBdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgZ3JvdXA/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgYW1pQnVpbGRlcjogSVJ1bm5lckltYWdlQnVpbGRlcjtcbiAgcHJpdmF0ZSByZWFkb25seSBhbWk6IFJ1bm5lckFtaTtcbiAgcHJpdmF0ZSByZWFkb25seSByb2xlOiBpYW0uUm9sZTtcbiAgcHJpdmF0ZSByZWFkb25seSBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGU7XG4gIHByaXZhdGUgcmVhZG9ubHkgc3RvcmFnZVNpemU6IGNkay5TaXplO1xuICBwcml2YXRlIHJlYWRvbmx5IHN0b3JhZ2VPcHRpb25zPzogU3RvcmFnZU9wdGlvbnM7XG4gIHByaXZhdGUgcmVhZG9ubHkgc3BvdDogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBzcG90TWF4UHJpY2U6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgcHJpdmF0ZSByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICBwcml2YXRlIHJlYWRvbmx5IHN1Ym5ldHM6IGVjMi5JU3VibmV0W107XG4gIHByaXZhdGUgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM6IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRMYWJlbHM6IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBFYzJSdW5uZXJQcm92aWRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLmxhYmVscyA9IHByb3BzPy5sYWJlbHMgPz8gWydlYzInXTtcbiAgICB0aGlzLmdyb3VwID0gcHJvcHM/Lmdyb3VwO1xuICAgIHRoaXMudnBjID0gcHJvcHM/LnZwYyA/PyBlYzIuVnBjLmZyb21Mb29rdXAodGhpcywgJ0RlZmF1bHQgVlBDJywgeyBpc0RlZmF1bHQ6IHRydWUgfSk7XG4gICAgdGhpcy5zZWN1cml0eUdyb3VwcyA9IHByb3BzPy5zZWN1cml0eUdyb3VwID8gW3Byb3BzLnNlY3VyaXR5R3JvdXBdIDogKHByb3BzPy5zZWN1cml0eUdyb3VwcyA/PyBbbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdTRycsIHsgdnBjOiB0aGlzLnZwYyB9KV0pO1xuICAgIHRoaXMuc3VibmV0cyA9IHByb3BzPy5zdWJuZXQgPyBbcHJvcHMuc3VibmV0XSA6IHRoaXMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHM/LnN1Ym5ldFNlbGVjdGlvbikuc3VibmV0cztcbiAgICB0aGlzLmluc3RhbmNlVHlwZSA9IHByb3BzPy5pbnN0YW5jZVR5cGUgPz8gZWMyLkluc3RhbmNlVHlwZS5vZihlYzIuSW5zdGFuY2VDbGFzcy5NNkksIGVjMi5JbnN0YW5jZVNpemUuTEFSR0UpO1xuICAgIHRoaXMuc3RvcmFnZVNpemUgPSBwcm9wcz8uc3RvcmFnZVNpemUgPz8gY2RrLlNpemUuZ2liaWJ5dGVzKDMwKTsgLy8gMzAgaXMgdGhlIG1pbmltdW0gZm9yIFdpbmRvd3NcbiAgICB0aGlzLnN0b3JhZ2VPcHRpb25zID0gcHJvcHM/LnN0b3JhZ2VPcHRpb25zO1xuICAgIHRoaXMuc3BvdCA9IHByb3BzPy5zcG90ID8/IGZhbHNlO1xuICAgIHRoaXMuc3BvdE1heFByaWNlID0gcHJvcHM/LnNwb3RNYXhQcmljZTtcbiAgICB0aGlzLmRlZmF1bHRMYWJlbHMgPSBwcm9wcz8uZGVmYXVsdExhYmVscyA/PyB0cnVlO1xuXG4gICAgdGhpcy5hbWlCdWlsZGVyID0gcHJvcHM/LmltYWdlQnVpbGRlciA/PyBwcm9wcz8uYW1pQnVpbGRlciA/PyBFYzJSdW5uZXJQcm92aWRlci5pbWFnZUJ1aWxkZXIodGhpcywgJ0FtaSBCdWlsZGVyJywge1xuICAgICAgdnBjOiBwcm9wcz8udnBjLFxuICAgICAgc3VibmV0U2VsZWN0aW9uOiBwcm9wcz8uc3VibmV0U2VsZWN0aW9uLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IHRoaXMuc2VjdXJpdHlHcm91cHMsXG4gICAgfSk7XG4gICAgdGhpcy5hbWkgPSB0aGlzLmFtaUJ1aWxkZXIuYmluZEFtaSgpO1xuXG4gICAgaWYgKHRoaXMuYW1pQnVpbGRlciBpbnN0YW5jZW9mIEF3c0ltYWdlQnVpbGRlclJ1bm5lckltYWdlQnVpbGRlcikge1xuICAgICAgaWYgKHRoaXMuYW1pQnVpbGRlci5zdG9yYWdlU2l6ZSAmJiB0aGlzLnN0b3JhZ2VTaXplLnRvQnl0ZXMoKSA8IHRoaXMuYW1pQnVpbGRlci5zdG9yYWdlU2l6ZS50b0J5dGVzKCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBSdW5uZXIgc3RvcmFnZSBzaXplICgke3RoaXMuc3RvcmFnZVNpemUudG9HaWJpYnl0ZXMoKX0gR2lCKSBtdXN0IGJlIGF0IGxlYXN0IHRoZSBzYW1lIGFzIHRoZSBpbWFnZSBidWlsZGVyIHN0b3JhZ2Ugc2l6ZSAoJHt0aGlzLmFtaUJ1aWxkZXIuc3RvcmFnZVNpemUudG9HaWJpYnl0ZXMoKX0gR2lCKWApO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmICghdGhpcy5hbWkuYXJjaGl0ZWN0dXJlLmluc3RhbmNlVHlwZU1hdGNoKHRoaXMuaW5zdGFuY2VUeXBlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBTUkgYXJjaGl0ZWN0dXJlICgke3RoaXMuYW1pLmFyY2hpdGVjdHVyZS5uYW1lfSkgZG9lc24ndCBtYXRjaCBydW5uZXIgaW5zdGFuY2UgdHlwZSAoJHt0aGlzLmluc3RhbmNlVHlwZX0gLyAke3RoaXMuaW5zdGFuY2VUeXBlLmFyY2hpdGVjdHVyZX0pYCk7XG4gICAgfVxuXG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXMucm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydzdGF0ZXM6U2VuZFRhc2tGYWlsdXJlJywgJ3N0YXRlczpTZW5kVGFza1N1Y2Nlc3MnLCAnc3RhdGVzOlNlbmRUYXNrSGVhcnRiZWF0J10sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLCAvLyBubyBzdXBwb3J0IGZvciBzdGF0ZU1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuIDooXG4gICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICdhd3M6UmVzb3VyY2VUYWcvYXdzOmNsb3VkZm9ybWF0aW9uOnN0YWNrLWlkJzogY2RrLlN0YWNrLm9mKHRoaXMpLnN0YWNrSWQsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pKTtcbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsLmFkZFRvUHJpbmNpcGFsUG9saWN5KE1JTklNQUxfRUMyX1NTTV9TRVNTSU9OX01BTkFHRVJfUE9MSUNZX1NUQVRFTUVOVCk7XG5cbiAgICB0aGlzLmxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAoXG4gICAgICB0aGlzLFxuICAgICAgJ0xvZ3MnLFxuICAgICAge1xuICAgICAgICByZXRlbnRpb246IHByb3BzPy5sb2dSZXRlbnRpb24gPz8gUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEgsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0sXG4gICAgKTtcbiAgICB0aGlzLmxvZ0dyb3VwLmdyYW50V3JpdGUodGhpcyk7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgc3RlcCBmdW5jdGlvbiB0YXNrKHMpIHRvIHN0YXJ0IGEgbmV3IHJ1bm5lci5cbiAgICpcbiAgICogQ2FsbGVkIGJ5IEdpdGh1YlJ1bm5lcnMgYW5kIHNob3VsZG4ndCBiZSBjYWxsZWQgbWFudWFsbHkuXG4gICAqXG4gICAqIEBwYXJhbSBwYXJhbWV0ZXJzIHdvcmtmbG93IGpvYiBkZXRhaWxzXG4gICAqL1xuICBnZXRTdGVwRnVuY3Rpb25UYXNrKHBhcmFtZXRlcnM6IFJ1bm5lclJ1bnRpbWVQYXJhbWV0ZXJzKTogc3RlcGZ1bmN0aW9ucy5JQ2hhaW5hYmxlIHtcbiAgICAvLyB3ZSBuZWVkIHRvIGJ1aWxkIHVzZXIgZGF0YSBpbiB0d28gc3RlcHMgYmVjYXVzZSBwYXNzaW5nIHRoZSB0ZW1wbGF0ZSBhcyB0aGUgZmlyc3QgcGFyYW1ldGVyIHRvIHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguZm9ybWF0IGZhaWxzIG9uIHN5bnRheFxuXG4gICAgY29uc3QgcGFyYW1zID0gW1xuICAgICAgc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC50YXNrVG9rZW4sXG4gICAgICB0aGlzLmxvZ0dyb3VwLmxvZ0dyb3VwTmFtZSxcbiAgICAgIHBhcmFtZXRlcnMucnVubmVyTmFtZVBhdGgsXG4gICAgICBwYXJhbWV0ZXJzLmdpdGh1YkRvbWFpblBhdGgsXG4gICAgICBwYXJhbWV0ZXJzLm93bmVyUGF0aCxcbiAgICAgIHBhcmFtZXRlcnMucmVwb1BhdGgsXG4gICAgICBwYXJhbWV0ZXJzLnJ1bm5lclRva2VuUGF0aCxcbiAgICAgIHRoaXMubGFiZWxzLmpvaW4oJywnKSxcbiAgICAgIHBhcmFtZXRlcnMucmVnaXN0cmF0aW9uVXJsLFxuICAgICAgdGhpcy5ncm91cCA/ICctLXJ1bm5lcmdyb3VwJyA6ICcnLFxuICAgICAgLy8gdGhpcyBpcyBzcGxpdCBpbnRvIDIgZm9yIHBvd2Vyc2hlbGwgb3RoZXJ3aXNlIGl0IHdpbGwgcGFzcyBcIi0tcnVubmVyZ3JvdXAgbmFtZVwiIGFzIGEgc2luZ2xlIGFyZ3VtZW50IGFuZCBjb25maWcuc2ggd2lsbCBmYWlsXG4gICAgICB0aGlzLmdyb3VwID8gdGhpcy5ncm91cCA6ICcnLFxuICAgICAgdGhpcy5kZWZhdWx0TGFiZWxzID8gJycgOiAnLS1uby1kZWZhdWx0LWxhYmVscycsXG4gICAgXTtcblxuICAgIGNvbnN0IHBhc3NVc2VyRGF0YSA9IG5ldyBzdGVwZnVuY3Rpb25zLlBhc3ModGhpcywgYCR7dGhpcy5sYWJlbHMuam9pbignLCAnKX0gZGF0YWAsIHtcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgdXNlcmRhdGFUZW1wbGF0ZTogdGhpcy5hbWkub3MuaXMoT3MuV0lORE9XUykgPyB3aW5kb3dzVXNlckRhdGFUZW1wbGF0ZSA6IGxpbnV4VXNlckRhdGFUZW1wbGF0ZSxcbiAgICAgIH0sXG4gICAgICByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLmVjMicpLFxuICAgIH0pO1xuXG4gICAgLy8gd2UgdXNlIGVjMjpSdW5JbnN0YW5jZXMgYmVjYXVzZSB3ZSBtdXN0XG4gICAgLy8gd2UgY2FuJ3QgdXNlIGZsZWV0cyBiZWNhdXNlIHRoZXkgZG9uJ3QgbGV0IHVzIG92ZXJyaWRlIHVzZXIgZGF0YSwgc2VjdXJpdHkgZ3JvdXBzIG9yIGV2ZW4gZGlzayBzaXplXG4gICAgLy8gd2UgY2FuJ3QgdXNlIHJlcXVlc3RTcG90SW5zdGFuY2VzIGJlY2F1c2UgaXQgZG9lc24ndCBzdXBwb3J0IGxhdW5jaCB0ZW1wbGF0ZXMsIGFuZCBpdCdzIGRlcHJlY2F0ZWRcbiAgICAvLyBlYzI6UnVuSW5zdGFuY2VzIGFsc28gc2VlbWVkIGxpa2UgdGhlIG9ubHkgb25lIHRvIGltbWVkaWF0ZWx5IHJldHVybiBhbiBlcnJvciB3aGVuIHNwb3QgY2FwYWNpdHkgaXMgbm90IGF2YWlsYWJsZVxuXG4gICAgLy8gd2UgYnVpbGQgYSBjb21wbGljYXRlZCBjaGFpbiBvZiBzdGF0ZXMgaGVyZSBiZWNhdXNlIGVjMjpSdW5JbnN0YW5jZXMgY2FuIG9ubHkgdHJ5IG9uZSBzdWJuZXQgYXQgYSB0aW1lXG4gICAgLy8gaWYgc29tZW9uZSBjYW4gZmlndXJlIG91dCBhIGdvb2Qgd2F5IHRvIHVzZSBNYXAgZm9yIHRoaXMsIHBsZWFzZSBvcGVuIGEgUFJcblxuICAgIC8vIGJ1aWxkIGEgc3RhdGUgZm9yIGVhY2ggc3VibmV0IHdlIHdhbnQgdG8gdHJ5XG4gICAgY29uc3QgaW5zdGFuY2VQcm9maWxlID0gbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlIFByb2ZpbGUnLCB7XG4gICAgICByb2xlczogW3RoaXMucm9sZS5yb2xlTmFtZV0sXG4gICAgfSk7XG4gICAgY29uc3Qgcm9vdERldmljZVJlc291cmNlID0gYW1pUm9vdERldmljZSh0aGlzLCB0aGlzLmFtaS5sYXVuY2hUZW1wbGF0ZS5sYXVuY2hUZW1wbGF0ZUlkKTtcbiAgICByb290RGV2aWNlUmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMuYW1pQnVpbGRlcik7XG4gICAgY29uc3Qgc3VibmV0UnVubmVycyA9IHRoaXMuc3VibmV0cy5tYXAoKHN1Ym5ldCwgaW5kZXgpID0+IHtcbiAgICAgIHJldHVybiBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZSh0aGlzLCBgJHt0aGlzLmxhYmVscy5qb2luKCcsICcpfSBzdWJuZXQke2luZGV4KzF9YCwge1xuICAgICAgICBjb21tZW50OiBzdWJuZXQuc3VibmV0SWQsXG4gICAgICAgIGludGVncmF0aW9uUGF0dGVybjogSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU4sXG4gICAgICAgIHNlcnZpY2U6ICdlYzInLFxuICAgICAgICBhY3Rpb246ICdydW5JbnN0YW5jZXMnLFxuICAgICAgICBoZWFydGJlYXRUaW1lb3V0OiBzdGVwZnVuY3Rpb25zLlRpbWVvdXQuZHVyYXRpb24oRHVyYXRpb24ubWludXRlcygxMCkpLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgTGF1bmNoVGVtcGxhdGU6IHtcbiAgICAgICAgICAgIExhdW5jaFRlbXBsYXRlSWQ6IHRoaXMuYW1pLmxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlSWQsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBNaW5Db3VudDogMSxcbiAgICAgICAgICBNYXhDb3VudDogMSxcbiAgICAgICAgICBJbnN0YW5jZVR5cGU6IHRoaXMuaW5zdGFuY2VUeXBlLnRvU3RyaW5nKCksXG4gICAgICAgICAgVXNlckRhdGE6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguYmFzZTY0RW5jb2RlKFxuICAgICAgICAgICAgc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5mb3JtYXQoXG4gICAgICAgICAgICAgIHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQuZWMyLnVzZXJkYXRhVGVtcGxhdGUnKSxcbiAgICAgICAgICAgICAgLi4ucGFyYW1zLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICApLFxuICAgICAgICAgIEluc3RhbmNlSW5pdGlhdGVkU2h1dGRvd25CZWhhdmlvcjogZWMyLkluc3RhbmNlSW5pdGlhdGVkU2h1dGRvd25CZWhhdmlvci5URVJNSU5BVEUsXG4gICAgICAgICAgSWFtSW5zdGFuY2VQcm9maWxlOiB7XG4gICAgICAgICAgICBBcm46IGluc3RhbmNlUHJvZmlsZS5hdHRyQXJuLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgTWV0YWRhdGFPcHRpb25zOiB7XG4gICAgICAgICAgICBIdHRwVG9rZW5zOiAncmVxdWlyZWQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgU2VjdXJpdHlHcm91cElkczogdGhpcy5zZWN1cml0eUdyb3Vwcy5tYXAoc2cgPT4gc2cuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgICAgICBTdWJuZXRJZDogc3VibmV0LnN1Ym5ldElkLFxuICAgICAgICAgIEJsb2NrRGV2aWNlTWFwcGluZ3M6IFt7XG4gICAgICAgICAgICBEZXZpY2VOYW1lOiByb290RGV2aWNlUmVzb3VyY2UucmVmLFxuICAgICAgICAgICAgRWJzOiB7XG4gICAgICAgICAgICAgIERlbGV0ZU9uVGVybWluYXRpb246IHRydWUsXG4gICAgICAgICAgICAgIFZvbHVtZVNpemU6IHRoaXMuc3RvcmFnZVNpemUudG9HaWJpYnl0ZXMoKSxcbiAgICAgICAgICAgICAgVm9sdW1lVHlwZTogdGhpcy5zdG9yYWdlT3B0aW9ucz8udm9sdW1lVHlwZSxcbiAgICAgICAgICAgICAgSW9wczogdGhpcy5zdG9yYWdlT3B0aW9ucz8uaW9wcyxcbiAgICAgICAgICAgICAgVGhyb3VnaHB1dDogdGhpcy5zdG9yYWdlT3B0aW9ucz8udGhyb3VnaHB1dCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfV0sXG4gICAgICAgICAgSW5zdGFuY2VNYXJrZXRPcHRpb25zOiB0aGlzLnNwb3QgPyB7XG4gICAgICAgICAgICBNYXJrZXRUeXBlOiAnc3BvdCcsXG4gICAgICAgICAgICBTcG90T3B0aW9uczoge1xuICAgICAgICAgICAgICBNYXhQcmljZTogdGhpcy5zcG90TWF4UHJpY2UsXG4gICAgICAgICAgICAgIFNwb3RJbnN0YW5jZVR5cGU6ICdvbmUtdGltZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICAgICAgVGFnU3BlY2lmaWNhdGlvbnM6IFsgLy8gbWFudWFsbHkgcHJvcGFnYXRlIHRhZ3NcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVzb3VyY2VUeXBlOiAnaW5zdGFuY2UnLFxuICAgICAgICAgICAgICBUYWdzOiBbe1xuICAgICAgICAgICAgICAgIEtleTogJ0dpdEh1YlJ1bm5lcnM6UHJvdmlkZXInLFxuICAgICAgICAgICAgICAgIFZhbHVlOiB0aGlzLm5vZGUucGF0aCxcbiAgICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZXNvdXJjZVR5cGU6ICd2b2x1bWUnLFxuICAgICAgICAgICAgICBUYWdzOiBbe1xuICAgICAgICAgICAgICAgIEtleTogJ0dpdEh1YlJ1bm5lcnM6UHJvdmlkZXInLFxuICAgICAgICAgICAgICAgIFZhbHVlOiB0aGlzLm5vZGUucGF0aCxcbiAgICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIGlhbVJlc291cmNlczogWycqJ10sXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIC8vIHN0YXJ0IHdpdGggdGhlIGZpcnN0IHN1Ym5ldFxuICAgIHBhc3NVc2VyRGF0YS5uZXh0KHN1Ym5ldFJ1bm5lcnNbMF0pO1xuXG4gICAgLy8gY2hhaW4gdXAgdGhlIHJlc3Qgb2YgdGhlIHN1Ym5ldHNcbiAgICBmb3IgKGxldCBpID0gMTsgaSA8IHN1Ym5ldFJ1bm5lcnMubGVuZ3RoOyBpKyspIHtcbiAgICAgIHN1Ym5ldFJ1bm5lcnNbaS0xXS5hZGRDYXRjaChzdWJuZXRSdW5uZXJzW2ldLCB7XG4gICAgICAgIGVycm9yczogWydFYzIuRWMyRXhjZXB0aW9uJywgJ1N0YXRlcy5UaW1lb3V0J10sXG4gICAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQubGFzdFN1Ym5ldEVycm9yJyksXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gcGFzc1VzZXJEYXRhO1xuICB9XG5cbiAgZ3JhbnRTdGF0ZU1hY2hpbmUoc3RhdGVNYWNoaW5lUm9sZTogaWFtLklHcmFudGFibGUpIHtcbiAgICBzdGF0ZU1hY2hpbmVSb2xlLmdyYW50UHJpbmNpcGFsLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10sXG4gICAgICByZXNvdXJjZXM6IFt0aGlzLnJvbGUucm9sZUFybl0sXG4gICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICdpYW06UGFzc2VkVG9TZXJ2aWNlJzogJ2VjMi5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgc3RhdGVNYWNoaW5lUm9sZS5ncmFudFByaW5jaXBhbC5hZGRUb1ByaW5jaXBhbFBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ2VjMjpjcmVhdGVUYWdzJ10sXG4gICAgICByZXNvdXJjZXM6IFtTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnZWMyJyxcbiAgICAgICAgcmVzb3VyY2U6ICcqJyxcbiAgICAgIH0pXSxcbiAgICB9KSk7XG5cbiAgICBzdGF0ZU1hY2hpbmVSb2xlLmdyYW50UHJpbmNpcGFsLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlJ10sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAnaWFtOkFXU1NlcnZpY2VOYW1lJzogJ3Nwb3QuYW1hem9uYXdzLmNvbScsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pKTtcbiAgfVxuXG4gIHN0YXR1cyhzdGF0dXNGdW5jdGlvblJvbGU6IGlhbS5JR3JhbnRhYmxlKTogSVJ1bm5lclByb3ZpZGVyU3RhdHVzIHtcbiAgICBzdGF0dXNGdW5jdGlvblJvbGUuZ3JhbnRQcmluY2lwYWwuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydlYzI6RGVzY3JpYmVMYXVuY2hUZW1wbGF0ZVZlcnNpb25zJ10sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgIH0pKTtcblxuICAgIHJldHVybiB7XG4gICAgICB0eXBlOiB0aGlzLmNvbnN0cnVjdG9yLm5hbWUsXG4gICAgICBsYWJlbHM6IHRoaXMubGFiZWxzLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IHRoaXMuc2VjdXJpdHlHcm91cHMubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICByb2xlQXJuOiB0aGlzLnJvbGUucm9sZUFybixcbiAgICAgIGxvZ0dyb3VwOiB0aGlzLmxvZ0dyb3VwLmxvZ0dyb3VwTmFtZSxcbiAgICAgIGFtaToge1xuICAgICAgICBsYXVuY2hUZW1wbGF0ZTogdGhpcy5hbWkubGF1bmNoVGVtcGxhdGUubGF1bmNoVGVtcGxhdGVJZCB8fCAndW5rbm93bicsXG4gICAgICAgIGFtaUJ1aWxkZXJMb2dHcm91cDogdGhpcy5hbWkubG9nR3JvdXA/LmxvZ0dyb3VwTmFtZSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbmV0d29yayBjb25uZWN0aW9ucyBhc3NvY2lhdGVkIHdpdGggdGhpcyByZXNvdXJjZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgY29ubmVjdGlvbnMoKTogZWMyLkNvbm5lY3Rpb25zIHtcbiAgICByZXR1cm4gbmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzOiB0aGlzLnNlY3VyaXR5R3JvdXBzIH0pO1xuICB9XG59XG5cbi8qKlxuICogQGRlcHJlY2F0ZWQgdXNlIHtAbGluayBFYzJSdW5uZXJQcm92aWRlcn1cbiAqL1xuZXhwb3J0IGNsYXNzIEVjMlJ1bm5lciBleHRlbmRzIEVjMlJ1bm5lclByb3ZpZGVyIHtcbn1cbiJdfQ==