"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseProvider = exports.Os = exports.Architecture = exports.RunnerVersion = void 0;
exports.amiRootDevice = amiRootDevice;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const ami_root_device_function_1 = require("./ami-root-device-function");
const utils_1 = require("../utils");
/**
 * Defines desired GitHub Actions runner version.
 */
class RunnerVersion {
    /**
     * Use the latest version available at the time the runner provider image is built.
     */
    static latest() {
        return new RunnerVersion('latest');
    }
    /**
     * Use a specific version.
     *
     * @see https://github.com/actions/runner/releases
     *
     * @param version GitHub Runner version
     */
    static specific(version) {
        return new RunnerVersion(version);
    }
    constructor(version) {
        this.version = version;
    }
    /**
     * Check if two versions are the same.
     *
     * @param other version to compare
     */
    is(other) {
        return this.version == other.version;
    }
}
exports.RunnerVersion = RunnerVersion;
_a = JSII_RTTI_SYMBOL_1;
RunnerVersion[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerVersion", version: "0.14.15" };
/**
 * CPU architecture enum for an image.
 */
class Architecture {
    static of(architecture) {
        return new Architecture(architecture);
    }
    constructor(name) {
        this.name = name;
    }
    /**
    * Checks if the given architecture is the same as this one.
    *
    * @param arch architecture to compare
    */
    is(arch) {
        return arch.name == this.name;
    }
    /**
     * Checks if this architecture is in a given list.
     *
     * @param arches architectures to check
     */
    isIn(arches) {
        for (const arch of arches) {
            if (this.is(arch)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Checks if a given EC2 instance type matches this architecture.
     *
     * @param instanceType instance type to check
     */
    instanceTypeMatch(instanceType) {
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.X86_64) {
            return this.is(Architecture.X86_64);
        }
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.ARM_64) {
            return this.is(Architecture.ARM64);
        }
        throw new Error('Unknown instance type architecture');
    }
}
exports.Architecture = Architecture;
_b = JSII_RTTI_SYMBOL_1;
Architecture[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Architecture", version: "0.14.15" };
/**
 * ARM64
 */
Architecture.ARM64 = Architecture.of('ARM64');
/**
 * X86_64
 */
Architecture.X86_64 = Architecture.of('X86_64');
/**
 * OS enum for an image.
 */
class Os {
    static of(os) {
        return new Os(os);
    }
    constructor(name) {
        this.name = name;
    }
    /**
    * Checks if the given OS is the same as this one.
    *
    * @param os OS to compare
    */
    is(os) {
        return os.name == this.name;
    }
    /**
     * Checks if this OS is in a given list.
     *
     * @param oses list of OS to check
     */
    isIn(oses) {
        for (const os of oses) {
            if (this.is(os)) {
                return true;
            }
        }
        return false;
    }
}
exports.Os = Os;
_c = JSII_RTTI_SYMBOL_1;
Os[_c] = { fqn: "@cloudsnorkel/cdk-github-runners.Os", version: "0.14.15" };
/**
* Linux
*
* @deprecated use {@link LINUX_UBUNTU}, {@link LINUX_UBUNTU_2404}, {@link LINUX_AMAZON_2} or {@link LINUX_AMAZON_2023}
*/
Os.LINUX = Os.of('Linux');
/**
 * Ubuntu Linux
 */
Os.LINUX_UBUNTU = Os.of('Ubuntu Linux');
/**
* Ubuntu Linux 22.04
*/
Os.LINUX_UBUNTU_2204 = Os.of('Ubuntu Linux 22.04');
/**
 * Ubuntu Linux 24.04
 */
Os.LINUX_UBUNTU_2404 = Os.of('Ubuntu Linux 24.04');
/**
 * Amazon Linux 2
 */
Os.LINUX_AMAZON_2 = Os.of('Amazon Linux 2');
/**
 * Amazon Linux 2023
 */
Os.LINUX_AMAZON_2023 = Os.of('Amazon Linux 2023');
/**
 * @internal
 */
Os._ALL_LINUX_VERSIONS = [Os.LINUX, Os.LINUX_UBUNTU, Os.LINUX_UBUNTU_2204, Os.LINUX_UBUNTU_2404, Os.LINUX_AMAZON_2, Os.LINUX_AMAZON_2023];
/**
   * @internal
   */
Os._ALL_LINUX_AMAZON_VERSIONS = [Os.LINUX_AMAZON_2, Os.LINUX_AMAZON_2023];
/**
   * @internal
   */
Os._ALL_LINUX_UBUNTU_VERSIONS = [Os.LINUX_UBUNTU, Os.LINUX_UBUNTU_2204, Os.LINUX_UBUNTU_2404];
/**
* Windows
*/
Os.WINDOWS = Os.of('Windows');
/**
 * Base class for all providers with common methods used by all providers.
 *
 * @internal
 */
class BaseProvider extends constructs_1.Construct {
    constructor(scope, id, _props) {
        super(scope, id);
        cdk.Tags.of(this).add('GitHubRunners:Provider', this.node.path);
    }
    labelsFromProperties(defaultLabel, propsLabel, propsLabels) {
        if (propsLabels && propsLabel) {
            throw new Error('Must supply either `label` or `labels` in runner properties, but not both. Try removing the `label` property.');
        }
        if (propsLabels) {
            return propsLabels;
        }
        if (propsLabel) {
            return [propsLabel];
        }
        return [defaultLabel];
    }
}
exports.BaseProvider = BaseProvider;
/**
 * Use custom resource to determine the root device name of a given AMI, Launch Template, or SSM parameter pointing to AMI.
 *
 * TODO move somewhere more common as it's used by both providers and AMI builder now
 *
 * @internal
 */
function amiRootDevice(scope, ami) {
    const crHandler = (0, utils_1.singletonLambda)(ami_root_device_function_1.AmiRootDeviceFunction, scope, 'AMI Root Device Reader', {
        description: 'Custom resource handler that discovers the boot drive device name for a given AMI',
        timeout: cdk.Duration.minutes(1),
        logGroup: (0, utils_1.singletonLogGroup)(scope, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD),
        loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
        initialPolicy: [
            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: [
                    'ssm:GetParameter',
                    'ec2:DescribeImages',
                    'ec2:DescribeLaunchTemplateVersions',
                    'imagebuilder:GetImage',
                ],
                resources: ['*'],
            }),
        ],
    });
    return new aws_cdk_lib_1.CustomResource(scope, 'AMI Root Device', {
        serviceToken: crHandler.functionArn,
        resourceType: 'Custom::AmiRootDevice',
        properties: {
            Ami: ami ?? '',
        },
    });
}
//# sourceMappingURL=data:application/json;base64,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