"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticRunnerImage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild_1 = require("./codebuild");
const providers_1 = require("../providers");
/**
 * Helper class with methods to use static images that are built outside the context of this project.
 */
class StaticRunnerImage {
    /**
     * Create a builder (that doesn't actually build anything) from an existing image in an existing repository. The image must already have GitHub Actions runner installed. You are responsible to update it and remove it when done.
     *
     * @param repository ECR repository
     * @param tag image tag
     * @param architecture image architecture
     * @param os image OS
     */
    static fromEcrRepository(repository, tag = 'latest', architecture = providers_1.Architecture.X86_64, os = providers_1.Os.LINUX) {
        return {
            bindDockerImage() {
                return {
                    imageRepository: repository,
                    imageTag: tag,
                    architecture,
                    os,
                    runnerVersion: providers_1.RunnerVersion.latest(),
                    _dependable: repository.repositoryArn,
                };
            },
            bindAmi() {
                throw new Error('fromEcrRepository() cannot be used to build AMIs');
            },
        };
    }
    /**
     * Create a builder from an existing Docker Hub image. The image must already have GitHub Actions runner installed. You are responsible to update it and remove it when done.
     *
     * We create a CodeBuild image builder behind the scenes to copy the image over to ECR. This helps avoid Docker Hub rate limits and prevent failures.
     *
     * @param scope
     * @param id
     * @param image Docker Hub image with optional tag
     * @param architecture image architecture
     * @param os image OS
     */
    static fromDockerHub(scope, id, image, architecture = providers_1.Architecture.X86_64, os = providers_1.Os.LINUX) {
        return new codebuild_1.CodeBuildRunnerImageBuilder(scope, id, {
            os,
            architecture,
            baseDockerImage: image,
        });
    }
}
exports.StaticRunnerImage = StaticRunnerImage;
_a = JSII_RTTI_SYMBOL_1;
StaticRunnerImage[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.StaticRunnerImage", version: "0.14.15" };
//# sourceMappingURL=data:application/json;base64,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