"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunnerImageComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_image_builder_1 = require("./aws-image-builder");
const providers_1 = require("../providers");
/**
 * Components are used to build runner images. They can run commands in the image, copy files into the image, and run some Docker commands.
 */
class RunnerImageComponent {
    /**
     * Define a custom component that can run commands in the image, copy files into the image, and run some Docker commands.
     *
     * The order of operations is (1) assets (2) commands (3) docker commands.
     *
     * Use this to customize the image for the runner.
     *
     * **WARNING:** Docker commands are not guaranteed to be included before the next component
     */
    static custom(props) {
        return new class extends RunnerImageComponent {
            get name() {
                if (props.name && !props.name.match(/[a-zA-Z0-9\-]/)) {
                    throw new Error(`Invalid component name: ${props.name}. Name must only contain alphanumeric characters and dashes.`);
                }
                return `Custom-${props.name ?? 'Undefined'}`;
            }
            getCommands(_os, _architecture) {
                return props.commands ?? [];
            }
            getAssets(_os, _architecture) {
                return props.assets ?? [];
            }
            getDockerCommands(_os, _architecture) {
                return props.dockerCommands ?? [];
            }
        }();
    }
    /**
     * A component to install the required packages for the runner.
     */
    static requiredPackages() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RequiredPackages';
            }
            getCommands(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get upgrade -y',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y curl sudo jq bash zip unzip iptables software-properties-common ca-certificates',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum update -y',
                        'yum install -y jq tar gzip bzip2 which binutils zip unzip sudo shadow-utils',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf upgrade -y',
                        'dnf install -y jq tar gzip bzip2 which binutils zip unzip sudo shadow-utils findutils',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [];
                }
                throw new Error(`Unsupported OS for required packages: ${os.name}`);
            }
        };
    }
    /**
     * A component to install CloudWatch Agent for the runner so we can send logs.
     */
    static cloudWatchAgent() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'CloudWatchAgent';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    let archUrl;
                    if (architecture.is(providers_1.Architecture.X86_64)) {
                        archUrl = 'amd64';
                    }
                    else if (architecture.is(providers_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for required packages: ${architecture.name}`);
                    }
                    return [
                        `curl -sfLo /tmp/amazon-cloudwatch-agent.deb https://s3.amazonaws.com/amazoncloudwatch-agent/ubuntu/${archUrl}/latest/amazon-cloudwatch-agent.deb`,
                        'dpkg -i -E /tmp/amazon-cloudwatch-agent.deb',
                        'rm /tmp/amazon-cloudwatch-agent.deb',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y amazon-cloudwatch-agent',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf install -y amazon-cloudwatch-agent',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i https://s3.amazonaws.com/amazoncloudwatch-agent/windows/amd64/latest/amazon-cloudwatch-agent.msi /qn\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                    ];
                }
                throw new Error(`Unsupported OS for required packages: ${os.name}`);
            }
        };
    }
    /**
     * A component to prepare the required runner user.
     */
    static runnerUser() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RunnerUser';
            }
            getCommands(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'addgroup runner',
                        'adduser --system --disabled-password --home /home/runner --ingroup runner runner',
                        'echo "%runner   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    return [
                        '/usr/sbin/groupadd runner',
                        '/usr/sbin/useradd --system --shell /usr/sbin/nologin --home-dir /home/runner --gid runner runner',
                        'mkdir -p /home/runner',
                        'chown runner /home/runner',
                        'echo "%runner   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [];
                }
                throw new Error(`Unsupported OS for runner user: ${os.name}`);
            }
        };
    }
    /**
     * A component to install the AWS CLI.
     */
    static awsCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'AwsCli';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS) || os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    let archUrl;
                    if (architecture.is(providers_1.Architecture.X86_64)) {
                        archUrl = 'x86_64';
                    }
                    else if (architecture.is(providers_1.Architecture.ARM64)) {
                        archUrl = 'aarch64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for awscli: ${architecture.name}`);
                    }
                    return [
                        `curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-${archUrl}.zip" -o awscliv2.zip`,
                        'unzip -q awscliv2.zip',
                        './aws/install',
                        'rm -rf awscliv2.zip aws',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for awscli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static githubCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubCli';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'curl -fsSL https://cli.github.com/packages/githubcli-archive-keyring.gpg | sudo dd of=/usr/share/keyrings/githubcli-archive-keyring.gpg',
                        'echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/githubcli-archive-keyring.gpg] ' +
                            '  https://cli.github.com/packages stable main" | sudo tee /etc/apt/sources.list.d/github-cli.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y gh',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'curl -fsSSL https://cli.github.com/packages/rpm/gh-cli.repo -o /etc/yum.repos.d/gh-cli.repo',
                        'yum install -y gh',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'curl -fsSSL https://cli.github.com/packages/rpm/gh-cli.repo -o /etc/yum.repos.d/gh-cli.repo',
                        'dnf install -y gh',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                        '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                        '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i gh.msi /qn\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                        'del gh.msi',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for github cli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static git() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Git';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'add-apt-repository ppa:git-core/ppa',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y git',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y git',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf install -y git',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                        '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                        '$GIT_REVISION = ($GIT_VERSION -Split \'.windows.\')[1]',
                        'If ($GIT_REVISION -gt 1) {$GIT_VERSION_SHORT = "$GIT_VERSION_SHORT.$GIT_REVISION"}',
                        'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                        '$p = Start-Process git-setup.exe -PassThru -Wait -ArgumentList \'/VERYSILENT\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                        'del git-setup.exe',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for git: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub Actions Runner. This is the actual executable that connects to GitHub to ask for jobs and then execute them.
     *
     * @param runnerVersion The version of the runner to install. Usually you would set this to latest.
     */
    static githubRunner(runnerVersion) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubRunner';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS) || os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    let versionCommand;
                    if (runnerVersion.is(providers_1.RunnerVersion.latest())) {
                        versionCommand = 'RUNNER_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest | grep -oE "[^/v]+$"`';
                    }
                    else {
                        versionCommand = `RUNNER_VERSION='${runnerVersion.version}'`;
                    }
                    let archUrl;
                    if (architecture.is(providers_1.Architecture.X86_64)) {
                        archUrl = 'x64';
                    }
                    else if (architecture.is(providers_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for GitHub Runner: ${architecture.name}`);
                    }
                    let commands = [
                        versionCommand,
                        `curl -fsSLO "https://github.com/actions/runner/releases/download/v\${RUNNER_VERSION}/actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `tar -C /home/runner -xzf "actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `rm actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz`,
                        `echo -n ${runnerVersion.version} > /home/runner/RUNNER_VERSION`,
                    ];
                    if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                        commands.push('/home/runner/bin/installdependencies.sh');
                    }
                    else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                        commands.push('yum install -y openssl-libs krb5-libs zlib libicu60');
                    }
                    else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                        commands.push('dnf install -y openssl-libs krb5-libs zlib libicu-67.1');
                    }
                    commands.push('mkdir -p /opt/hostedtoolcache', 'chown runner /opt/hostedtoolcache');
                    return commands;
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    let runnerCommands;
                    if (runnerVersion.is(providers_1.RunnerVersion.latest())) {
                        runnerCommands = [
                            'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                            '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                            '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        ];
                    }
                    else {
                        runnerCommands = [`$RUNNER_VERSION = '${runnerVersion.version}'`];
                    }
                    runnerCommands = runnerCommands.concat([
                        // create directories
                        'mkdir C:\\hostedtoolcache\\windows',
                        'mkdir C:\\tools',
                        // download zstd and extract to C:\tools
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/facebook/zstd/releases/latest > $Env:TEMP\\latest-zstd',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-zstd',
                        '$ZSTD_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/facebook/zstd/releases/download/v$ZSTD_VERSION/zstd-v$ZSTD_VERSION-win64.zip" -OutFile zstd.zip',
                        'Expand-Archive zstd.zip -DestinationPath C:\\tools',
                        'Move-Item -Path C:\\tools\\zstd-v$ZSTD_VERSION-win64\\zstd.exe C:\\tools',
                        'Remove-Item -LiteralPath "C:\\tools\\zstd-v$ZSTD_VERSION-win64" -Force -Recurse',
                        'del zstd.zip',
                        // add C:\tools to PATH
                        '$persistedPaths = [Environment]::GetEnvironmentVariable(\'Path\', [EnvironmentVariableTarget]::Machine)',
                        '[Environment]::SetEnvironmentVariable("PATH", $persistedPaths + ";C:\\tools", [EnvironmentVariableTarget]::Machine)',
                    ]);
                    return runnerCommands.concat([
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                        'Expand-Archive actions.zip -DestinationPath C:\\actions',
                        'del actions.zip',
                        `echo ${runnerVersion.version} | Out-File -Encoding ASCII -NoNewline C:\\actions\\RUNNER_VERSION`,
                    ]);
                }
                throw new Error(`Unknown os/architecture combo for github runner: ${os.name}/${architecture.name}`);
            }
            getDockerCommands(_os, _architecture) {
                return [
                    `ENV RUNNER_VERSION=${runnerVersion.version}`,
                ];
            }
        }();
    }
    /**
     * A component to install Docker.
     *
     * On Windows this sets up dockerd for Windows containers without Docker Desktop. If you need Linux containers on Windows, you'll need to install Docker Desktop which doesn't seem to play well with servers (PRs welcome).
     */
    static docker() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Docker';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker.gpg',
                        'echo ' +
                            '  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu ' +
                            '  $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin',
                        'usermod -aG docker runner',
                        'ln -s /usr/libexec/docker/cli-plugins/docker-compose /usr/bin/docker-compose',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'amazon-linux-extras install docker',
                        'usermod -a -G docker runner',
                        'curl -sfLo /usr/bin/docker-compose https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s | tr \'[:upper:]\' \'[:lower:]\')-$(uname -m)',
                        'chmod +x /usr/bin/docker-compose',
                        'ln -s /usr/bin/docker-compose /usr/libexec/docker/cli-plugins/docker-compose',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf install -y docker',
                        'usermod -a -G docker runner',
                        'curl -sfLo /usr/bin/docker-compose https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s | tr \'[:upper:]\' \'[:lower:]\')-$(uname -m)',
                        'chmod +x /usr/bin/docker-compose',
                        'ln -s /usr/bin/docker-compose /usr/libexec/docker/cli-plugins/docker-compose',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        // figure out latest docker version
                        '$BaseUrl = "https://download.docker.com/win/static/stable/x86_64/"',
                        '$html = Invoke-WebRequest -UseBasicParsing -Uri $BaseUrl',
                        '$files = $html.Links.href | Where-Object { $_ -match \'^docker-[0-9\\.]+\\.zip$\' }',
                        'if (-not $files) { Write-Error "No docker-*.zip files found." ; exit 1 }',
                        '$latest = $files | Sort-Object { try { [Version]($_ -replace \'^docker-|\\.zip$\') } catch { [Version]"0.0.0" } } -Descending | Select-Object -First 1',
                        // download static binaries
                        'Invoke-WebRequest -UseBasicParsing -Uri $BaseUrl$latest -OutFile docker.zip',
                        // extract to C:\Program Files\Docker
                        'Expand-Archive docker.zip -DestinationPath "$Env:ProgramFiles"',
                        'del docker.zip',
                        // add to path
                        '$persistedPaths = [Environment]::GetEnvironmentVariable(\'Path\', [EnvironmentVariableTarget]::Machine)',
                        '[Environment]::SetEnvironmentVariable("PATH", $persistedPaths + ";$Env:ProgramFiles\\Docker", [EnvironmentVariableTarget]::Machine)',
                        '$env:PATH = $env:PATH + ";$Env:ProgramFiles\\Docker"',
                        // register docker service
                        'dockerd --register-service',
                        'if ($LASTEXITCODE -ne 0) { throw "Exit code is $LASTEXITCODE" }',
                        // enable containers feature
                        'Enable-WindowsOptionalFeature -Online -FeatureName containers -All -NoRestart',
                        // install docker-compose
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/docker/compose/releases/latest > $Env:TEMP\\latest-docker-compose',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-docker-compose',
                        '$LatestDockerCompose = ($LatestUrl -Split \'/\')[-1]',
                        'Invoke-WebRequest -UseBasicParsing -Uri  "https://github.com/docker/compose/releases/download/${LatestDockerCompose}/docker-compose-Windows-x86_64.exe" -OutFile $Env:ProgramFiles\\Docker\\docker-compose.exe',
                        'New-Item -ItemType directory -Path "$Env:ProgramFiles\\Docker\\cli-plugins"',
                        'Copy-Item -Path "$Env:ProgramFiles\\Docker\\docker-compose.exe" -Destination "$Env:ProgramFiles\\Docker\\cli-plugins\\docker-compose.exe"',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for docker: ${os.name}/${architecture.name}`);
            }
            shouldReboot(os, _architecture) {
                return os.is(providers_1.Os.WINDOWS);
            }
        }();
    }
    /**
     * A component to install Docker-in-Docker.
     *
     * @deprecated use `docker()`
     */
    static dockerInDocker() {
        return RunnerImageComponent.docker();
    }
    /**
     * A component to add a trusted certificate authority. This can be used to support GitHub Enterprise Server with self-signed certificate.
     *
     * @param source path to certificate file in PEM format
     * @param name unique certificate name to be used on runner file system
     */
    static extraCertificates(source, name) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = `Extra-Certificates-${name}`;
            }
            getCommands(os, architecture) {
                if (!name.match(/^[a-zA-Z0-9_-]+$/)) {
                    throw new Error(`Invalid certificate name: ${name}. Name must only contain alphanumeric characters, dashes and underscores.`);
                }
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'update-ca-certificates',
                    ];
                }
                else if (os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    return [
                        'update-ca-trust',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        `Import-Certificate -FilePath C:\\${name}.crt -CertStoreLocation Cert:\\LocalMachine\\Root`,
                        `Remove-Item C:\\${name}.crt`,
                    ];
                }
                throw new Error(`Unknown os/architecture combo for extra certificates: ${os.name}/${architecture.name}`);
            }
            getAssets(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        { source, target: `/usr/local/share/ca-certificates/${name}.crt` },
                    ];
                }
                else if (os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    return [
                        { source, target: `/etc/pki/ca-trust/source/anchors/${name}.crt` },
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        { source, target: `C:\\${name}.crt` },
                    ];
                }
                throw new Error(`Unsupported OS for extra certificates: ${os.name}`);
            }
        }();
    }
    /**
     * A component to set up the required Lambda entrypoint for Lambda runners.
     */
    static lambdaEntrypoint() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Lambda-Entrypoint';
            }
            getCommands(os, _architecture) {
                if (!os.isIn(providers_1.Os._ALL_LINUX_VERSIONS)) {
                    throw new Error(`Unsupported OS for Lambda entrypoint: ${os.name}`);
                }
                return [];
            }
            getAssets(_os, _architecture) {
                return [
                    {
                        source: path.join(__dirname, '..', '..', 'assets', 'providers', 'lambda-bootstrap.sh'),
                        target: '/bootstrap.sh',
                    },
                    {
                        source: path.join(__dirname, '..', '..', 'assets', 'providers', 'lambda-runner.sh'),
                        target: '/runner.sh',
                    },
                ];
            }
            getDockerCommands(_os, _architecture) {
                return [
                    'LABEL DISABLE_SOCI=1', // hacky way to disable soci v2 indexing on lambda as lambda will fail to start with an index
                    'ENTRYPOINT ["bash", "/bootstrap.sh"]',
                ];
            }
        };
    }
    /**
     * A component to add environment variables for jobs the runner executes.
     *
     * These variables only affect the jobs ran by the runner. They are not global. They do not affect other components.
     *
     * It is not recommended to use this component to pass secrets. Instead, use GitHub Secrets or AWS Secrets Manager.
     *
     * Must be used after the {@link githubRunner} component.
     */
    static environmentVariables(vars) {
        Object.entries(vars).forEach(e => {
            if (e[0].includes('\n') || e[1].includes('\n')) {
                throw new Error(`Environment variable cannot contain newlines: ${e}`);
            }
        });
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'EnvironmentVariables';
            }
            getCommands(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_VERSIONS)) {
                    return Object.entries(vars).map(e => `echo '${e[0]}=${e[1].replace(/'/g, "'\"'\"'")}' >> /home/runner/.env`);
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return Object.entries(vars).map(e => `Add-Content -Path C:\\actions\\.env -Value '${e[0]}=${e[1].replace(/'/g, "''")}'`);
                }
                else {
                    throw new Error(`Unsupported OS for environment variables component: ${os.name}`);
                }
            }
        };
    }
    /**
     * Returns assets to copy into the built image. Can be used to copy files into the image.
     */
    getAssets(_os, _architecture) {
        return [];
    }
    /**
     * Returns Docker commands to run to in built image. Can be used to add commands like `VOLUME`, `ENTRYPOINT`, `CMD`, etc.
     *
     * Docker commands are added after assets and normal commands.
     */
    getDockerCommands(_os, _architecture) {
        return [];
    }
    /**
     * Returns true if the image builder should be rebooted after this component is installed.
     */
    shouldReboot(_os, _architecture) {
        return false;
    }
    /**
     * Convert component to an AWS Image Builder component.
     *
     * @internal
     */
    _asAwsImageBuilderComponent(scope, id, os, architecture) {
        let platform;
        if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS) || os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
            platform = 'Linux';
        }
        else if (os.is(providers_1.Os.WINDOWS)) {
            platform = 'Windows';
        }
        else {
            throw new Error(`Unknown os/architecture combo for image builder component: ${os.name}/${architecture.name}`);
        }
        return new aws_image_builder_1.ImageBuilderComponent(scope, id, {
            platform: platform,
            commands: this.getCommands(os, architecture),
            assets: this.getAssets(os, architecture).map((asset, index) => {
                return {
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(scope, `${id} asset ${index}`, { path: asset.source }),
                    path: asset.target,
                };
            }),
            displayName: id,
            description: id,
            reboot: this.shouldReboot(os, architecture),
        });
    }
}
exports.RunnerImageComponent = RunnerImageComponent;
_a = JSII_RTTI_SYMBOL_1;
RunnerImageComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerImageComponent", version: "0.14.15" };
//# sourceMappingURL=data:application/json;base64,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