"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildImageBuilderFailedBuildNotifier = exports.CodeBuildRunnerImageBuilder = void 0;
const crypto = require("node:crypto");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_image_builder_1 = require("./aws-image-builder");
const build_image_function_1 = require("./build-image-function");
const common_1 = require("./common");
const providers_1 = require("../providers");
const utils_1 = require("../utils");
/**
 * @internal
 */
class CodeBuildRunnerImageBuilder extends common_1.RunnerImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props?.awsImageBuilderOptions) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('awsImageBuilderOptions are ignored when using CodeBuild runner image builder.');
        }
        this.os = props?.os ?? providers_1.Os.LINUX_UBUNTU;
        this.architecture = props?.architecture ?? providers_1.Architecture.X86_64;
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        this.logRetention = props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        this.vpc = props?.vpc;
        this.securityGroups = props?.securityGroups;
        this.subnetSelection = props?.subnetSelection;
        this.timeout = props?.codeBuildOptions?.timeout ?? aws_cdk_lib_1.Duration.hours(1);
        this.computeType = props?.codeBuildOptions?.computeType ?? aws_codebuild_1.ComputeType.SMALL;
        this.baseImage = props?.baseDockerImage ?? (0, aws_image_builder_1.defaultBaseDockerImage)(this.os);
        this.buildImage = props?.codeBuildOptions?.buildImage ?? this.getDefaultBuildImage();
        this.waitOnDeploy = props?.waitOnDeploy ?? true;
        this.dockerSetupCommands = props?.dockerSetupCommands ?? [];
        // warn against isolated networks
        if (props?.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // error out on no-nat networks because the build will hang
        if (props?.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC) {
            aws_cdk_lib_1.Annotations.of(this).addError('Public subnets do not work with CodeBuild as it cannot be assigned an IP. ' +
                'See https://docs.aws.amazon.com/codebuild/latest/userguide/vpc-support.html#best-practices-for-vpcs');
        }
        // check timeout
        if (this.timeout.toSeconds() > aws_cdk_lib_1.Duration.hours(8).toSeconds()) {
            aws_cdk_lib_1.Annotations.of(this).addError('CodeBuild runner image builder timeout must 8 hours or less.');
        }
        // create service role for CodeBuild
        this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            emptyOnDelete: true,
            lifecycleRules: [
                {
                    description: 'Remove soci indexes for replaced images',
                    tagStatus: aws_ecr_1.TagStatus.TAGGED,
                    tagPrefixList: ['sha256-'],
                    maxImageCount: 1,
                },
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
    }
    bindAmi() {
        throw new Error('CodeBuild image builder cannot be used to build AMI');
    }
    bindDockerImage() {
        if (this.boundDockerImage) {
            return this.boundDockerImage;
        }
        // log group for the image builds
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: this.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: this.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // generate buildSpec
        const [buildSpec, buildSpecHash] = this.getBuildSpec(this.repository);
        // create CodeBuild project that builds Dockerfile and pushes to repository
        const project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `Build docker image for self-hosted GitHub runner ${this.node.path} (${this.os.name}/${this.architecture.name})`,
            buildSpec,
            vpc: this.vpc,
            securityGroups: this.securityGroups,
            subnetSelection: this.subnetSelection,
            role: this.role,
            timeout: this.timeout,
            environment: {
                buildImage: this.buildImage,
                computeType: this.computeType,
                privileged: true,
            },
            logging: {
                cloudWatch: {
                    logGroup,
                },
            },
        });
        // permissions
        this.repository.grantPullPush(project);
        // call CodeBuild during deployment
        const completedImage = this.customResource(project, buildSpecHash);
        // rebuild image on a schedule
        this.rebuildImageOnSchedule(project, this.rebuildInterval);
        // return the image
        this.boundDockerImage = {
            imageRepository: this.repository,
            imageTag: 'latest',
            architecture: this.architecture,
            os: this.os,
            logGroup,
            runnerVersion: providers_1.RunnerVersion.specific('unknown'),
            _dependable: completedImage,
        };
        return this.boundDockerImage;
    }
    getDefaultBuildImage() {
        if (this.os.isIn(providers_1.Os._ALL_LINUX_VERSIONS)) {
            // CodeBuild just runs `docker build` so its OS doesn't really matter
            if (this.architecture.is(providers_1.Architecture.X86_64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.AMAZON_LINUX_2_5;
            }
            else if (this.architecture.is(providers_1.Architecture.ARM64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_3_0;
            }
        }
        if (this.os.is(providers_1.Os.WINDOWS)) {
            throw new Error('CodeBuild cannot be used to build Windows Docker images https://github.com/docker-library/docker/issues/49');
        }
        throw new Error(`Unable to find CodeBuild image for ${this.os.name}/${this.architecture.name}`);
    }
    getDockerfileGenerationCommands() {
        let hashedComponents = [];
        let commands = [];
        let dockerfile = `FROM ${this.baseImage}\nVOLUME /var/lib/docker\n`;
        for (let i = 0; i < this.components.length; i++) {
            const componentName = this.components[i].name;
            const assetDescriptors = this.components[i].getAssets(this.os, this.architecture);
            for (let j = 0; j < assetDescriptors.length; j++) {
                if (this.os.is(providers_1.Os.WINDOWS)) {
                    throw new Error("Can't add asset as we can't build Windows Docker images on CodeBuild");
                }
                const asset = new aws_cdk_lib_1.aws_s3_assets.Asset(this, `Component ${i} ${componentName} Asset ${j}`, {
                    path: assetDescriptors[j].source,
                });
                if (asset.isFile) {
                    commands.push(`aws s3 cp ${asset.s3ObjectUrl} asset${i}-${componentName}-${j}`);
                }
                else if (asset.isZipArchive) {
                    commands.push(`aws s3 cp ${asset.s3ObjectUrl} asset${i}-${componentName}-${j}.zip`);
                    commands.push(`unzip asset${i}-${componentName}-${j}.zip -d "asset${i}-${componentName}-${j}"`);
                }
                else {
                    throw new Error(`Unknown asset type: ${asset}`);
                }
                dockerfile += `COPY asset${i}-${componentName}-${j} ${assetDescriptors[j].target}\n`;
                hashedComponents.push(`__ ASSET FILE ${asset.assetHash} ${i}-${componentName}-${j} ${assetDescriptors[j].target}`);
                asset.grantRead(this);
            }
            const componentCommands = this.components[i].getCommands(this.os, this.architecture);
            const script = '#!/bin/bash\nset -exuo pipefail\n' + componentCommands.join('\n');
            commands.push(`cat > component${i}-${componentName}.sh <<'EOFGITHUBRUNNERSDOCKERFILE'\n${script}\nEOFGITHUBRUNNERSDOCKERFILE`);
            commands.push(`chmod +x component${i}-${componentName}.sh`);
            hashedComponents.push(`__ COMMAND ${i} ${componentName} ${script}`);
            dockerfile += `COPY component${i}-${componentName}.sh /tmp\n`;
            dockerfile += `RUN /tmp/component${i}-${componentName}.sh\n`;
            const dockerCommands = this.components[i].getDockerCommands(this.os, this.architecture);
            dockerfile += dockerCommands.join('\n') + '\n';
            hashedComponents.push(`__ DOCKER COMMAND ${i} ${dockerCommands.join('\n')}`);
        }
        commands.push(`cat > Dockerfile <<'EOFGITHUBRUNNERSDOCKERFILE'\n${dockerfile}\nEOFGITHUBRUNNERSDOCKERFILE`);
        return [commands, hashedComponents];
    }
    getBuildSpec(repository) {
        const thisStack = cdk.Stack.of(this);
        let archUrl;
        if (this.architecture.is(providers_1.Architecture.X86_64)) {
            archUrl = 'x86_64';
        }
        else if (this.architecture.is(providers_1.Architecture.ARM64)) {
            archUrl = 'arm64';
        }
        else {
            throw new Error(`Unsupported architecture for required CodeBuild: ${this.architecture.name}`);
        }
        const [commands, commandsHashedComponents] = this.getDockerfileGenerationCommands();
        const buildSpecVersion = 'v2'; // change this every time the build spec changes
        const hashedComponents = commandsHashedComponents.concat(buildSpecVersion, this.architecture.name, this.baseImage, this.os.name);
        const hash = crypto.createHash('md5').update(hashedComponents.join('\n')).digest('hex').slice(0, 10);
        const buildSpec = aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
            version: 0.2,
            env: {
                variables: {
                    REPO_ARN: repository.repositoryArn,
                    REPO_URI: repository.repositoryUri,
                    WAIT_HANDLE: 'unspecified',
                    BASH_ENV: 'codebuild-log.sh',
                },
                shell: 'bash',
            },
            phases: {
                pre_build: {
                    commands: [
                        'echo "exec > >(tee -a /tmp/codebuild.log) 2>&1" > codebuild-log.sh',
                        `aws ecr get-login-password --region "$AWS_DEFAULT_REGION" | docker login --username AWS --password-stdin ${thisStack.account}.dkr.ecr.${thisStack.region}.amazonaws.com`,
                    ].concat(this.dockerSetupCommands),
                },
                build: {
                    commands: commands.concat('docker build --progress plain . -t "$REPO_URI"', 'docker push "$REPO_URI"'),
                },
                post_build: {
                    commands: [
                        'rm -f codebuild-log.sh && STATUS="SUCCESS"',
                        'if [ $CODEBUILD_BUILD_SUCCEEDING -ne 1 ]; then STATUS="FAILURE"; fi',
                        'cat <<EOF > /tmp/payload.json\n' +
                            '{\n' +
                            '  "Status": "$STATUS",\n' +
                            '  "UniqueId": "build",\n' +
                            // we remove non-printable characters from the log because CloudFormation doesn't like them
                            // https://github.com/aws-cloudformation/cloudformation-coverage-roadmap/issues/1601
                            '  "Reason": `sed \'s/[^[:print:]]//g\' /tmp/codebuild.log | tail -c 400 | jq -Rsa .`,\n' +
                            // for lambda always get a new value because there is always a new image hash
                            '  "Data": "$RANDOM"\n' +
                            '}\n' +
                            'EOF',
                        'if [ "$WAIT_HANDLE" != "unspecified" ]; then jq . /tmp/payload.json; curl -fsSL -X PUT -H "Content-Type:" -d "@/tmp/payload.json" "$WAIT_HANDLE"; fi',
                        // generate and push soci index
                        // we do this after finishing the build, so we don't have to wait. it's also not required, so it's ok if it fails
                        'if [ `docker inspect --format=\'{{json .Config.Labels.DISABLE_SOCI}}\' "$REPO_URI"` = "null" ]; then\n' +
                            'docker rmi "$REPO_URI"\n' + // it downloads the image again to /tmp, so save on space
                            'LATEST_SOCI_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/CloudSnorkel/standalone-soci-indexer/releases/latest | grep -oE "[^/]+$"`\n' +
                            `curl -fsSL https://github.com/CloudSnorkel/standalone-soci-indexer/releases/download/$\{LATEST_SOCI_VERSION}/standalone-soci-indexer_Linux_${archUrl}.tar.gz | tar xz\n` +
                            './standalone-soci-indexer "$REPO_URI"\n' +
                            'fi',
                    ],
                },
            },
        });
        return [buildSpec, hash];
    }
    customResource(project, buildSpecHash) {
        const crHandler = (0, utils_1.singletonLambda)(build_image_function_1.BuildImageFunction, this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images',
            timeout: cdk.Duration.minutes(3),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['codebuild:StartBuild'],
                    resources: [project.projectArn],
                }),
            ],
        });
        crHandler.role.attachInlinePolicy(policy);
        let waitHandleRef = 'unspecified';
        let waitDependable = '';
        if (this.waitOnDeploy) {
            // Wait handle lets us wait for longer than an hour for the image build to complete.
            // We generate a new wait handle for build spec changes to guarantee a new image is built.
            // This also helps make sure the changes are good. If they have a bug, the deployment will fail instead of just the scheduled build.
            // Finally, it's recommended by CloudFormation docs to not reuse wait handles or old responses may interfere in some cases.
            const handle = new aws_cdk_lib_1.aws_cloudformation.CfnWaitConditionHandle(this, `Build Wait Handle ${buildSpecHash}`);
            const wait = new aws_cdk_lib_1.aws_cloudformation.CfnWaitCondition(this, `Build Wait ${buildSpecHash}`, {
                handle: handle.ref,
                timeout: this.timeout.toSeconds().toString(), // don't wait longer than the build timeout
                count: 1,
            });
            waitHandleRef = handle.ref;
            waitDependable = wait.ref;
        }
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Builder', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageBuilder',
            properties: {
                RepoName: this.repository.repositoryName,
                ProjectName: project.projectName,
                WaitHandle: waitHandleRef,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(project);
        cr.node.addDependency(this.role);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler.role);
        cr.node.addDependency(crHandler);
        return waitDependable; // user needs to wait on wait handle which is triggered when the image is built
    }
    rebuildImageOnSchedule(project, rebuildInterval) {
        rebuildInterval = rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (rebuildInterval.toMilliseconds() != 0) {
            const scheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Build Schedule', {
                description: `Rebuild runner image for ${this.repository.repositoryName}`,
                schedule: aws_cdk_lib_1.aws_events.Schedule.rate(rebuildInterval),
            });
            scheduleRule.addTarget(new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project));
        }
    }
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: this.securityGroups,
        });
    }
    get grantPrincipal() {
        return this.role;
    }
}
exports.CodeBuildRunnerImageBuilder = CodeBuildRunnerImageBuilder;
/**
 * @internal
 */
class CodeBuildImageBuilderFailedBuildNotifier {
    constructor(topic) {
        this.topic = topic;
    }
    visit(node) {
        if (node instanceof CodeBuildRunnerImageBuilder) {
            const builder = node;
            const projectNode = builder.node.tryFindChild('CodeBuild');
            if (projectNode) {
                const project = projectNode;
                project.notifyOnBuildFailed('BuildFailed', this.topic);
            }
            else {
                cdk.Annotations.of(builder).addWarning('Unused builder cannot get notifications of failed builds');
            }
        }
    }
}
exports.CodeBuildImageBuilderFailedBuildNotifier = CodeBuildImageBuilderFailedBuildNotifier;
//# sourceMappingURL=data:application/json;base64,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