"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workflow = void 0;
exports.generateBuildWorkflowWithDockerSetupCommands = generateBuildWorkflowWithDockerSetupCommands;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const common_2 = require("../common");
/**
 * Image builder workflow.
 *
 * @internal
 */
class Workflow extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.name = (0, common_2.uniqueImageBuilderName)(this);
        this.version = this.generateVersion('Workflow', this.name, {
            type: props.type,
            data: props.data,
        });
        const workflow = new aws_cdk_lib_1.aws_imagebuilder.CfnWorkflow(this, 'Workflow', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            version: this.version,
            type: props.type,
            data: JSON.stringify(props.data),
        });
        this.arn = workflow.attrArn;
    }
}
exports.Workflow = Workflow;
/**
 * Returns a new build workflow based on arn:aws:imagebuilder:us-east-1:aws:workflow/build/build-container/1.0.1/1.
 *
 * It adds a DockerSetup step after bootstrapping but before the Docker image is built.
 *
 * @internal
 */
function generateBuildWorkflowWithDockerSetupCommands(scope, id, dockerSetupCommands) {
    return new Workflow(scope, id, {
        type: 'BUILD',
        data: {
            name: 'build-container',
            description: 'Workflow to build a container image',
            schemaVersion: 1,
            steps: [
                {
                    name: 'LaunchBuildInstance',
                    action: 'LaunchInstance',
                    onFailure: 'Abort',
                    inputs: {
                        waitFor: 'ssmAgent',
                    },
                },
                {
                    name: 'BootstrapBuildInstance',
                    action: 'BootstrapInstanceForContainer',
                    onFailure: 'Abort',
                    if: {
                        stringEquals: 'DOCKER',
                        value: '$.imagebuilder.imageType',
                    },
                    inputs: {
                        'instanceId.$': '$.stepOutputs.LaunchBuildInstance.instanceId',
                    },
                },
                {
                    // this is the part we add
                    name: 'DockerSetup',
                    action: 'RunCommand',
                    onFailure: 'Abort',
                    if: {
                        stringEquals: 'DOCKER',
                        value: '$.imagebuilder.imageType',
                    },
                    inputs: {
                        'documentName': 'AWS-RunShellScript',
                        'parameters': {
                            commands: dockerSetupCommands,
                        },
                        'instanceId.$': '$.stepOutputs.LaunchBuildInstance.instanceId',
                    },
                },
                {
                    name: 'ApplyBuildComponents',
                    action: 'ExecuteComponents',
                    onFailure: 'Abort',
                    inputs: {
                        'instanceId.$': '$.stepOutputs.LaunchBuildInstance.instanceId',
                    },
                },
            ],
            outputs: [
                {
                    name: 'InstanceId',
                    value: '$.stepOutputs.LaunchBuildInstance.instanceId',
                },
            ],
        },
    });
}
//# sourceMappingURL=data:application/json;base64,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