"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.increaseVersion = increaseVersion;
exports.handler = handler;
const client_imagebuilder_1 = require("@aws-sdk/client-imagebuilder");
const semver_1 = require("semver");
const lambda_helpers_1 = require("../../lambda-helpers");
const ib = new client_imagebuilder_1.ImagebuilderClient();
/**
 * Exported for unit tests.
 * @internal
 */
function increaseVersion(allVersions) {
    let version = (0, semver_1.maxSatisfying)(allVersions, '>=0.0.0');
    if (version === null) {
        version = '1.0.0';
    }
    console.log(`Found versions ${allVersions} -- latest is ${version}`);
    version = (0, semver_1.inc)(version, 'patch');
    if (version === null) {
        throw new Error('Unable to bump version');
    }
    return version;
}
async function handler(event, context) {
    console.log({ ...event, ResponseURL: '...' });
    try {
        const objectType = event.ResourceProperties.ObjectType;
        const objectName = event.ResourceProperties.ObjectName;
        switch (event.RequestType) {
            case 'Create':
            case 'Update':
                let allVersions = [];
                try {
                    switch (objectType) {
                        case 'Component': {
                            let result = {};
                            do {
                                result = await ib.send(new client_imagebuilder_1.ListComponentsCommand({
                                    filters: [{
                                            name: 'name',
                                            values: [objectName],
                                        }],
                                    nextToken: result.nextToken,
                                }));
                                allVersions = allVersions.concat(result.componentVersionList.map(i => i.version || '1.0.0'));
                            } while (result.nextToken);
                            break;
                        }
                        case 'ImageRecipe': {
                            let result = {};
                            do {
                                result = await ib.send(new client_imagebuilder_1.ListImageRecipesCommand({
                                    filters: [{
                                            name: 'name',
                                            values: [objectName],
                                        }],
                                    nextToken: result.nextToken,
                                }));
                                allVersions = allVersions.concat(result.imageRecipeSummaryList.map(i => i.arn?.split('/').pop() || '1.0.0'));
                            } while (result.nextToken);
                            break;
                        }
                        case 'ContainerRecipe': {
                            let result = {};
                            do {
                                result = await ib.send(new client_imagebuilder_1.ListContainerRecipesCommand({
                                    filters: [{
                                            name: 'name',
                                            values: [objectName],
                                        }],
                                    nextToken: result.nextToken,
                                }));
                                allVersions = allVersions.concat(result.containerRecipeSummaryList.map(i => i.arn?.split('/').pop() || '1.0.0'));
                            } while (result.nextToken);
                            break;
                        }
                        case 'Workflow': {
                            let result = {};
                            do {
                                result = await ib.send(new client_imagebuilder_1.ListWorkflowsCommand({
                                    filters: [{
                                            name: 'name',
                                            values: [objectName],
                                        }],
                                    nextToken: result.nextToken,
                                }));
                                allVersions = allVersions.concat(result.workflowVersionList.map(i => i.arn?.split('/').pop() || '1.0.0'));
                            } while (result.nextToken);
                            break;
                        }
                    }
                }
                catch (e) {
                    if (e.code !== 'ResourceNotFoundException') {
                        throw e;
                    }
                    else {
                        console.log('Resource not found, assuming first version');
                    }
                }
                const version = increaseVersion(allVersions);
                await (0, lambda_helpers_1.customResourceRespond)(event, 'SUCCESS', 'OK', version, {});
                break;
            case 'Delete':
                await (0, lambda_helpers_1.customResourceRespond)(event, 'SUCCESS', 'OK', event.PhysicalResourceId, {});
                break;
        }
    }
    catch (e) {
        console.log(e);
        await (0, lambda_helpers_1.customResourceRespond)(event, 'FAILED', e.message || 'Internal Error', context.logStreamName, {});
    }
}
//# sourceMappingURL=data:application/json;base64,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