"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerRecipe = void 0;
exports.defaultBaseDockerImage = defaultBaseDockerImage;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const providers_1 = require("../../providers");
const common_2 = require("../common");
/**
 * Image builder recipe for a Docker container image.
 *
 * @internal
 */
class ContainerRecipe extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        this.name = (0, common_2.uniqueImageBuilderName)(this);
        this.version = this.generateVersion('ContainerRecipe', this.name, {
            platform: props.platform,
            components,
            dockerfileTemplate: props.dockerfileTemplate,
            tags: props.tags,
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: this.name,
            version: this.version,
            parentImage: props.parentImage,
            platformOverride: props.platform == 'Linux' ? 'Linux' : undefined,
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: props.dockerfileTemplate,
            tags: props.tags,
        });
        this.arn = recipe.attrArn;
    }
}
exports.ContainerRecipe = ContainerRecipe;
/**
 * Default base Docker image for given OS.
 *
 * @internal
 */
function defaultBaseDockerImage(os) {
    if (os.is(providers_1.Os.WINDOWS)) {
        return 'mcr.microsoft.com/windows/servercore:ltsc2019-amd64';
    }
    else if (os.is(providers_1.Os.LINUX_UBUNTU) || os.is(providers_1.Os.LINUX_UBUNTU_2204)) {
        return 'public.ecr.aws/lts/ubuntu:22.04';
    }
    else if (os.is(providers_1.Os.LINUX_UBUNTU_2404)) {
        return 'public.ecr.aws/lts/ubuntu:24.04';
    }
    else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
        return 'public.ecr.aws/amazonlinux/amazonlinux:2';
    }
    else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
        return 'public.ecr.aws/amazonlinux/amazonlinux:2023';
    }
    else {
        throw new Error(`OS ${os.name} not supported for Docker runner image`);
    }
}
//# sourceMappingURL=data:application/json;base64,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