import * as cdk from 'aws-cdk-lib';
import { aws_ec2 as ec2, aws_iam as iam, aws_imagebuilder as imagebuilder, aws_logs as logs, aws_s3_assets as s3_assets, aws_sns as sns } from 'aws-cdk-lib';
import { Construct, IConstruct } from 'constructs';
import { ImageBuilderObjectBase } from './common';
import { RunnerAmi, RunnerImage } from '../../providers';
import { RunnerImageBuilderBase, RunnerImageBuilderProps } from '../common';
export interface AwsImageBuilderRunnerImageBuilderProps {
    /**
     * The instance type used to build the image.
     *
     * @default m6i.large
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * Size of volume available for builder instances. This modifies the boot volume size and doesn't add any additional volumes.
     *
     * Use this if you're building images with big components and need more space.
     *
     * @default default size for AMI (usually 30GB for Linux and 50GB for Windows)
     */
    readonly storageSize?: cdk.Size;
    /**
     * Options for fast launch.
     *
     * This is only supported for Windows AMIs.
     *
     * @default disabled
     */
    readonly fastLaunchOptions?: FastLaunchOptions;
}
/**
 * Options for fast launch.
 */
export interface FastLaunchOptions {
    /**
     * Enable fast launch for AMIs generated by this builder. It creates a snapshot of the root volume and uses it to launch new instances faster.
     *
     * This is only supported for Windows AMIs.
     *
     * @note this feature comes with additional resource costs. See the documentation for more details. https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/win-fast-launch-manage-costs.html
     * @note enabling fast launch on an existing builder will not enable it for existing AMIs. It will only affect new AMIs. If you want immediate effect, trigger a new image build. Alternatively, you can create a new builder with fast launch enabled and use it for new AMIs.
     *
     * @default false
     */
    readonly enabled?: boolean;
    /**
     * The maximum number of parallel instances that are launched for creating resources.
     *
     * Must be at least 6.
     *
     * @default 6
     */
    readonly maxParallelLaunches?: number;
    /**
     * The number of pre-provisioned snapshots to keep on hand for a fast-launch enabled Windows AMI.
     *
     * @default 1
     */
    readonly targetResourceCount?: number;
}
/**
 * An asset including file or directory to place inside the built image.
 */
export interface ImageBuilderAsset {
    /**
     * Path to place asset in the image.
     */
    readonly path: string;
    /**
     * Asset to place in the image.
     */
    readonly asset: s3_assets.Asset;
}
/**
 * Properties for ImageBuilderComponent construct.
 */
export interface ImageBuilderComponentProperties {
    /**
     * Component platform. Must match the builder platform.
     */
    readonly platform: 'Linux' | 'Windows';
    /**
     * Component display name.
     */
    readonly displayName: string;
    /**
     * Component description.
     */
    readonly description: string;
    /**
     * Shell commands to run when adding this component to the image.
     *
     * On Linux, these are bash commands. On Windows, there are PowerShell commands.
     */
    readonly commands: string[];
    /**
     * Optional assets to add to the built image.
     */
    readonly assets?: ImageBuilderAsset[];
    /**
     * Require a reboot after installing this component.
     *
     * @default false
     */
    readonly reboot?: boolean;
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *     'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
 *   ],
 * }
 * ```
 *
 * @deprecated Use `RunnerImageComponent` instead as this be internal soon.
 */
export declare class ImageBuilderComponent extends ImageBuilderObjectBase {
    /**
     * Component ARN.
     */
    readonly arn: string;
    /**
     * Supported platform for the component.
     */
    readonly platform: 'Windows' | 'Linux';
    private readonly assets;
    constructor(scope: Construct, id: string, props: ImageBuilderComponentProperties);
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee: iam.IGrantable): void;
    prefixCommandsWithErrorHandling(platform: 'Windows' | 'Linux', commands: string[]): string[];
}
/**
 * @internal
 */
export declare class AwsImageBuilderRunnerImageBuilder extends RunnerImageBuilderBase {
    private boundDockerImage?;
    private boundAmi?;
    private readonly os;
    private readonly architecture;
    private readonly baseImage;
    private readonly baseAmi;
    private readonly logRetention;
    private readonly logRemovalPolicy;
    private readonly vpc;
    private readonly securityGroups;
    private readonly subnetSelection;
    private readonly rebuildInterval;
    private readonly boundComponents;
    private readonly instanceType;
    private infrastructure;
    private readonly role;
    private readonly fastLaunchOptions?;
    readonly storageSize?: cdk.Size;
    private readonly waitOnDeploy;
    private readonly dockerSetupCommands;
    private readonly tags;
    constructor(scope: Construct, id: string, props?: RunnerImageBuilderProps);
    private platform;
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bindDockerImage(): RunnerImage;
    private dockerImageCleaner;
    protected createLog(id: string, recipeName: string): logs.LogGroup;
    protected createInfrastructure(managedPolicies: iam.IManagedPolicy[]): imagebuilder.CfnInfrastructureConfiguration;
    protected createImage(infra: imagebuilder.CfnInfrastructureConfiguration, dist: imagebuilder.CfnDistributionConfiguration, log: logs.LogGroup, imageRecipeArn?: string, containerRecipeArn?: string): imagebuilder.CfnImage;
    private amiOrContainerId;
    protected createPipeline(infra: imagebuilder.CfnInfrastructureConfiguration, dist: imagebuilder.CfnDistributionConfiguration, log: logs.LogGroup, imageRecipeArn?: string, containerRecipeArn?: string): imagebuilder.CfnImagePipeline;
    /**
     * The network connections associated with this resource.
     */
    get connections(): ec2.Connections;
    get grantPrincipal(): iam.IPrincipal;
    bindAmi(): RunnerAmi;
    private amiCleaner;
    private bindComponents;
    private imageCleaner;
}
/**
 * @internal
 */
export declare class AwsImageBuilderFailedBuildNotifier implements cdk.IAspect {
    private topic;
    static createFilteringTopic(scope: Construct, targetTopic: sns.Topic): sns.ITopic;
    constructor(topic: sns.ITopic);
    visit(node: IConstruct): void;
}
