"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiRecipe = void 0;
exports.defaultBaseAmi = defaultBaseAmi;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const providers_1 = require("../../providers");
const common_2 = require("../common");
/**
 * Image builder recipe for Amazon Machine Image (AMI).
 *
 * @internal
 */
class AmiRecipe extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const blockDeviceMappings = props.storageSize ? [
            {
                deviceName: (0, providers_1.amiRootDevice)(this, props.baseAmi).ref,
                ebs: {
                    volumeSize: props.storageSize.toGibibytes(),
                    deleteOnTermination: true,
                },
            },
        ] : undefined;
        this.name = (0, common_2.uniqueImageBuilderName)(this);
        this.version = this.generateVersion('ImageRecipe', this.name, {
            platform: props.platform,
            components,
            parentAmi: props.baseAmi,
            tags: props.tags,
            blockDeviceMappings,
        });
        let workingDirectory;
        if (props.platform == 'Linux') {
            workingDirectory = '/home/runner';
        }
        else if (props.platform == 'Windows') {
            workingDirectory = 'C:/'; // must exist or Image Builder fails and must not be empty or git will stall installing from the default windows\system32
        }
        else {
            throw new Error(`Unsupported AMI recipe platform: ${props.platform}`);
        }
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'Recipe', {
            name: this.name,
            version: this.version,
            parentImage: props.baseAmi,
            components,
            workingDirectory,
            tags: props.tags,
            blockDeviceMappings,
        });
        this.arn = recipe.attrArn;
    }
}
exports.AmiRecipe = AmiRecipe;
/**
 * Default base AMI for given OS and architecture.
 *
 * @internal
 */
function defaultBaseAmi(scope, os, architecture) {
    const stack = cdk.Stack.of(scope);
    let arch;
    if (architecture.is(providers_1.Architecture.X86_64)) {
        arch = 'x86';
    }
    else if (architecture.is(providers_1.Architecture.ARM64)) {
        arch = 'arm64';
    }
    else {
        throw new Error(`Unsupported architecture for base AMI: ${architecture.name}`);
    }
    if (os.is(providers_1.Os.LINUX_UBUNTU) || os.is(providers_1.Os.LINUX_UBUNTU_2204) || os.is(providers_1.Os.LINUX)) {
        return stack.formatArn({
            service: 'imagebuilder',
            resource: 'image',
            account: 'aws',
            resourceName: `ubuntu-server-22-lts-${arch}/x.x.x`,
        });
    }
    if (os.is(providers_1.Os.LINUX_UBUNTU_2404)) {
        return stack.formatArn({
            service: 'imagebuilder',
            resource: 'image',
            account: 'aws',
            resourceName: `ubuntu-server-24-lts-${arch}/x.x.x`,
        });
    }
    if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
        return stack.formatArn({
            service: 'imagebuilder',
            resource: 'image',
            account: 'aws',
            resourceName: `amazon-linux-2-${arch}/x.x.x`,
        });
    }
    if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
        return stack.formatArn({
            service: 'imagebuilder',
            resource: 'image',
            account: 'aws',
            resourceName: `amazon-linux-2023-${arch}/x.x.x`,
        });
    }
    if (os.is(providers_1.Os.WINDOWS)) {
        return stack.formatArn({
            service: 'imagebuilder',
            resource: 'image',
            account: 'aws',
            resourceName: `windows-server-2022-english-full-base-${arch}/x.x.x`,
        });
    }
    throw new Error(`OS ${os.name} not supported for AMI runner image`);
}
//# sourceMappingURL=data:application/json;base64,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