# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nitter_scraper']

package_data = \
{'': ['*'], 'nitter_scraper': ['templates/*']}

install_requires = \
['docker>=4.3.1,<5.0.0',
 'jinja2>=2.11.2,<3.0.0',
 'loguru>=0.5.1,<0.6.0',
 'pendulum>=2.1.2,<3.0.0',
 'pydantic>=1.6.1,<2.0.0',
 'requests-html>=0.10.0,<0.11.0']

setup_kwargs = {
    'name': 'nitter-scraper',
    'version': '0.3.3',
    'description': 'Scrape Twitter API without authentication using Nitter.',
    'long_description': '# Nitter Scraper\n\nThis library is a work around for anyone who enjoyed the simplicity of the [twitter-scraper](https://github.com/bisguzar/twitter-scraper/) library and needs a quick replacement until it comes back up.  Nitter Scraper leverages running a docker instance of [nitter](https://github.com/zedeus/nitter) to scrape tweets and profile information. I attempted to make the api work as closely as possible to the original to minimize refactoring of other projects.\n\n## Use Nitter Scraper with docker\nIf you have docker installed you can use the NitterDockerContainer manager. It takes care of starting/stoping a local nitter docker container and provides the address to scrape from. If you don\'t use the NitterDockerContainer the get_tweets and get_profile methods will resolve to scraping from https://nitter.net\n\n```\nfrom nitter_scraper import NitterDockerContainer, get_tweets\n\nwith NitterDockerContainer(host="0.0.0.0", port=8008) as nitter:\n    for tweet in get_tweets("dgnsrekt", address=nitter.address):\n        print(tweet.json(indent=4))\n\n```\n\n## Prerequisites\n\n* Docker\n* Python ^3.7\n\n## Run the examples.\n```\ngit clone git@github.com:dgnsrekt/nitter_scraper.git\ncd nitter_scraper\npoetry install\npoetry shell\n```\nAdd twitter usernames to the example.txt delimited by newlines.\n\nDocker must be properly installed to run the examples with the NitterDockerContainer context manager.\n\nRun profile scraping example\n```\npython3 example_profile.py\n```\nRun tweet scraping example\n```\npython3 example_tweet.py\n```\n## Profile & Tweet objects\nThe Profile and Tweet classes inherit from pydantic\'s BaseModel. This makes it easy to convert to a dict or json.\n```\nprofile = get_profile("dgnsrekt")\n\nprofile.json(indent=4)\n\nprofile.dict()\n```\n\n### Profile Schema\nThese are the fields uses in the Profile class which is returned from the get_profile() method.\n```\n{\n    "title": "Profile",\n    "type": "object",\n    "properties": {\n        "username": {\n            "title": "Username",\n            "type": "string"\n        },\n        "name": {\n            "title": "Name",\n            "type": "string"\n        },\n        "profile_photo": {\n            "title": "Profile Photo",\n            "type": "string"\n        },\n        "tweets_count": {\n            "title": "Tweets Count",\n            "type": "integer"\n        },\n        "following_count": {\n            "title": "Following Count",\n            "type": "integer"\n        },\n        "followers_count": {\n            "title": "Followers Count",\n            "type": "integer"\n        },\n        "likes_count": {\n            "title": "Likes Count",\n            "type": "integer"\n        },\n        "is_verified": {\n            "title": "Is Verified",\n            "type": "boolean"\n        },\n        "banner_photo": {\n            "title": "Banner Photo",\n            "type": "string"\n        },\n        "biography": {\n            "title": "Biography",\n            "type": "string"\n        },\n        "user_id": {\n            "title": "User Id",\n            "type": "integer"\n        },\n        "location": {\n            "title": "Location",\n            "type": "string"\n        },\n        "website": {\n            "title": "Website",\n            "type": "string"\n        }\n    },\n    "required": [\n        "username",\n        "name",\n        "profile_photo",\n        "tweets_count",\n        "following_count",\n        "followers_count",\n        "likes_count",\n        "is_verified"\n    ]\n}\n\n\n```\n\n### Tweet Schema\nThese are the fields uses in the Tweet class which is returned from the get_tweets() method.\n\n```\n{\n    "title": "Tweet",\n    "type": "object",\n    "properties": {\n        "tweet_id": {\n            "title": "Tweet Id",\n            "type": "integer"\n        },\n        "tweet_url": {\n            "title": "Tweet Url",\n            "type": "string"\n        },\n        "username": {\n            "title": "Username",\n            "type": "string"\n        },\n        "is_retweet": {\n            "title": "Is Retweet",\n            "type": "boolean"\n        },\n        "is_pinned": {\n            "title": "Is Pinned",\n            "type": "boolean"\n        },\n        "time": {\n            "title": "Time",\n            "type": "string",\n            "format": "date-time"\n        },\n        "text": {\n            "title": "Text",\n            "type": "string"\n        },\n        "replies": {\n            "title": "Replies",\n            "type": "integer"\n        },\n        "retweets": {\n            "title": "Retweets",\n            "type": "integer"\n        },\n        "likes": {\n            "title": "Likes",\n            "type": "integer"\n        },\n        "entries": {\n            "$ref": "#/definitions/Entries"\n        }\n    },\n    "required": [\n        "tweet_id",\n        "tweet_url",\n        "username",\n        "is_retweet",\n        "is_pinned",\n        "time",\n        "text",\n        "replies",\n        "retweets",\n        "likes",\n        "entries"\n    ],\n    "definitions": {\n        "Entries": {\n            "title": "Entries",\n            "type": "object",\n            "properties": {\n                "hashtags": {\n                    "title": "Hashtags",\n                    "type": "array",\n                    "items": {\n                        "type": "string"\n                    }\n                },\n                "cashtags": {\n                    "title": "Cashtags",\n                    "type": "array",\n                    "items": {\n                        "type": "string"\n                    }\n                },\n                "urls": {\n                    "title": "Urls",\n                    "type": "array",\n                    "items": {\n                        "type": "string"\n                    }\n                },\n                "photos": {\n                    "title": "Photos",\n                    "type": "array",\n                    "items": {\n                        "type": "string"\n                    }\n                },\n                "videos": {\n                    "title": "Videos",\n                    "type": "array",\n                    "items": {\n                        "type": "string"\n                    }\n                }\n            },\n            "required": [\n                "hashtags",\n                "cashtags",\n                "urls",\n                "photos",\n                "videos"\n            ]\n        }\n    }\n}\n\n```\n\n\n## Limitation\n\n* So far i\'ve been able to pull about 800 tweets per user without issue.\n* Unable to implement a way to scrape trends.\n* birthday and is_private are not implemented in the profile.\n* If the user does not have a banner the user_id cannot be scraped.\n* The user_id cannot be scraped from tweets.\n\nMore docs coming soon.\n\n## TODO\n* setup.py\n* pypi\n\n## Contact Information\nTelegram = Twitter = Tradingview = Discord = @dgnsrekt\n\nEmail = dgnsrekt@pm.me\n',
    'author': 'dgnsrekt',
    'author_email': 'dgnsrekt@pm.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dgnsrekt/nitter_scraper',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
