def ler_POF(path, header=True, format='csv'):
    '''
    Realiza a leitura dos microdados da POF 2017/2018 diretamente do arquivo .zip baixado do site do IBGE
    e exporta os dados como um arquivo .csv.
    :param path: caminho para o arquivo .zip
    :param header: boolean, Default True - acrescenta o código da variável como nome de cada coluna.
    https://github.com/otavio-s-s/lerMicrodados
     '''
     
    import pandas as pd
    from zipfile import ZipFile

    assert format in ['xlsx', 'csv'], "Formato precisa ser xlsx ou csv"

    pof_2017 = {"ALUGUEL_ESTIMADO": [[2, 4, 1, 9, 2, 1, 2, 7, 2, 10, 2, 2, 12, 10, 1, 2, 14, 14, 10],
                                      ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                       'QUADRO', 'V9001', 'V9002', 'V8000', 'V9010', 'V9011', 'DEFLATOR', 'V8000_DEFLA',
                                       'COD_IMPUT_VALOR', 'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "CADERNETA_COLETIVA": [[2, 4, 1, 9, 2, 1, 2, 3, 7, 2, 10, 12, 10, 1, 2, 14, 14, 10, 9, 4, 5, 9],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                         'QUADRO', 'SEQ', 'V9001', 'V9002', 'V8000', 'DEFLATOR', 'V8000_DEFLA',
                                         'COD_IMPUT_VALOR', 'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL',
                                         'V9005', 'V9007', 'V9009', 'QTD_FINAL']],

                  "CARACTERISTICAS_DIETA": [[2, 4, 1, 9, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 14, 15, 10] ,
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE ', 'V7101', 'V7102', 'V71031', 'V71032', 'V71033', 'V71034', 'V71035', 'V71036', 'V71037', 'V71038', 'V7104', 'V71051', 'V71052', 'V71053', 'V71054', 'V71055', 'V71056', 'V71A01', 'V71A02', 'V72C01', 'V72C02', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],

                  "CONDICOES_VIDA": [[2, 4, 1, 9, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 14, 14, 10],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE', 'V61041', 'V61042', 'V61043', 'V61044', 'V61045', 'V61046', 'V6108', 'V6109', 'V6110', 'V6111', 'V6112', 'V6113', 'V6114', 'V6115', 'V6116', 'V6117', 'V6118', 'V6119', 'V6120', 'V6121', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],

                  "CONSUMO_ALIMENTAR": [[2, 4, 1, 9, 2, 1, 2, 2, 2, 4, 2, 7, 3, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 7, 9, 6, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 15, 10, 15, 1],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE ', 'QUADRO', 'SEQ', 'V9005', 'V9007', 'V9001', 'V9015', 'V9016', 'V9017', 'V9018', 'V9019', 'V9020', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V9029', 'V9030', 'COD_UNIDADE_MEDIDA_FINAL', 'COD_PREPARACAO_FINAL', 'GRAMATURA1', 'QTD', 'COD_TBCA', 'ENERGIA_KCAL ', 'ENERGIA_KJ', 'PTN', 'CHOTOT', 'FIBRA', 'LIP', 'COLEST', 'AGSAT', 'AGMONO', 'AGPOLI', 'AGTRANS', 'CALCIO', 'FERRO', 'SODIO', 'MAGNESIO', 'FOSFORO', 'POTASSIO', 'COBRE', 'ZINCO', 'VITA_RAE', 'TIAMINA', 'RIBOFLAVINA', 'NIACINA', 'PIRIDOXAMINA', 'COBALAMINA', 'VITD', 'VITE', 'VITC', 'FOLATO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL', 'DIA_SEMANA', 'DIA_ATIPICO']],

                 "DESPESA_COLETIVA": [
                     [2, 4, 1, 9, 2, 1, 2, 2, 7, 2, 4, 10, 2, 2, 1, 10, 1, 12, 10, 10, 1, 1, 2, 14, 14, 10],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'QUADRO', 'SEQ',
                      'V9001', 'V9002', 'V9005', 'V8000', 'V9010', 'V9011', 'V9012', 'V1904', 'V1905', 'DEFLATOR',
                      'V8000_DEFLA', 'V1904_DEFLA', 'COD_IMPUT_VALOR', 'COD_IMPUT_QUANTIDADE', 'FATOR_ANUALIZACAO',
                      'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],

                 "DESPESA_INDIVIDUAL": [[2, 4, 1, 9, 2, 1, 2, 2, 2, 7, 2, 10, 2, 2, 1, 1, 1, 12, 10, 1, 2, 14, 14, 10],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                         'COD_INFORMANTE', 'QUADRO', 'SEQ', 'V9001', 'V9002', 'V8000', 'V9010', 'V9011',
                                         'V9012', 'V4104', 'V4105', 'DEFLATOR', 'V8000_DEFLA', 'COD_IMPUT_VALOR',
                                         'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "DOMICILIO": [
                     [2, 4, 1, 9, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                      1, 1, 14, 14],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'V0201', 'V0202', 'V0203',
                      'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V02101', 'V02102', 'V02103', 'V02104',
                      'V02105', 'V02111', 'V02112', 'V02113', 'V0212', 'V0213', 'V02141', 'V02142', 'V0215', 'V02161',
                      'V02162', 'V02163', 'V02164', 'V0217', 'V0219', 'V0220', 'V0221', 'PESO', 'PESO_FINAL']],

                  "INVENTARIO": [[2, 4, 1, 9, 2, 1, 2, 2, 7, 2, 2, 4, 1, 14, 14, 10],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'QUADRO', 'SEQ', 'V9001', 'V9005', 'V9002', 'V1404', 'V9012', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],

                 "MORADOR": [
                     [2, 4, 1, 9, 2, 1, 2, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                      1, 1, 2, 1, 1, 2, 1, 1, 2, 1, 1, 1, 2, 1, 2, 14, 14, 10],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE',
                      'V0306', 'V0401', 'V04021', 'V04022', 'V04023', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407',
                      'V0408', 'V0409', 'V0410', 'V0411', 'V0412', 'V0413', 'V0414', 'V0415', 'V0416', 'V041711',
                      'V041712', 'V041721', 'V041722', 'V041731', 'V041732', 'V041741', 'V041742', 'V0418', 'V0419',
                      'V0420', 'V0421', 'V0422', 'V0423', 'V0424', 'V0425', 'V0426', 'V0427', 'V0428', 'V0429', 'V0430',
                      'ANOS_ESTUDO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],

                 "OUTROS_RENDIMENTOS": [[2, 4, 1, 9, 2, 1, 2, 2, 2, 7, 10, 10, 2, 2, 12, 10, 10, 1, 1, 14, 14, 10],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                         'COD_INFORMANTE', 'QUADRO', 'SEQ', 'V9001', 'V8500', 'V8501', 'V9010', 'V9011',
                                         'DEFLATOR', 'V8500_DEFLA', 'V8501_DEFLA', 'COD_IMPUT_VALOR',
                                         'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],

                 "RENDIMENTO_TRABALHO": [
                     [2, 4, 1, 9, 2, 1, 2, 2, 1, 1, 7, 1, 1, 1, 1, 1, 1, 7, 7, 7, 7, 2, 2, 3, 1, 12, 10, 10, 10, 10, 1,
                      1, 14, 14, 10],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE',
                      'QUADRO', 'SUB_QUADRO', 'SEQ', 'V9001', 'V5302', 'V53021', 'V5303', 'V5304', 'V5305', 'V5307',
                      'V8500', 'V531112', 'V531122', 'V531132', 'V9010', 'V9011', 'V5314', 'V5315', 'DEFLATOR',
                      'V8500_DEFLA', 'V531112_DEFLA', 'V531122_DEFLA', 'V531132_DEFLA', 'COD_IMPUT_VALOR',
                      'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']]}

    zip_file = ZipFile(path, mode='r')
    for text_file in zip_file.infolist():
        file = text_file.filename
        name = '{}'.format(text_file.filename.split('.')[0])
        name = name.split('/')[1]
        new_file = pd.read_fwf(zip_file.open(file), widths=pof_2017[name][0], index=False, header=None, dtype=str)
        if header:
            new_file.columns = pof_2017[name][1]
        
        if format == 'csv':
            new_file.to_csv('{}.csv'.format(name), index=False)
            print('{}.csv exportado com sucesso.'.format(name))
        else:
            try:
                new_file.to_excel('{}.xlsx'.format(name), index=False)
                print('{}.xlsx exportado com sucesso.'.format(name))       
            except:
                print(f'Erro gravando arquivo {name}.xlsx. Salvando em formato csv.')
                new_file.to_csv('{}.csv'.format(name), index=False)
                print('{}.csv exportado com sucesso.'.format(name))    

def ler_PNAD(path, ano, header=True):
    '''
    Realiza a leitura dos microdados da PNAD 2013, 2014 ou 2015 diretamente do arquivo .zip baixado do site do IBGE
    e exporta os dados como um arquivo .csv.
    :param path: aminho para o arquivo .zip
    :param ano: ano da PNAD.
    :param header:  boolean, Default True - acrescenta o código da variável como nome de cada coluna.
    https://github.com/otavio-s-s/lerMicrodados
    '''

    import pandas as pd
    from zipfile import ZipFile

    anos = [2013, 2014, 2015]

    if ano not in anos:
        raise ValueError('Ano inválido. Os anos aceitavéis são: {}'.format(anos))

    pnads = {
    'pnad_2013': {'PES2013': [[4, 8, 3, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4, 5, 1, 4, 5, 1, 1, 12, 1, 12, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 2, 1, 12, 1, 12, 1, 1, 1, 1, 2, 1, 1, 1, 2, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 1, 1, 2, 2, 2, 2, 1, 1, 2, 2, 1, 1, 1, 2, 4, 1, 1, 2, 2, 1, 1, 1, 2, 2, 2, 1, 1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 2, 12, 12, 12, 12, 12, 2, 2, 1, 1, 5, 5, 1, 1, 1, 2, 12, 2, 1, 1, 1, 1, 1, 12, 8], ['V0101',  'V0102', 'V0103', 'V0301', 'V0302', 'V3031', 'V3032', 'V3033', 'V8005', 'V0401', 'V0402', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407', 'V0408', 'V0409', 'V0410', 'V4111', 'V4112', 'V4011', 'V0412', 'V0501', 'V0502', 'V5030', 'V0504', 'V0505', 'V5061', 'V5062', 'V5063', 'V5064', 'V5065', 'V0507', 'V5080', 'V5090', 'V0510', 'V0511', 'V5121', 'V5122', 'V5123', 'V5124', 'V5125', 'V5126', 'V0601', 'V0602', 'V6002', 'V6020', 'V6003', 'V6030', 'V0604', 'V0605', 'V0606', 'V6007', 'V6070', 'V0608', 'V0609', 'V0610', 'V0611', 'V06111', 'V061111', 'V061112', 'V061113', 'V061114', 'V061115', 'V061116', 'V06112', 'V0612', 'V0701', 'V0702', 'V0703', 'V0704', 'V0705', 'V7060', 'V7070', 'V0708', 'V7090', 'V7100', 'V0711', 'V7121', 'V7122', 'V7124', 'V7125', 'V7127', 'V7128', 'V0713', 'V0714', 'V0715', 'V0716', 'V9001', 'V9002', 'V9003', 'V9004', 'V9005', 'V9906', 'V9907', 'V9008', 'V9009', 'V9010', 'V9011', 'V9012', 'V9013', 'V9014', 'V9151', 'V9152', 'V9154', 'V9156', 'V9157', 'V9159', 'V9161', 'V9162', 'V9164', 'V9016', 'V9017', 'V9018', 'V9019', 'V9201', 'V9202', 'V9204', 'V9206', 'V9207', 'V9209', 'V9211', 'V9212', 'V9214', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V9029', 'V9030', 'V9031', 'V9032', 'V9033', 'V9034', 'V9035', 'V9036', 'V9037', 'V9038', 'V9039', 'V9040', 'V9041', 'V9042', 'V9043', 'V9044', 'V9045', 'V9046', 'V9047', 'V9048', 'V9049', 'V9050', 'V9051', 'V9052', 'V9531', 'V9532', 'V9534', 'V9535', 'V9537', 'V90531', 'V90532', 'V90533', 'V9054', 'V9055', 'V9056', 'V9057', 'V9058', 'V9059', 'V9060', 'V9611', 'V9612', 'V9062', 'V9063', 'V9064', 'V9065', 'V9066', 'V9067', 'V9068', 'V9069', 'V9070', 'V9971', 'V9972', 'V9073', 'V9074', 'V9075', 'V9076', 'V9077', 'V9078', 'V9079', 'V9080', 'V9081', 'V9082', 'V9083', 'V9084', 'V9085', 'V9861', 'V9862', 'V9087', 'V9088', 'V9891', 'V9892', 'V9990', 'V9991', 'V9092', 'V9093', 'V9094', 'V9095', 'V9096', 'V9097', 'V9981', 'V9982', 'V9984', 'V9985', 'V9987', 'V9099', 'V9100', 'V9101', 'V1021', 'V1022', 'V1024', 'V1025', 'V1027', 'V1028', 'V9103', 'V9104', 'V9105', 'V9106', 'V9107', 'V9108', 'V1091', 'V1092', 'V9910', 'V9911', 'V9112', 'V9113', 'V9114', 'V9115', 'V9116', 'V9117', 'V9118', 'V9119', 'V9120', 'V9121', 'V9921', 'V9122', 'V9123', 'V9124', 'V1251', 'V1252', 'V1254', 'V1255', 'V1257', 'V1258', 'V1260', 'V1261', 'V1263', 'V1264', 'V1266', 'V1267', 'V1269', 'V1270', 'V1272', 'V1273', 'V9126', 'V1101', 'V1141', 'V1142', 'V1151', 'V1152', 'V1153', 'V1154', 'V1161', 'V1162', 'V1163', 'V1164', 'V1107', 'V1181', 'V1182', 'V1109', 'V1110', 'V1111', 'V1112', 'V1113', 'V1114', 'V1115', 'V4801', 'V4802', 'V4803', 'V4704', 'V4805', 'V4706', 'V4707', 'V4808', 'V4809', 'V4810', 'V4711', 'V4812', 'V4713', 'V4814', 'V4715', 'V4816', 'V4817', 'V4718', 'V4719', 'V4720', 'V4721', 'V4722', 'V4723', 'V4724', 'V4727', 'V4728', 'V4729', 'V4732', 'V4735', 'V4838', 'V6502', 'V4741', 'V4742', 'V4743', 'V4745', 'V4746', 'V4747', 'V4748', 'V4749', 'V4750', 'V9993']],
               'DOM2013': [[4, 8, 3, 2, 2, 2, 1, 1, 1, 1, 2, 2, 1, 12, 12, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 4, 2, 12, 3, 12, 6, 9, 3, 5, 12, 4, 7, 7, 2, 12, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 8], ['V0101', 'V0102', 'V0103', 'V0104', 'V0105', 'V0106', 'V0201', 'V0202', 'V0203', 'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V0210', 'V0211', 'V0212', 'V0213', 'V0214', 'V0215', 'V0216', 'V2016', 'V0217', 'V0218', 'V0219', 'V0220', 'V2020', 'V0221', 'V0222', 'V0223', 'V0224', 'V0225', 'V0226', 'V0227', 'V02270', 'V02271', 'V02272', 'V02273', 'V02274', 'V2027', 'V0228', 'V0229', 'V0230', 'V0231', 'V0232', 'V02321', 'V02322', 'V02323', 'V02324', 'V02325', 'V02326', 'V02327', 'V02424', 'V02425', 'V02426', 'V2032', 'V4105', 'V4107', 'V4600', 'V4601', 'V4602', 'V4604', 'V4605', 'V4606', 'V4607', 'V4608', 'V4609', 'V4610', 'V4611', 'V4614', 'UPA', 'V4617', 'V4618', 'V4620', 'V4621', 'V4622', 'V4623A', 'V4624', 'V2101', 'V2103', 'V2105', 'V2107', 'V2109', 'V2113', 'V2115', 'V2117', 'V2121', 'V2123', 'V2124', 'V2126', 'V2128', 'V2130', 'V2132', 'V2134', 'V2138', 'V2139', 'V9992']]},

    'pnad_2014': {'PES2014': [[4, 8, 3, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4, 5, 1, 4, 5, 1, 1, 12, 1, 12, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 2, 1, 12, 1, 12, 1, 1, 1, 1, 2, 1, 1, 1, 2, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 1, 1, 2, 2, 2, 2, 1, 1, 2, 2, 1, 1, 1, 2, 4, 1, 1, 2, 2, 1, 1, 1, 2, 2, 2, 1, 1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 2, 12, 12, 12, 12, 12, 2, 2, 1, 1, 5, 5, 1, 1, 1, 2, 12, 2, 1, 1, 1, 1, 1, 12, 1, 2, 1, 2, 1, 1, 4, 5, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 1, 1, 1, 5, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 8], ['V0101', 'V0102', 'V0103', 'V0301', 'V0302', 'V3031', 'V3032', 'V3033', 'V8005', 'V0401', 'V0402', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407', 'V0408', 'V0409', 'V0410', 'V4111', 'V4112', 'V4011', 'V0412', 'V0501', 'V0502', 'V5030', 'V0504', 'V0505', 'V5061', 'V5062', 'V5063', 'V5064', 'V5065', 'V0507', 'V5080', 'V5090', 'V0510', 'V0511', 'V5121', 'V5122', 'V5123', 'V5124', 'V5125', 'V5126', 'V0601', 'V0602', 'V6002', 'V6020', 'V6003', 'V6030', 'V0604', 'V0605', 'V0606', 'V6007', 'V6070', 'V0608', 'V0609', 'V0610', 'V0611', 'V06111', 'V061111', 'V061112', 'V061113', 'V061114', 'V061115', 'V061116', 'V06112', 'V0612', 'V0701', 'V0702', 'V0703', 'V0704', 'V0705', 'V7060', 'V7070', 'V0708', 'V7090', 'V7100', 'V0711', 'V7121', 'V7122', 'V7124', 'V7125', 'V7127', 'V7128', 'V0713', 'V0714', 'V0715', 'V0716', 'V9001', 'V9002', 'V9003', 'V9004', 'V9005', 'V9906', 'V9907', 'V9008', 'V9009', 'V9010', 'V9011', 'V9012', 'V9013', 'V9014', 'V9151', 'V9152', 'V9154', 'V9156', 'V9157', 'V9159', 'V9161', 'V9162', 'V9164', 'V9016', 'V9017', 'V9018', 'V9019', 'V9201', 'V9202', 'V9204', 'V9206', 'V9207', 'V9209', 'V9211', 'V9212', 'V9214', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V90281', 'V90282', 'V90283', 'V90284', 'V90285', 'V9029', 'V9030', 'V9031', 'V9032', 'V9033', 'V9034', 'V9035', 'V9036', 'V9037', 'V9038', 'V9039', 'V9040', 'V9041', 'V9042', 'V9043', 'V9044', 'V9045', 'V9046', 'V9047', 'V9048', 'V9049', 'V9050', 'V9051', 'V9052', 'V90521', 'V90522', 'V90523', 'V90524', 'V90525', 'V90526', 'V9531', 'V9532', 'V9534', 'V9535', 'V9537', 'V90531', 'V90532', 'V90533', 'V9054', 'V9055', 'V9056', 'V9057', 'V9058', 'V9059', 'V9060', 'V9611', 'V9612', 'V9062', 'V9063', 'V9064', 'V9065', 'V9066', 'V9067', 'V9068', 'V9069', 'V9070', 'V9971', 'V9972', 'V9073', 'V9074', 'V9075', 'V9076', 'V9077', 'V9078', 'V9079', 'V9080', 'V9081', 'V9082', 'V9083', 'V9084', 'V9085', 'V9861', 'V9862', 'V9087', 'V9088', 'V9891', 'V9892', 'V9990', 'V9991', 'V9092', 'V9093', 'V9094', 'V9095', 'V9096', 'V9097', 'V9981', 'V9982', 'V9984', 'V9985', 'V9987', 'V9099', 'V9100', 'V9101', 'V1021', 'V1022', 'V1024', 'V1025', 'V1027', 'V1028', 'V9103', 'V9104', 'V9105', 'V9106', 'V9107', 'V9108', 'V1091', 'V1092', 'V9910', 'V9911', 'V9112', 'V9113', 'V9114', 'V9115', 'V9116', 'V9117', 'V9118', 'V9119', 'V9120', 'V9121', 'V9921', 'V9122', 'V9123', 'V9124', 'V1251', 'V1252', 'V1254', 'V1255', 'V1257', 'V1258', 'V1260', 'V1261', 'V1263', 'V1264', 'V1266', 'V1267', 'V1269', 'V1270', 'V1272', 'V1273', 'V9126', 'V1101', 'V1141', 'V1142', 'V1151', 'V1152', 'V1153', 'V1154', 'V1161', 'V1162', 'V1163', 'V1164', 'V1107', 'V1181', 'V1182', 'V1109', 'V1110', 'V1111', 'V1112', 'V1113', 'V1114', 'V1115', 'V4801', 'V4802', 'V4803', 'V4704', 'V4805', 'V4706', 'V4707', 'V4808', 'V4809', 'V4810', 'V4711', 'V4812', 'V4713', 'V4814', 'V4715', 'V4816', 'V4817', 'V4718', 'V4719', 'V4720', 'V4721', 'V4722', 'V4723', 'V4724', 'V4727', 'V4728', 'V4729', 'V4732', 'V4735', 'V4838', 'V6502', 'V4741', 'V4742', 'V4743', 'V4745', 'V4746', 'V4747', 'V4748', 'V4749', 'V4750', 'V0305', 'V32000', 'V32001', 'V32002', 'V32003', 'V32004', 'V32005', 'V32006', 'V32007', 'V32008', 'V32009', 'V32010', 'V32011', 'V32012', 'V32013', 'V32014', 'V32015', 'V32016', 'V32017', 'V32018', 'V32019', 'V32020', 'V32021', 'V32022', 'V32023', 'V32024', 'V32025', 'V32026', 'V32027', 'V32028', 'V32029', 'V32030', 'V32031', 'V32032', 'V32033', 'V32034', 'V32035', 'V32036', 'V32037', 'V32038', 'V32039', 'V31001', 'V31002', 'V31003', 'V31004', 'V31005', 'V31007', 'V31008', 'V31009', 'V31010', 'V31012', 'V31013', 'V31015', 'V31016', 'V31017', 'V31018', 'V31020', 'V31025', 'V31026', 'V31027', 'V31028', 'V31029', 'V31030', 'V31032', 'V31033', 'V31034', 'V31035', 'V31037', 'V31038', 'V31039', 'V31040', 'V31042', 'V31043', 'V31044', 'V31045', 'V31046', 'V31047', 'V31048', 'V31050', 'V31055', 'V31056', 'V31057', 'V31058', 'V31059', 'V31060', 'V31061', 'V31062', 'V31064', 'V31065', 'V31066', 'V31067', 'V31068', 'V31069', 'V31070', 'V31071', 'V31072', 'V31073', 'V31074', 'V31076', 'V31077', 'V31078', 'V31079', 'V31080', 'V31081', 'V31082', 'V31083', 'V31084', 'V31085', 'V31086', 'V31087', 'V31088', 'V31089', 'V31090', 'V31091', 'V31092', 'V31093', 'V31094', 'V9993']],
                'DOM2014': [[4, 8, 3, 2, 2, 2, 1, 1, 1, 1, 2, 2, 1, 12, 12, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 4, 2, 12, 3, 12, 6, 9, 3, 5, 12, 4, 7, 7, 2, 12, 2, 1, 1, 9, 5, 1, 1, 1, 1, 8], ['V0101', 'V0102', 'V0103', 'V0104', 'V0105', 'V0106', 'V0201', 'V0202', 'V0203', 'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V0210', 'V0211', 'V0212', 'V0213', 'V0214', 'V0215', 'V0216', 'V2016', 'V0217', 'V0218', 'V0219', 'V0220', 'V2020', 'V0221', 'V0222', 'V0223', 'V0224', 'V0225', 'V0226', 'V0227', 'V02270', 'V02271', 'V02272', 'V02273', 'V02274', 'V2027', 'V0228', 'V0229', 'V0230', 'V0231', 'V0232', 'V02321', 'V02322', 'V02323', 'V02324', 'V02325', 'V02326', 'V02327', 'V02424', 'V02425', 'V02426', 'V2032', 'V4105', 'V4107', 'V4600', 'V4601', 'V4602', 'V4604', 'V4605', 'V4606', 'V4607', 'V4608', 'V4609', 'V4610', 'V4611', 'V4614', 'UPA', 'V4617', 'V4618', 'V4620', 'V4621', 'V4622', 'V4624', 'V4625', 'V4626', 'V4627', 'V33001', 'V33002', 'V33003', 'V33004', 'V9992']]},

    'pnad_2015': {'PES2015': [[4, 8, 3, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4, 5, 1, 4, 5, 1, 1, 12, 1, 12, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 2, 1, 12, 1, 12, 1, 1, 1, 1, 2, 1, 1, 1, 2, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 1, 1, 2, 2, 2, 2, 1, 1, 2, 2, 1, 1, 1, 2, 4, 1, 1, 2, 2, 1, 1, 1, 2, 2, 2, 1, 1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 2, 12, 12, 12, 12, 12, 2, 2, 1, 1, 5, 5, 1, 1, 1, 2, 12, 2, 1, 1, 1, 1, 1, 12, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 5, 5, 1, 1, 1, 1, 1, 2, 1, 1, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 5, 2, 1, 1, 1, 1, 2, 1, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 5, 5, 8], ['V0101', 'V0102', 'V0103', 'V0301', 'V0302', 'V3031', 'V3032', 'V3033', 'V8005', 'V0401', 'V0402', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407', 'V0408', 'V4111', 'V4112', 'V4011', 'V0412', 'V0501', 'V0502', 'V5030', 'V0504', 'V0505', 'V5061', 'V5062', 'V5063', 'V5064', 'V5065', 'V0507', 'V5080', 'V5090', 'V0510', 'V0511', 'V5121', 'V5122', 'V5123', 'V5124', 'V5125', 'V5126', 'V0601', 'V0602', 'V6002', 'V6020', 'V6003', 'V6030', 'V0604', 'V0605', 'V0606', 'V6007', 'V6070', 'V0608', 'V0609', 'V0610', 'V0611', 'V06111', 'V061111', 'V061112', 'V061113', 'V061114', 'V061115', 'V061116', 'V06112', 'V0612', 'V0701', 'V0702', 'V0703', 'V0704', 'V0705', 'V7060', 'V7070', 'V0708', 'V7090', 'V7100', 'V0711', 'V7121', 'V7122', 'V7124', 'V7125', 'V7127', 'V7128', 'V0713', 'V0714', 'V0715', 'V0716', 'V9001', 'V9002', 'V9003', 'V9004', 'V9005', 'V9906', 'V9907', 'V9008', 'V9009', 'V9010', 'V90101', 'V9011', 'V9012', 'V90121', 'V9013', 'V9014', 'V9151', 'V9152', 'V9154', 'V9156', 'V9157', 'V9159', 'V9161', 'V9162', 'V9164', 'V9016', 'V9017', 'V9018', 'V9019', 'V9201', 'V9202', 'V9204', 'V9206', 'V9207', 'V9209', 'V9211', 'V9212', 'V9214', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V9029', 'V9030', 'V9031', 'V9032', 'V9033', 'V9034', 'V9035', 'V9036', 'V9037', 'V9038', 'V9039', 'V9040', 'V9041', 'V9042', 'V9043', 'V9044', 'V9045', 'V9046', 'V9047', 'V9048', 'V9049', 'V9050', 'V9051', 'V9052', 'V9531', 'V9532', 'V9534', 'V9535', 'V9537', 'V90531', 'V90532', 'V90533', 'V9054', 'V9055', 'V9056', 'V9057', 'V9058', 'V9059', 'V9060', 'V9611', 'V9612', 'V9062', 'V9063', 'V9064', 'V9065', 'V9066', 'V9067', 'V9068', 'V9069', 'V9070', 'V9971', 'V9972', 'V9073', 'V9074', 'V9075', 'V9076', 'V9077', 'V9078', 'V9079', 'V9080', 'V9081', 'V9082', 'V9083', 'V9084', 'V9085', 'V9861', 'V9862', 'V9087', 'V90871', 'V908721', 'V908722', 'V908723', 'V908724', 'V908725', 'V908726', 'V90873', 'V90874', 'V9088', 'V90881', 'V90882', 'V908831', 'V908832', 'V908833', 'V908834', 'V908835', 'V908836', 'V908837', 'V90884', 'V908851', 'V908852', 'V908853', 'V908854', 'V908855', 'V908856', 'V90886', 'V90887', 'V908881', 'V908882', 'V908883', 'V908884', 'V908885', 'V908886', 'V908887', 'V9891', 'V9892', 'V9990', 'V9991', 'V9092', 'V9093', 'V9094', 'V9095', 'V9096', 'V9097', 'V9981', 'V9982', 'V9984', 'V9985', 'V9987', 'V9099', 'V9100', 'V9101', 'V1021', 'V1022', 'V1024', 'V1025', 'V1027', 'V1028', 'V9103', 'V9104', 'V9105', 'V9106', 'V9107', 'V9108', 'V1091', 'V1092', 'V9910', 'V9911', 'V9112', 'V9113', 'V9114', 'V9115', 'V9116', 'V9117', 'V9118', 'V9119', 'V9120', 'V9121', 'V9921', 'V9122', 'V9123', 'V9124', 'V1251', 'V1252', 'V1254', 'V1255', 'V1257', 'V1258', 'V1260', 'V1261', 'V1263', 'V1264', 'V1266', 'V1267', 'V1269', 'V1270', 'V1272', 'V1273', 'V9126', 'V1101', 'V1141', 'V1142', 'V1151', 'V1152', 'V1153', 'V1154', 'V1161', 'V1162', 'V1163', 'V1164', 'V1107', 'V1181', 'V1182', 'V1109', 'V1110', 'V1111', 'V1112', 'V1113', 'V1114', 'V1115', 'V4801', 'V4802', 'V4803', 'V4704', 'V4805', 'V4706', 'V4707', 'V4808', 'V4809', 'V4810', 'V4711', 'V4812', 'V4713', 'V4814', 'V4715', 'V4816', 'V4817', 'V4718', 'V4719', 'V4720', 'V4721', 'V4722', 'V4723', 'V4724', 'V4727', 'V4728', 'V4729', 'V4732', 'V4735', 'V4838', 'V6502', 'V4741', 'V4742', 'V4743', 'V4745', 'V4746', 'V4747', 'V4748', 'V4749', 'V4750', 'V38011', 'V38012', 'V3802', 'V3803', 'V3804', 'V3805', 'V3806', 'V3807', 'V3808', 'V3809', 'V37001', 'V37002', 'V3701', 'V3702', 'V3703', 'V3704', 'V3705', 'V3706', 'V37071', 'V37072', 'V37073', 'V37074', 'V37075', 'V37091', 'V37092', 'V37093', 'V37094', 'V37095', 'V37096', 'V37097', 'V37098', 'V3719', 'V3720', 'V36001', 'V36002', 'V3601', 'V3602', 'V3603', 'V3604', 'V3605', 'V3606', 'V3607', 'V3608', 'V3609', 'V3610', 'V3611', 'V3612', 'V3613', 'V3614', 'V3615', 'V3616', 'V3617', 'V3618', 'V3619', 'V3620', 'V3621', 'V3622', 'V3623', 'V3624', 'V3625', 'V3626', 'V3627', 'V3628', 'V3629', 'V3630', 'V3631', 'V3632', 'V3633', 'V3634', 'V3637', 'V3638', 'V9993']],
               'DOM2015': [[4, 8, 3, 2, 2, 2, 1, 1, 1, 1, 2, 2, 1, 12, 12, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 4, 2, 12, 3, 12, 6, 9, 3, 5, 12, 4, 7, 7, 2, 12, 2, 1, 1, 7, 5, 7, 5, 8], ['V0101', 'V0102', 'V0103', 'V0104', 'V0105', 'V0106', 'V0201', 'V0202', 'V0203', 'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V0210', 'V0211', 'V0212', 'V0213', 'V0214', 'V0215', 'V0216', 'V2016', 'V0217', 'V0218', 'V0219', 'V0220', 'V2020', 'V0221', 'V0222', 'V0223', 'V0224', 'V0225', 'V0226', 'V0227', 'V02270', 'V02271', 'V02272', 'V02273', 'V02274', 'V2027', 'V0228', 'V0229', 'V0230', 'V0231', 'V0232', 'V02321', 'V02322', 'V02323', 'V02324', 'V02325', 'V02326', 'V02327', 'V02424', 'V02425', 'V02426', 'V2032', 'V4105', 'V4107', 'V4600', 'V4601', 'V4602', 'V4604', 'V4605', 'V4606', 'V4607', 'V4608', 'V4609', 'V4610', 'V4611', 'V4614', 'UPA', 'V4617', 'V4618', 'V4620', 'V4621', 'V4622', 'V4624', 'V4628', 'V4629', 'V4630', 'V4632', 'V4633', 'V9992']]}}

    zip_file = ZipFile(path)
    for text_file in zip_file.infolist():
        dict_name = 'pnad_{}'.format(ano)
        file = text_file.filename
        name = '{}'.format(file.split('.')[0]).split('/')[1]
        new_file = pd.read_fwf(zip_file.open(file), widths=pnads[dict_name][name][0], index=False, header=None, dtype=str)
        if header:
            new_file.columns = pnads[dict_name][name][1]
        new_file.to_csv('{}.csv'.format(name), index=False)
        print('{}.csv exportado com sucesso.'.format(name))


def ler_PNS(path, header=True):
    '''
    Realiza a leitura dos microdados da PNS 2013 diretamente do arquivo .zip baixado do site do IBGE
    e exporta os dados como um arquivo .csv.
    :param path: caminho para o arquivo .zip
    :param header: boolean, Default True - acrescenta o código da variável como nome de cada coluna.
    https://github.com/otavio-s-s/lerMicrodados
    '''
    import pandas as pd
    from zipfile import ZipFile

    pns_2013 = {'PESPNS2013': [['V0024',
                                'UPA_PNS',
                                'V0006_PNS',
                                'UPA',
                                'V0025',
                                'C001',
                                'C00301',
                                'C004',
                                'C006',
                                'C00701',
                                'C00702',
                                'C00703',
                                'C008',
                                'C009',
                                'C010',
                                'C011',
                                'C012',
                                'D001',
                                'D002',
                                'D003',
                                'D004',
                                'D005',
                                'D006',
                                'D007',
                                'D008',
                                'D009',
                                'D010',
                                'D011',
                                'D012',
                                'D013',
                                'D014',
                                'D015',
                                'E001',
                                'E002',
                                'E003',
                                'E004',
                                'E005',
                                'E006',
                                'E007',
                                'E008',
                                'E01001',
                                'E01002',
                                'E01003',
                                'E011',
                                'E01201',
                                'E014',
                                'E01501',
                                'E01601',
                                'E01602',
                                'E01603',
                                'E01604',
                                'E01605',
                                'E017',
                                'E01801',
                                'E01802',
                                'E01803',
                                'E01804',
                                'E01805',
                                'E019',
                                'E020',
                                'E021',
                                'E022',
                                'E023',
                                'E024',
                                'E025',
                                'E02501',
                                'E02502',
                                'E026',
                                'E027',
                                'F001',
                                'F00102',
                                'F007',
                                'F00702',
                                'F008',
                                'F00802',
                                'VDF001',
                                'VDF00102',
                                'G001',
                                'G002',
                                'G00201',
                                'G003',
                                'G004',
                                'G005',
                                'G006',
                                'G007',
                                'G00701',
                                'G008',
                                'G009',
                                'G010',
                                'G014',
                                'G015',
                                'G01501',
                                'G016',
                                'G017',
                                'G018',
                                'G021',
                                'G022',
                                'G02201',
                                'G023',
                                'G024',
                                'G02501',
                                'G02502',
                                'G02503',
                                'G026',
                                'G027',
                                'G032',
                                'I001',
                                'I002',
                                'I003',
                                'I004',
                                'I005',
                                'I006',
                                'I007',
                                'I00701',
                                'I008',
                                'I009',
                                'I010',
                                'I011',
                                'J001',
                                'J002',
                                'J003',
                                'J004',
                                'J005',
                                'J006',
                                'J007',
                                'J008',
                                'J009',
                                'J010',
                                'J011',
                                'J012',
                                'J013',
                                'J014',
                                'J015',
                                'J016',
                                'J017',
                                'J018',
                                'J019',
                                'J020',
                                'J021',
                                'J022',
                                'J023',
                                'J024',
                                'J025',
                                'J026',
                                'J027',
                                'J029',
                                'J030',
                                'J031',
                                'J032',
                                'J033',
                                'J034',
                                'J035',
                                'J036',
                                'J037',
                                'J038',
                                'J039',
                                'J04001',
                                'J04002',
                                'J041',
                                'J042',
                                'J043',
                                'J044',
                                'J045',
                                'J046',
                                'J047',
                                'J048',
                                'J049',
                                'J050',
                                'J051',
                                'J052',
                                'J053',
                                'J054',
                                'J055',
                                'J056',
                                'J057',
                                'J058',
                                'J059',
                                'J060',
                                'K001',
                                'K002',
                                'K003',
                                'K004',
                                'K005',
                                'K006',
                                'K007',
                                'K008',
                                'K009',
                                'K010',
                                'K011',
                                'K012',
                                'K013',
                                'K014',
                                'K015',
                                'K016',
                                'K017',
                                'K018',
                                'K019',
                                'K020',
                                'K021',
                                'K022',
                                'K023',
                                'K024',
                                'K025',
                                'K026',
                                'K027',
                                'K028',
                                'K029',
                                'K030',
                                'K031',
                                'K032',
                                'K033',
                                'K034',
                                'K035',
                                'K036',
                                'K039',
                                'K040',
                                'K041',
                                'K042',
                                'K043',
                                'K044',
                                'K045',
                                'K046',
                                'K047',
                                'K048',
                                'K049',
                                'K050',
                                'K051',
                                'K052',
                                'K053',
                                'K054',
                                'K055',
                                'K056',
                                'K057',
                                'K058',
                                'K059',
                                'K06001',
                                'K06002',
                                'K06003',
                                'K06101',
                                'K06102',
                                'K062',
                                'L00201',
                                'L00202',
                                'L00203',
                                'L003',
                                'L004',
                                'L005',
                                'L006',
                                'L007',
                                'L008',
                                'L009',
                                'L010',
                                'L011',
                                'L012',
                                'L013',
                                'L014',
                                'L015',
                                'L01613',
                                'L01614',
                                'L01615',
                                'L01616',
                                'L01617',
                                'L01618',
                                'L01619',
                                'L01620',
                                'L01621',
                                'L01622',
                                'L01623',
                                'L01624',
                                'L01701',
                                'L01702',
                                'L01703',
                                'L01704',
                                'L01705',
                                'L01706',
                                'L01707',
                                'L01708',
                                'L01709',
                                'L01710',
                                'L01711',
                                'L01712',
                                'L01713',
                                'L01714',
                                'L01715',
                                'L01716',
                                'L018',
                                'L019',
                                'L020',
                                'M001',
                                'M002',
                                'M003',
                                'M005',
                                'M006',
                                'M007',
                                'M008',
                                'M009',
                                'M010',
                                'M01101',
                                'M01102',
                                'M01103',
                                'M01104',
                                'M01105',
                                'M01106',
                                'M01107',
                                'M01108',
                                'M01301',
                                'M01302',
                                'M01303',
                                'M014',
                                'M015',
                                'M016',
                                'M017',
                                'M018',
                                'M019',
                                'N001',
                                'N002',
                                'N003',
                                'N004',
                                'N005',
                                'N006',
                                'N007',
                                'N008',
                                'N010',
                                'N011',
                                'N012',
                                'N013',
                                'N014',
                                'N015',
                                'N016',
                                'N017',
                                'N018',
                                'N019',
                                'N020',
                                'N021',
                                'N022',
                                'N023',
                                'O001',
                                'O002',
                                'O003',
                                'O004',
                                'O005',
                                'O006',
                                'O007',
                                'O008',
                                'O009',
                                'O00901',
                                'O010',
                                'O011',
                                'O012',
                                'O013',
                                'O014',
                                'O015',
                                'O016',
                                'O017',
                                'O01801',
                                'O01802',
                                'O019',
                                'O020',
                                'O021',
                                'O02101',
                                'O022',
                                'O023',
                                'O024',
                                'O025',
                                'O027',
                                'O028',
                                'O029',
                                'O030',
                                'O031',
                                'O032',
                                'O033',
                                'O034',
                                'O035',
                                'O036',
                                'O037',
                                'O038',
                                'O039',
                                'O040',
                                'O041',
                                'O042',
                                'O043',
                                'O044',
                                'O045',
                                'O046',
                                'O047',
                                'O048',
                                'P001',
                                'P00101',
                                'P002',
                                'P003',
                                'P00301',
                                'P004',
                                'P00401',
                                'P005',
                                'P006',
                                'P007',
                                'P008',
                                'P009',
                                'P010',
                                'P011',
                                'P012',
                                'P013',
                                'P014',
                                'P015',
                                'P016',
                                'P017',
                                'P018',
                                'P019',
                                'P020',
                                'P021',
                                'P022',
                                'P023',
                                'P024',
                                'P025',
                                'P026',
                                'P02601',
                                'P027',
                                'P028',
                                'P029',
                                'P030',
                                'P031',
                                'P032',
                                'P033',
                                'P034',
                                'P035',
                                'P036',
                                'P03701',
                                'P03702',
                                'P038',
                                'P039',
                                'P03901',
                                'P03902',
                                'P03903',
                                'P040',
                                'P04101',
                                'P04102',
                                'P042',
                                'P04301',
                                'P04302',
                                'P044',
                                'P04401',
                                'P04403',
                                'P04404',
                                'P045',
                                'P046',
                                'P047',
                                'P048',
                                'P049',
                                'P050',
                                'P051',
                                'P052',
                                'P053',
                                'P05401',
                                'P05402',
                                'P05403',
                                'P05404',
                                'P05405',
                                'P05406',
                                'P05407',
                                'P05408',
                                'P05409',
                                'P05410',
                                'P05411',
                                'P05412',
                                'P05413',
                                'P05414',
                                'P05415',
                                'P05416',
                                'P05417',
                                'P05418',
                                'P05419',
                                'P05421',
                                'P05422',
                                'P055',
                                'P056',
                                'P05601',
                                'P05602',
                                'P05603',
                                'P05604',
                                'P05605',
                                'P057',
                                'P058',
                                'P05801',
                                'P05802',
                                'P05901',
                                'P05902',
                                'P05903',
                                'P05904',
                                'P060',
                                'P061',
                                'P062',
                                'P063',
                                'P064',
                                'P065',
                                'P066',
                                'P067',
                                'P068',
                                'P069',
                                'P07001',
                                'P07002',
                                'P07003',
                                'P071',
                                'P072',
                                'Q001',
                                'Q002',
                                'Q003',
                                'Q004',
                                'Q005',
                                'Q006',
                                'Q007',
                                'Q008',
                                'Q009',
                                'Q010',
                                'Q011',
                                'Q012',
                                'Q013',
                                'Q014',
                                'Q015',
                                'Q016',
                                'Q017',
                                'Q01801',
                                'Q01802',
                                'Q01803',
                                'Q01804',
                                'Q01805',
                                'Q01806',
                                'Q01807',
                                'Q01808',
                                'Q01901',
                                'Q01902',
                                'Q01903',
                                'Q01904',
                                'Q01905',
                                'Q020',
                                'Q021',
                                'Q022',
                                'Q023',
                                'Q024',
                                'Q026',
                                'Q027',
                                'Q028',
                                'Q029',
                                'Q030',
                                'Q031',
                                'Q032',
                                'Q033',
                                'Q03401',
                                'Q03402',
                                'Q035',
                                'Q036',
                                'Q037',
                                'Q038',
                                'Q039',
                                'Q040',
                                'Q041',
                                'Q042',
                                'Q043',
                                'Q044',
                                'Q045',
                                'Q04601',
                                'Q04602',
                                'Q04603',
                                'Q04604',
                                'Q04605',
                                'Q04606',
                                'Q04607',
                                'Q04608',
                                'Q04609',
                                'Q04701',
                                'Q04702',
                                'Q04703',
                                'Q04704',
                                'Q04705',
                                'Q048',
                                'Q049',
                                'Q050',
                                'Q051',
                                'Q052',
                                'Q053',
                                'Q054',
                                'Q05501',
                                'Q05502',
                                'Q05503',
                                'Q05504',
                                'Q05505',
                                'Q05506',
                                'Q05507',
                                'Q05508',
                                'Q05509',
                                'Q056',
                                'Q057',
                                'Q058',
                                'Q059',
                                'Q060',
                                'Q061',
                                'Q06201',
                                'Q06202',
                                'Q06203',
                                'Q06204',
                                'Q06205',
                                'Q06206',
                                'Q063',
                                'Q06301',
                                'Q06302',
                                'Q06303',
                                'Q06304',
                                'Q064',
                                'Q06501',
                                'Q06502',
                                'Q06503',
                                'Q06504',
                                'Q066',
                                'Q067',
                                'Q068',
                                'Q069',
                                'Q070',
                                'Q071',
                                'Q07201',
                                'Q07202',
                                'Q07203',
                                'Q07204',
                                'Q07205',
                                'Q07206',
                                'Q073',
                                'Q074',
                                'Q075',
                                'Q076',
                                'Q07701',
                                'Q07702',
                                'Q078',
                                'Q079',
                                'Q080',
                                'Q08101',
                                'Q08102',
                                'Q08103',
                                'Q08104',
                                'Q08105',
                                'Q082',
                                'Q083',
                                'Q084',
                                'Q085',
                                'Q08601',
                                'Q08603',
                                'Q08604',
                                'Q08605',
                                'Q087',
                                'Q088',
                                'Q089',
                                'Q09001',
                                'Q09003',
                                'Q09004',
                                'Q09005',
                                'Q091',
                                'Q092',
                                'Q093',
                                'Q094',
                                'Q095',
                                'Q09601',
                                'Q09602',
                                'Q09603',
                                'Q097',
                                'Q098',
                                'Q100',
                                'Q101',
                                'Q102',
                                'Q103',
                                'Q104',
                                'Q105',
                                'Q106',
                                'Q107',
                                'Q108',
                                'Q109',
                                'Q110',
                                'Q11001',
                                'Q11002',
                                'Q11003',
                                'Q11004',
                                'Q111',
                                'Q112',
                                'Q113',
                                'Q11401',
                                'Q11402',
                                'Q11403',
                                'Q115',
                                'Q116',
                                'Q11601',
                                'Q11602',
                                'Q11603',
                                'Q117',
                                'Q11801',
                                'Q11802',
                                'Q11803',
                                'Q11804',
                                'Q119',
                                'Q120',
                                'Q121',
                                'Q122',
                                'Q123',
                                'Q124',
                                'Q125',
                                'Q12601',
                                'Q12602',
                                'Q12603',
                                'Q12604',
                                'Q12605',
                                'Q127',
                                'Q128',
                                'Q130',
                                'Q131',
                                'Q132',
                                'Q133',
                                'Q134',
                                'Q135',
                                'Q136',
                                'Q137',
                                'R001',
                                'R002',
                                'R003',
                                'R004',
                                'R005',
                                'R006',
                                'R007',
                                'R008',
                                'R009',
                                'R010',
                                'R011',
                                'R012',
                                'R013',
                                'R014',
                                'R015',
                                'R016',
                                'R017',
                                'R018',
                                'R019',
                                'R020',
                                'R021',
                                'R022',
                                'R023',
                                'R024',
                                'R025',
                                'R026',
                                'R027',
                                'R028',
                                'R029',
                                'R030',
                                'R031',
                                'R032',
                                'R033',
                                'R034',
                                'R035',
                                'R03601',
                                'R03602',
                                'R03603',
                                'R03604',
                                'R03605',
                                'R03606',
                                'R03607',
                                'R03608',
                                'R03609',
                                'R03610',
                                'R03611',
                                'R037',
                                'R038',
                                'R039',
                                'R040',
                                'R041',
                                'R04101',
                                'R042',
                                'R04201',
                                'R043',
                                'R044',
                                'R045',
                                'R046',
                                'R047',
                                'R048',
                                'R04901',
                                'R04902',
                                'R04903',
                                'S001',
                                'S002',
                                'S003',
                                'S004',
                                'S005',
                                'S006',
                                'S007',
                                'S008',
                                'S009',
                                'S01001',
                                'S01002',
                                'S01003',
                                'S01004',
                                'S01005',
                                'S01101',
                                'S01102',
                                'S01103',
                                'S012',
                                'S013',
                                'S01401',
                                'S01402',
                                'S01403',
                                'S01404',
                                'S01405',
                                'S015',
                                'S016',
                                'S017',
                                'S018',
                                'S019',
                                'S020',
                                'S021',
                                'S022',
                                'S023',
                                'S024',
                                'S025',
                                'S026',
                                'S027',
                                'S028',
                                'S029',
                                'S030',
                                'S031',
                                'S032',
                                'S033',
                                'S034',
                                'S035',
                                'S036',
                                'S037',
                                'S038',
                                'S039',
                                'S040',
                                'S041',
                                'S042',
                                'S043',
                                'S044',
                                'S045',
                                'S046',
                                'S047',
                                'S048',
                                'S049',
                                'S050',
                                'S051',
                                'S052',
                                'S053',
                                'S054',
                                'S055',
                                'S056',
                                'S057',
                                'S058',
                                'U001',
                                'U00201',
                                'U00202',
                                'U00203',
                                'U004',
                                'U005',
                                'U006',
                                'U009',
                                'U010',
                                'U011',
                                'U014',
                                'U015',
                                'U01701',
                                'U01702',
                                'U01801',
                                'U01802',
                                'U019',
                                'U020',
                                'U021',
                                'U022',
                                'U023',
                                'U02301',
                                'U024',
                                'U02401',
                                'U025',
                                'X001',
                                'X002',
                                'X003',
                                'X004',
                                'X005',
                                'X006',
                                'X007',
                                'X008',
                                'X011',
                                'X012',
                                'X01401',
                                'X01402',
                                'X01501',
                                'X01502',
                                'X016',
                                'X017',
                                'X018',
                                'X019',
                                'X02001',
                                'X02002',
                                'X02003',
                                'X02004',
                                'X02005',
                                'X02006',
                                'X02201',
                                'X02202',
                                'X02203',
                                'X02204',
                                'X02205',
                                'X02206',
                                'X024',
                                'X02501',
                                'X02502',
                                'X02503',
                                'X02504',
                                'X02505',
                                'X02506',
                                'X02507',
                                'X02508',
                                'X02509',
                                'X02510',
                                'W00101',
                                'W00102',
                                'W00103',
                                'W00201',
                                'W00202',
                                'W00203',
                                'W00301',
                                'W00302',
                                'W00303',
                                'W00401',
                                'W00402',
                                'W00403',
                                'W00404',
                                'W00405',
                                'W00406',
                                'W00407',
                                'W00408',
                                'V0028',
                                'V0029',
                                'V00281',
                                'V00291',
                                'V00282',
                                'V00292',
                                'V00283',
                                'V00293',
                                'VDD004',
                                'VDE001',
                                'VDE002',
                                'VDDATAM'],
                               [8,
                                7,
                                4,
                                9,
                                1,
                                2,
                                2,
                                2,
                                1,
                                2,
                                2,
                                4,
                                3,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                2,
                                1,
                                4,
                                1,
                                5,
                                1,
                                8,
                                1,
                                8,
                                1,
                                3,
                                1,
                                8,
                                1,
                                8,
                                1,
                                3,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                8,
                                1,
                                8,
                                1,
                                8,
                                1,
                                8,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                3,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                2,
                                2,
                                2,
                                1,
                                2,
                                2,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                4,
                                2,
                                2,
                                4,
                                2,
                                2,
                                4,
                                2,
                                2,
                                4,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                2,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                3,
                                1,
                                1,
                                3,
                                1,
                                3,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                2,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                2,
                                2,
                                1,
                                1,
                                2,
                                2,
                                2,
                                2,
                                2,
                                9,
                                1,
                                2,
                                2,
                                2,
                                2,
                                2,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                2,
                                2,
                                2,
                                2,
                                2,
                                2,
                                2,
                                2,
                                2,
                                4,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                2,
                                3,
                                3,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                2,
                                1,
                                1,
                                1,
                                2,
                                2,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                2,
                                1,
                                2,
                                1,
                                2,
                                1,
                                1,
                                1,
                                2,
                                2,
                                2,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                2,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                5,
                                5,
                                5,
                                5,
                                5,
                                5,
                                5,
                                5,
                                5,
                                3,
                                3,
                                3,
                                3,
                                3,
                                3,
                                5,
                                5,
                                14,
                                14,
                                14,
                                14,
                                9,
                                17,
                                3,
                                5,
                                1,
                                1,
                                1,
                                8]], 'DOMPNS2013': [['V0001',
                                                     'V0024',
                                                     'UPA_PNS',
                                                     'V0006_PNS',
                                                     'UPA',
                                                     'V0015',
                                                     'V0020',
                                                     'V0022',
                                                     'V0026',
                                                     'V0031',
                                                     'A001',
                                                     'A002',
                                                     'A003',
                                                     'A004',
                                                     'A005',
                                                     'A006',
                                                     'A007',
                                                     'A008',
                                                     'A009',
                                                     'A010',
                                                     'A011',
                                                     'A012',
                                                     'A013',
                                                     'A014',
                                                     'A015',
                                                     'A016',
                                                     'A017',
                                                     'A01801',
                                                     'A01802',
                                                     'A01803',
                                                     'A01804',
                                                     'A01805',
                                                     'A01806',
                                                     'A01807',
                                                     'A01808',
                                                     'A01809',
                                                     'A01810',
                                                     'A01811',
                                                     'A01812',
                                                     'A01813',
                                                     'A01814',
                                                     'A01815',
                                                     'A01816',
                                                     'A01817',
                                                     'A01818',
                                                     'A019',
                                                     'A020',
                                                     'A021',
                                                     'A022',
                                                     'A02301',
                                                     'A02302',
                                                     'A02303',
                                                     'A02304',
                                                     'A024',
                                                     'B001',
                                                     'B002',
                                                     'B003',
                                                     'B004',
                                                     'V0028',
                                                     'V0029',
                                                     'V00281',
                                                     'V00291',
                                                     'V00282',
                                                     'V00292',
                                                     'V00283',
                                                     'V00293',
                                                     'VDC001',
                                                     'VDC002',
                                                     'VDDATAD'],
                                                    [2,
                                                     8,
                                                     7,
                                                     4,
                                                     9,
                                                     2,
                                                     4,
                                                     2,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     2,
                                                     2,
                                                     1,
                                                     1,
                                                     2,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     3,
                                                     3,
                                                     3,
                                                     3,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     1,
                                                     14,
                                                     14,
                                                     14,
                                                     14,
                                                     9,
                                                     17,
                                                     3,
                                                     5,
                                                     2,
                                                     2,
                                                     8]]}

    zip_file = ZipFile(path)
    for text_file in zip_file.infolist():
        if text_file.filename == 'Dados/DOMPNS2013.txt' or text_file.filename == 'Dados/PESPNS2013.txt':
            file = text_file.filename
            name = '{}'.format(file.split('.')[0]).split('/')[1]
            new_file = pd.read_fwf(zip_file.open(file), widths=pns_2013[name][1], index=False, header=None, dtype=str)
            if header:
                new_file.columns = pns_2013[name][0]
            new_file.to_csv('{}.csv'.format(name), index=False)
            print('{}.csv exportado com sucesso.'.format(name))


