"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exaustive list of all available AWS regions.
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1)
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com)
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names,
     * cn.com.amazonaws.vpce for China regions,
     * com.amazonaws.vpce otherwise.
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     * @param service the service name (e.g: s3.amazonaws.com)
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
}
exports.RegionInfo = RegionInfo;
//# sourceMappingURL=data:application/json;base64,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