"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Provides default values for certain regional information points.
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix. This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param service   the name of the service (s3, s3.amazonaws.com, ...)
     * @param region    the region in which the service principal is needed.
     * @param urlSuffix the URL suffix for the partition in which the region is located.
     */
    static servicePrincipal(service, region, urlSuffix) {
        const matches = service.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
        if (!matches) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return service;
        }
        service = matches[1]; // Simplify the service name down to something like "s3"
        // Exceptions for Service Principals in us-iso-*
        const US_ISO_EXCEPTIONS = new Set([
            'cloudhsm',
            'config',
            'states',
            'workspaces',
        ]);
        // Exceptions for Service Principals in us-isob-*
        const US_ISOB_EXCEPTIONS = new Set([
            'dms',
            'states',
        ]);
        // Account for idiosyncratic Service Principals in `us-iso-*` regions
        if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
            switch (service) {
                // Services with universal principal
                case ('states'):
                    return `${service}.amazonaws.com`;
                // Services with a partitional principal
                default:
                    return `${service}.${urlSuffix}`;
            }
        }
        // Account for idiosyncratic Service Principals in `us-isob-*` regions
        if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
            switch (service) {
                // Services with universal principal
                case ('states'):
                    return `${service}.amazonaws.com`;
                // Services with a partitional principal
                default:
                    return `${service}.${urlSuffix}`;
            }
        }
        switch (service) {
            // Services with a regional AND partitional principal
            case 'codedeploy':
            case 'logs':
                return `${service}.${region}.${urlSuffix}`;
            // Services with a regional principal
            case 'states':
                return `${service}.${region}.amazonaws.com`;
            // Services with a partitional principal
            case 'ec2':
                return `${service}.${urlSuffix}`;
            // Services with a universal principal across all regions/partitions (the default case)
            default:
                return `${service}.amazonaws.com`;
        }
    }
}
exports.Default = Default;
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do
 * not have a synthesize-time region literal available (all you have is
 * `{ "Ref": "AWS::Region" }`)
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVmYXVsdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlZmF1bHQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQTs7R0FFRztBQUNILE1BQWEsT0FBTztJQStFaEIsZ0JBQXdCLENBQUM7SUF4RXpCOzs7Ozs7Ozs7T0FTRztJQUNJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFlLEVBQUUsTUFBYyxFQUFFLFNBQWlCO1FBQzdFLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUMsc0ZBQXNGLENBQUMsQ0FBQztRQUN0SCxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ1Ysa0VBQWtFO1lBQ2xFLE9BQU87WUFDUCxxQkFBcUI7WUFDckIsd0JBQXdCO1lBQ3hCLGtCQUFrQjtZQUNsQixxQkFBcUI7WUFDckIsT0FBTyxPQUFPLENBQUM7U0FDbEI7UUFDRCxPQUFPLEdBQUcsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsd0RBQXdEO1FBQzlFLGdEQUFnRDtRQUNoRCxNQUFNLGlCQUFpQixHQUFHLElBQUksR0FBRyxDQUFDO1lBQzlCLFVBQVU7WUFDVixRQUFRO1lBQ1IsUUFBUTtZQUNSLFlBQVk7U0FDZixDQUFDLENBQUM7UUFDSCxpREFBaUQ7UUFDakQsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsQ0FBQztZQUMvQixLQUFLO1lBQ0wsUUFBUTtTQUNYLENBQUMsQ0FBQztRQUNILHFFQUFxRTtRQUNyRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLElBQUksaUJBQWlCLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ2hFLFFBQVEsT0FBTyxFQUFFO2dCQUNiLG9DQUFvQztnQkFDcEMsS0FBSyxDQUFDLFFBQVEsQ0FBQztvQkFDWCxPQUFPLEdBQUcsT0FBTyxnQkFBZ0IsQ0FBQztnQkFDdEMsd0NBQXdDO2dCQUN4QztvQkFDSSxPQUFPLEdBQUcsT0FBTyxJQUFJLFNBQVMsRUFBRSxDQUFDO2FBQ3hDO1NBQ0o7UUFDRCxzRUFBc0U7UUFDdEUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNsRSxRQUFRLE9BQU8sRUFBRTtnQkFDYixvQ0FBb0M7Z0JBQ3BDLEtBQUssQ0FBQyxRQUFRLENBQUM7b0JBQ1gsT0FBTyxHQUFHLE9BQU8sZ0JBQWdCLENBQUM7Z0JBQ3RDLHdDQUF3QztnQkFDeEM7b0JBQ0ksT0FBTyxHQUFHLE9BQU8sSUFBSSxTQUFTLEVBQUUsQ0FBQzthQUN4QztTQUNKO1FBQ0QsUUFBUSxPQUFPLEVBQUU7WUFDYixxREFBcUQ7WUFDckQsS0FBSyxZQUFZLENBQUM7WUFDbEIsS0FBSyxNQUFNO2dCQUNQLE9BQU8sR0FBRyxPQUFPLElBQUksTUFBTSxJQUFJLFNBQVMsRUFBRSxDQUFDO1lBQy9DLHFDQUFxQztZQUNyQyxLQUFLLFFBQVE7Z0JBQ1QsT0FBTyxHQUFHLE9BQU8sSUFBSSxNQUFNLGdCQUFnQixDQUFDO1lBQ2hELHdDQUF3QztZQUN4QyxLQUFLLEtBQUs7Z0JBQ04sT0FBTyxHQUFHLE9BQU8sSUFBSSxTQUFTLEVBQUUsQ0FBQztZQUNyQyx1RkFBdUY7WUFDdkY7Z0JBQ0ksT0FBTyxHQUFHLE9BQU8sZ0JBQWdCLENBQUM7U0FDekM7SUFDTCxDQUFDOztBQTlFTCwwQkFnRkM7QUEvRUc7Ozs7R0FJRztBQUNvQix3Q0FBZ0MsR0FBRyxvQkFBb0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogUHJvdmlkZXMgZGVmYXVsdCB2YWx1ZXMgZm9yIGNlcnRhaW4gcmVnaW9uYWwgaW5mb3JtYXRpb24gcG9pbnRzLlxuICovXG5leHBvcnQgY2xhc3MgRGVmYXVsdCB7XG4gICAgLyoqXG4gICAgICogVGhlIGRlZmF1bHQgdmFsdWUgZm9yIGEgVlBDIEVuZHBvaW50IFNlcnZpY2UgbmFtZSBwcmVmaXgsIHVzZWZ1bCBpZiB5b3UgZG9cbiAgICAgKiBub3QgaGF2ZSBhIHN5bnRoZXNpemUtdGltZSByZWdpb24gbGl0ZXJhbCBhdmFpbGFibGUgKGFsbCB5b3UgaGF2ZSBpc1xuICAgICAqIGB7IFwiUmVmXCI6IFwiQVdTOjpSZWdpb25cIiB9YClcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZQQ19FTkRQT0lOVF9TRVJWSUNFX05BTUVfUFJFRklYID0gJ2NvbS5hbWF6b25hd3MudnBjZSc7XG4gICAgLyoqXG4gICAgICogQ29tcHV0ZXMgYSBcInN0YW5kYXJkXCIgQVdTIFNlcnZpY2UgcHJpbmNpcGFsIGZvciBhIGdpdmVuIHNlcnZpY2UsIHJlZ2lvbiBhbmQgc3VmZml4LiBUaGlzIGlzIHVzZWZ1bCBmb3IgZXhhbXBsZSB3aGVuXG4gICAgICogeW91IG5lZWQgdG8gY29tcHV0ZSBhIHNlcnZpY2UgcHJpbmNpcGFsIG5hbWUsIGJ1dCB5b3UgZG8gbm90IGhhdmUgYSBzeW50aGVzaXplLXRpbWUgcmVnaW9uIGxpdGVyYWwgYXZhaWxhYmxlIChzb1xuICAgICAqIGFsbCB5b3UgaGF2ZSBpcyBgeyBcIlJlZlwiOiBcIkFXUzo6UmVnaW9uXCIgfWApLiBUaGlzIHdheSB5b3UgZ2V0IHRoZSBzYW1lIGRlZmF1bHRpbmcgYmVoYXZpb3IgdGhhdCBpcyBub3JtYWxseSB1c2VkXG4gICAgICogZm9yIGJ1aWx0LWluIGRhdGEuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gc2VydmljZSAgIHRoZSBuYW1lIG9mIHRoZSBzZXJ2aWNlIChzMywgczMuYW1hem9uYXdzLmNvbSwgLi4uKVxuICAgICAqIEBwYXJhbSByZWdpb24gICAgdGhlIHJlZ2lvbiBpbiB3aGljaCB0aGUgc2VydmljZSBwcmluY2lwYWwgaXMgbmVlZGVkLlxuICAgICAqIEBwYXJhbSB1cmxTdWZmaXggdGhlIFVSTCBzdWZmaXggZm9yIHRoZSBwYXJ0aXRpb24gaW4gd2hpY2ggdGhlIHJlZ2lvbiBpcyBsb2NhdGVkLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgc2VydmljZVByaW5jaXBhbChzZXJ2aWNlOiBzdHJpbmcsIHJlZ2lvbjogc3RyaW5nLCB1cmxTdWZmaXg6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgICAgIGNvbnN0IG1hdGNoZXMgPSBzZXJ2aWNlLm1hdGNoKC9eKFteLl0rKSg/Oig/OlxcLmFtYXpvbmF3c1xcLmNvbSg/OlxcLmNuKT8pfCg/OlxcLmMyc1xcLmljXFwuZ292KXwoPzpcXC5zYzJzXFwuc2dvdlxcLmdvdikpPyQvKTtcbiAgICAgICAgaWYgKCFtYXRjaGVzKSB7XG4gICAgICAgICAgICAvLyBSZXR1cm4gXCJzZXJ2aWNlXCIgaWYgaXQgZG9lcyBub3QgbG9vayBsaWtlIGFueSBvZiB0aGUgZm9sbG93aW5nOlxuICAgICAgICAgICAgLy8gLSBzM1xuICAgICAgICAgICAgLy8gLSBzMy5hbWF6b25hd3MuY29tXG4gICAgICAgICAgICAvLyAtIHMzLmFtYXpvbmF3cy5jb20uY25cbiAgICAgICAgICAgIC8vIC0gczMuYzJzLmljLmdvdlxuICAgICAgICAgICAgLy8gLSBzMy5zYzJzLnNnb3YuZ292XG4gICAgICAgICAgICByZXR1cm4gc2VydmljZTtcbiAgICAgICAgfVxuICAgICAgICBzZXJ2aWNlID0gbWF0Y2hlc1sxXTsgLy8gU2ltcGxpZnkgdGhlIHNlcnZpY2UgbmFtZSBkb3duIHRvIHNvbWV0aGluZyBsaWtlIFwiczNcIlxuICAgICAgICAvLyBFeGNlcHRpb25zIGZvciBTZXJ2aWNlIFByaW5jaXBhbHMgaW4gdXMtaXNvLSpcbiAgICAgICAgY29uc3QgVVNfSVNPX0VYQ0VQVElPTlMgPSBuZXcgU2V0KFtcbiAgICAgICAgICAgICdjbG91ZGhzbScsXG4gICAgICAgICAgICAnY29uZmlnJyxcbiAgICAgICAgICAgICdzdGF0ZXMnLFxuICAgICAgICAgICAgJ3dvcmtzcGFjZXMnLFxuICAgICAgICBdKTtcbiAgICAgICAgLy8gRXhjZXB0aW9ucyBmb3IgU2VydmljZSBQcmluY2lwYWxzIGluIHVzLWlzb2ItKlxuICAgICAgICBjb25zdCBVU19JU09CX0VYQ0VQVElPTlMgPSBuZXcgU2V0KFtcbiAgICAgICAgICAgICdkbXMnLFxuICAgICAgICAgICAgJ3N0YXRlcycsXG4gICAgICAgIF0pO1xuICAgICAgICAvLyBBY2NvdW50IGZvciBpZGlvc3luY3JhdGljIFNlcnZpY2UgUHJpbmNpcGFscyBpbiBgdXMtaXNvLSpgIHJlZ2lvbnNcbiAgICAgICAgaWYgKHJlZ2lvbi5zdGFydHNXaXRoKCd1cy1pc28tJykgJiYgVVNfSVNPX0VYQ0VQVElPTlMuaGFzKHNlcnZpY2UpKSB7XG4gICAgICAgICAgICBzd2l0Y2ggKHNlcnZpY2UpIHtcbiAgICAgICAgICAgICAgICAvLyBTZXJ2aWNlcyB3aXRoIHVuaXZlcnNhbCBwcmluY2lwYWxcbiAgICAgICAgICAgICAgICBjYXNlICgnc3RhdGVzJyk6XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBgJHtzZXJ2aWNlfS5hbWF6b25hd3MuY29tYDtcbiAgICAgICAgICAgICAgICAvLyBTZXJ2aWNlcyB3aXRoIGEgcGFydGl0aW9uYWwgcHJpbmNpcGFsXG4gICAgICAgICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIGAke3NlcnZpY2V9LiR7dXJsU3VmZml4fWA7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgLy8gQWNjb3VudCBmb3IgaWRpb3N5bmNyYXRpYyBTZXJ2aWNlIFByaW5jaXBhbHMgaW4gYHVzLWlzb2ItKmAgcmVnaW9uc1xuICAgICAgICBpZiAocmVnaW9uLnN0YXJ0c1dpdGgoJ3VzLWlzb2ItJykgJiYgVVNfSVNPQl9FWENFUFRJT05TLmhhcyhzZXJ2aWNlKSkge1xuICAgICAgICAgICAgc3dpdGNoIChzZXJ2aWNlKSB7XG4gICAgICAgICAgICAgICAgLy8gU2VydmljZXMgd2l0aCB1bml2ZXJzYWwgcHJpbmNpcGFsXG4gICAgICAgICAgICAgICAgY2FzZSAoJ3N0YXRlcycpOlxuICAgICAgICAgICAgICAgICAgICByZXR1cm4gYCR7c2VydmljZX0uYW1hem9uYXdzLmNvbWA7XG4gICAgICAgICAgICAgICAgLy8gU2VydmljZXMgd2l0aCBhIHBhcnRpdGlvbmFsIHByaW5jaXBhbFxuICAgICAgICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBgJHtzZXJ2aWNlfS4ke3VybFN1ZmZpeH1gO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHN3aXRjaCAoc2VydmljZSkge1xuICAgICAgICAgICAgLy8gU2VydmljZXMgd2l0aCBhIHJlZ2lvbmFsIEFORCBwYXJ0aXRpb25hbCBwcmluY2lwYWxcbiAgICAgICAgICAgIGNhc2UgJ2NvZGVkZXBsb3knOlxuICAgICAgICAgICAgY2FzZSAnbG9ncyc6XG4gICAgICAgICAgICAgICAgcmV0dXJuIGAke3NlcnZpY2V9LiR7cmVnaW9ufS4ke3VybFN1ZmZpeH1gO1xuICAgICAgICAgICAgLy8gU2VydmljZXMgd2l0aCBhIHJlZ2lvbmFsIHByaW5jaXBhbFxuICAgICAgICAgICAgY2FzZSAnc3RhdGVzJzpcbiAgICAgICAgICAgICAgICByZXR1cm4gYCR7c2VydmljZX0uJHtyZWdpb259LmFtYXpvbmF3cy5jb21gO1xuICAgICAgICAgICAgLy8gU2VydmljZXMgd2l0aCBhIHBhcnRpdGlvbmFsIHByaW5jaXBhbFxuICAgICAgICAgICAgY2FzZSAnZWMyJzpcbiAgICAgICAgICAgICAgICByZXR1cm4gYCR7c2VydmljZX0uJHt1cmxTdWZmaXh9YDtcbiAgICAgICAgICAgIC8vIFNlcnZpY2VzIHdpdGggYSB1bml2ZXJzYWwgcHJpbmNpcGFsIGFjcm9zcyBhbGwgcmVnaW9ucy9wYXJ0aXRpb25zICh0aGUgZGVmYXVsdCBjYXNlKVxuICAgICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgICAgICByZXR1cm4gYCR7c2VydmljZX0uYW1hem9uYXdzLmNvbWA7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcigpIHsgfVxufVxuIl19