"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => e.dependencies.every(d => !remaining.has(d)));
        selectable.sort((a, b) => a.key < b.key ? -1 : b.key < a.key ? 1 : 0);
        for (const selected of selectable) {
            ret.push(selected.element);
            remaining.delete(selected.key);
        }
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(', ')}`);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,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