"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        switch (artifact.type) {
            case cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK:
                return new cloudformation_artifact_1.CloudFormationStackArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.CDK_TREE:
                return new tree_cloud_artifact_1.TreeCloudArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.ASSET_MANIFEST:
                return new asset_manifest_artifact_1.AssetManifestArtifact(assembly, id, artifact);
            default:
                return undefined;
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.artifacts.find(a => a.id === id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
}
exports.CloudArtifact = CloudArtifact;
// needs to be defined at the end to avoid a cyclic dependency
const asset_manifest_artifact_1 = require("./asset-manifest-artifact");
const cloudformation_artifact_1 = require("./cloudformation-artifact");
const tree_cloud_artifact_1 = require("./tree-cloud-artifact");
//# sourceMappingURL=data:application/json;base64,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