"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 */
class WaiterStateMachine extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const role = new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.isCompleteHandler.functionArn],
        }));
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.timeoutHandler.functionArn],
        }));
        const definition = core_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: props.interval.toSeconds(),
                            MaxAttempts: props.maxAttempts,
                            BackoffRate: props.backoffRate,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: props.isCompleteHandler.functionArn,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: props.timeoutHandler.functionArn,
                },
            },
        });
        const resource = new core_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
    }
    grantStartExecution(identity) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
//# sourceMappingURL=data:application/json;base64,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