"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("./token");
/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 */
class Size {
    constructor(amount, unit) {
        if (!token_1.Token.isUnresolved(amount) && amount < 0) {
            throw new Error(`Storage amounts cannot be negative. Received: ${amount}`);
        }
        this.amount = amount;
        this.unit = unit;
    }
    /**
     * Create a Storage representing an amount kibibytes.
     * 1 KiB = 1024 bytes
     */
    static kibibytes(amount) {
        return new Size(amount, StorageUnit.Kibibytes);
    }
    /**
     * Create a Storage representing an amount mebibytes.
     * 1 MiB = 1024 KiB
     */
    static mebibytes(amount) {
        return new Size(amount, StorageUnit.Mebibytes);
    }
    /**
     * Create a Storage representing an amount gibibytes.
     * 1 GiB = 1024 MiB
     */
    static gibibytes(amount) {
        return new Size(amount, StorageUnit.Gibibytes);
    }
    /**
     * Create a Storage representing an amount tebibytes.
     * 1 TiB = 1024 GiB
     */
    static tebibytes(amount) {
        return new Size(amount, StorageUnit.Tebibytes);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     * 1 PiB = 1024 TiB
     */
    static pebibyte(amount) {
        return new Size(amount, StorageUnit.Pebibytes);
    }
    /**
     * Return this storage as a total number of kibibytes.
     */
    toKibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Kibibytes, opts);
    }
    /**
     * Return this storage as a total number of mebibytes.
     */
    toMebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Mebibytes, opts);
    }
    /**
     * Return this storage as a total number of gibibytes.
     */
    toGibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Gibibytes, opts);
    }
    /**
     * Return this storage as a total number of tebibytes.
     */
    toTebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Tebibytes, opts);
    }
    /**
     * Return this storage as a total number of pebibytes.
     */
    toPebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Pebibytes, opts);
    }
}
exports.Size = Size;
/**
 * Rounding behaviour when converting between units of `Size`.
 */
var SizeRoundingBehavior;
(function (SizeRoundingBehavior) {
    /** Fail the conversion if the result is not an integer. */
    SizeRoundingBehavior[SizeRoundingBehavior["FAIL"] = 0] = "FAIL";
    /** If the result is not an integer, round it to the closest integer less than the result */
    SizeRoundingBehavior[SizeRoundingBehavior["FLOOR"] = 1] = "FLOOR";
    /** Don't round. Return even if the result is a fraction. */
    SizeRoundingBehavior[SizeRoundingBehavior["NONE"] = 2] = "NONE";
})(SizeRoundingBehavior = exports.SizeRoundingBehavior || (exports.SizeRoundingBehavior = {}));
class StorageUnit {
    constructor(label, inKibiBytes) {
        this.label = label;
        this.inKibiBytes = inKibiBytes;
        // MAX_SAFE_INTEGER is 2^53, so by representing storage in kibibytes,
        // the highest storage we can represent is 8 exbibytes.
    }
    toString() {
        return this.label;
    }
}
StorageUnit.Kibibytes = new StorageUnit('kibibytes', 1);
StorageUnit.Mebibytes = new StorageUnit('mebibytes', 1024);
StorageUnit.Gibibytes = new StorageUnit('gibibytes', 1024 * 1024);
StorageUnit.Tebibytes = new StorageUnit('tebibytes', 1024 * 1024 * 1024);
StorageUnit.Pebibytes = new StorageUnit('pebibytes', 1024 * 1024 * 1024 * 1024);
function convert(amount, fromUnit, toUnit, options = {}) {
    var _a;
    const rounding = (_a = options.rounding) !== null && _a !== void 0 ? _a : SizeRoundingBehavior.FAIL;
    if (fromUnit.inKibiBytes === toUnit.inKibiBytes) {
        return amount;
    }
    if (token_1.Token.isUnresolved(amount)) {
        throw new Error(`Unable to perform time unit conversion on un-resolved token ${amount}.`);
    }
    const multiplier = fromUnit.inKibiBytes / toUnit.inKibiBytes;
    const value = amount * multiplier;
    switch (rounding) {
        case SizeRoundingBehavior.NONE:
            return value;
        case SizeRoundingBehavior.FLOOR:
            return Math.floor(value);
        default:
        case SizeRoundingBehavior.FAIL:
            if (!Number.isInteger(value)) {
                throw new Error(`'${amount} ${fromUnit}' cannot be converted into a whole number of ${toUnit}.`);
            }
            return value;
    }
}
//# sourceMappingURL=data:application/json;base64,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