"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope) {
        throw new Error('prepareApp must be called on the root node');
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare it's template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (stack_1.Stack.isStack(stack) && stack.nested) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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