"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const assets_1 = require("./assets");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cloudformation_generated_1 = require("./cloudformation.generated");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const stack_1 = require("./stack");
const stack_synthesizers_1 = require("./stack-synthesizers");
const token_1 = require("./token");
const NESTED_STACK_SYMBOL = Symbol.for('@aws-cdk/core.NestedStack');
/**
 * A CloudFormation nested stack.
 *
 * When you apply template changes to update a top-level stack, CloudFormation
 * updates the top-level stack and initiates an update to its nested stacks.
 * CloudFormation updates the resources of modified nested stacks, but does not
 * update the resources of unmodified nested stacks.
 *
 * Furthermore, this stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 * Cross references of resource attributes between the parent stack and the
 * nested stack will automatically be translated to stack parameters and
 * outputs.
 *
 * @experimental
 */
class NestedStack extends stack_1.Stack {
    constructor(scope, id, props = {}) {
        const parentStack = findParentStack(scope);
        super(scope, id, {
            env: { account: parentStack.account, region: parentStack.region },
            synthesizer: new stack_synthesizers_1.NestedStackSynthesizer(parentStack.synthesizer),
        });
        this._parentStack = parentStack;
        // @deprecate: remove this in v2.0 (redundent)
        const parentScope = new construct_compat_1.Construct(scope, id + '.NestedStack');
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${this.node.uniqueId}.nested.template.json`;
        this.parameters = props.parameters || {};
        this.resource = new cloudformation_generated_1.CfnStack(parentScope, `${id}.NestedStackResource`, {
            templateUrl: lazy_1.Lazy.stringValue({ produce: () => this._templateUrl || '<unresolved>' }),
            parameters: lazy_1.Lazy.anyValue({ produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined }),
            notificationArns: props.notificationArns,
            timeoutInMinutes: props.timeout ? props.timeout.toMinutes() : undefined,
        });
        this.nestedStackResource = this.resource;
        // context-aware stack name: if resolved from within this stack, return AWS::StackName
        // if resolved from the outer stack, use the { Ref } of the AWS::CloudFormation::Stack resource
        // which resolves the ARN of the stack. We need to extract the stack name, which is the second
        // component after splitting by "/"
        this._contextualStackName = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_NAME, cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split('/', this.resource.ref)));
        this._contextualStackId = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_ID, this.resource.ref);
    }
    /**
     * Checks if `x` is an object of type `NestedStack`.
     */
    static isNestedStack(x) {
        return x != null && typeof (x) === 'object' && NESTED_STACK_SYMBOL in x;
    }
    /**
     * An attribute that represents the name of the nested stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return a token that parses the name from the stack ID.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackName" }`
     *
     * @attribute
     * @example mystack-mynestedstack-sggfrhxhum7w
     */
    get stackName() {
        return this._contextualStackName;
    }
    /**
     * An attribute that represents the ID of the stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return `{ "Ref": "LogicalIdOfNestedStackResource" }`.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackId" }`
     *
     * @attribute
     * @example "arn:aws:cloudformation:us-east-2:123456789012:stack/mystack-mynestedstack-sggfrhxhum7w/f449b250-b969-11e0-a185-5081d0136786"
     */
    get stackId() {
        return this._contextualStackId;
    }
    /**
     * Assign a value to one of the nested stack parameters.
     * @param name The parameter name (ID)
     * @param value The value to assign
     */
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    /**
     * Defines an asset at the parent stack which represents the template of this
     * nested stack.
     *
     * This private API is used by `App.prepare()` within a loop that rectifies
     * references every time an asset is added. This is because (at the moment)
     * assets are addressed using CloudFormation parameters.
     *
     * @returns `true` if a new asset was added or `false` if an asset was
     * previously added. When this returns `true`, App will do another reference
     * rectification cycle.
     *
     * @internal
     */
    _prepareTemplateAsset() {
        if (this._templateUrl) {
            return false;
        }
        const cfn = JSON.stringify(this._toCloudFormation());
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        const templateLocation = this._parentStack.addFileAsset({
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        });
        // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
        // be resolved as cross-stack references like any other (see "multi" tests).
        this._templateUrl = `https://s3.${this._parentStack.region}.${this._parentStack.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        return true;
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
}
exports.NestedStack = NestedStack;
/**
 * Validates the scope for a nested stack. Nested stacks must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    if (!scope) {
        throw new Error('Nested stacks cannot be defined as a root construct');
    }
    const parentStack = scope.node.scopes.reverse().find(p => stack_1.Stack.isStack(p));
    if (!parentStack) {
        throw new Error('Nested stacks must be defined within scope of another non-nested stack');
    }
    return parentStack;
}
//# sourceMappingURL=data:application/json;base64,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