"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGF6eS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxhenkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSwrQ0FBa0Q7QUFDbEQsbUNBQWdDO0FBa0ZoQzs7Ozs7R0FLRztBQUNILE1BQWEsSUFBSTtJQUNOLE1BQU0sQ0FBQyxXQUFXLENBQUMsUUFBeUIsRUFBRSxVQUFrQyxFQUFFO1FBQ3JGLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBQ00sTUFBTSxDQUFDLFdBQVcsQ0FBQyxRQUF5QjtRQUMvQyxPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBQ00sTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUF1QixFQUFFLFVBQWdDLEVBQUU7UUFDL0UsT0FBTyxhQUFLLENBQUMsTUFBTSxDQUFDLElBQUksUUFBUSxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBQ00sTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFzQixFQUFFLFVBQStCLEVBQUU7UUFDNUUsT0FBTyxJQUFJLE9BQU8sQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUNEO0lBQ0EsQ0FBQztDQUNKO0FBZkQsb0JBZUM7QUFDRCxNQUFlLFFBQVE7SUFFbkI7UUFDSSxJQUFJLENBQUMsYUFBYSxHQUFHLCtCQUFpQixFQUFFLENBQUM7SUFDN0MsQ0FBQztJQUVNLFFBQVE7UUFDWCxPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxNQUFNO1FBQ1QsT0FBTyxtQkFBbUIsQ0FBQztJQUMvQixDQUFDO0NBQ0o7QUFDRCxNQUFNLFVBQVcsU0FBUSxRQUFRO0lBQzdCLFlBQTZCLFFBQXlCO1FBQ2xELEtBQUssRUFBRSxDQUFDO1FBRGlCLGFBQVEsR0FBUixRQUFRLENBQWlCO0lBRXRELENBQUM7SUFDTSxPQUFPLENBQUMsT0FBd0I7UUFDbkMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUMxQyxDQUFDO0NBQ0o7QUFDRCxNQUFNLFVBQVcsU0FBUSxRQUFRO0lBQzdCLFlBQTZCLFFBQXlCO1FBQ2xELEtBQUssRUFBRSxDQUFDO1FBRGlCLGFBQVEsR0FBUixRQUFRLENBQWlCO0lBRXRELENBQUM7SUFDTSxPQUFPLENBQUMsT0FBd0I7UUFDbkMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUMxQyxDQUFDO0NBQ0o7QUFDRCxNQUFNLFFBQVMsU0FBUSxRQUFRO0lBQzNCLFlBQTZCLFFBQXVCLEVBQW1CLFVBQWdDLEVBQUU7UUFDckcsS0FBSyxFQUFFLENBQUM7UUFEaUIsYUFBUSxHQUFSLFFBQVEsQ0FBZTtRQUFtQixZQUFPLEdBQVAsT0FBTyxDQUEyQjtJQUV6RyxDQUFDO0lBQ00sT0FBTyxDQUFDLE9BQXdCO1FBQ25DLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzNDLElBQUksR0FBRyxLQUFLLFNBQVMsSUFBSSxHQUFHLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRTtZQUNqRSxPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztDQUNKO0FBQ0QsTUFBTSxPQUFRLFNBQVEsUUFBUTtJQUMxQixZQUE2QixRQUFzQixFQUFtQixVQUErQixFQUFFO1FBQ25HLEtBQUssRUFBRSxDQUFDO1FBRGlCLGFBQVEsR0FBUixRQUFRLENBQWM7UUFBbUIsWUFBTyxHQUFQLE9BQU8sQ0FBMEI7SUFFdkcsQ0FBQztJQUNNLE9BQU8sQ0FBQyxPQUF3QjtRQUNuQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMzQyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksR0FBRyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUU7WUFDdkUsT0FBTyxTQUFTLENBQUM7U0FDcEI7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7Q0FDSiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElSZXNvbHZhYmxlLCBJUmVzb2x2ZUNvbnRleHQgfSBmcm9tICcuL3Jlc29sdmFibGUnO1xuaW1wb3J0IHsgY2FwdHVyZVN0YWNrVHJhY2UgfSBmcm9tICcuL3N0YWNrLXRyYWNlJztcbmltcG9ydCB7IFRva2VuIH0gZnJvbSAnLi90b2tlbic7XG4vKipcbiAqIEludGVyZmFjZSBmb3IgbGF6eSBzdHJpbmcgcHJvZHVjZXJzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVN0cmluZ1Byb2R1Y2VyIHtcbiAgICAvKipcbiAgICAgKiBQcm9kdWNlIHRoZSBzdHJpbmcgdmFsdWVcbiAgICAgKi9cbiAgICBwcm9kdWNlKGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IHN0cmluZyB8IHVuZGVmaW5lZDtcbn1cbi8qKlxuICogSW50ZXJmYWNlIGZvciBsYXp5IGxpc3QgcHJvZHVjZXJzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUxpc3RQcm9kdWNlciB7XG4gICAgLyoqXG4gICAgICogUHJvZHVjZSB0aGUgbGlzdCB2YWx1ZVxuICAgICAqL1xuICAgIHByb2R1Y2UoY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG59XG4vKipcbiAqIEludGVyZmFjZSBmb3IgbGF6eSBudW1iZXIgcHJvZHVjZXJzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU51bWJlclByb2R1Y2VyIHtcbiAgICAvKipcbiAgICAgKiBQcm9kdWNlIHRoZSBudW1iZXIgdmFsdWVcbiAgICAgKi9cbiAgICBwcm9kdWNlKGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IG51bWJlciB8IHVuZGVmaW5lZDtcbn1cbi8qKlxuICogSW50ZXJmYWNlIGZvciBsYXp5IHVudHlwZWQgdmFsdWUgcHJvZHVjZXJzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUFueVByb2R1Y2VyIHtcbiAgICAvKipcbiAgICAgKiBQcm9kdWNlIHRoZSB2YWx1ZVxuICAgICAqL1xuICAgIHByb2R1Y2UoY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogYW55O1xufVxuLyoqXG4gKiBPcHRpb25zIGZvciBjcmVhdGluZyBhIGxhenkgc3RyaW5nIHRva2VuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGF6eVN0cmluZ1ZhbHVlT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVXNlIHRoZSBnaXZlbiBuYW1lIGFzIGEgZGlzcGxheSBoaW50XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGhpbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBkaXNwbGF5SGludD86IHN0cmluZztcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgY3JlYXRpbmcgYSBsYXp5IGxpc3QgdG9rZW5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXp5TGlzdFZhbHVlT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVXNlIHRoZSBnaXZlbiBuYW1lIGFzIGEgZGlzcGxheSBoaW50XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGhpbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBkaXNwbGF5SGludD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBJZiB0aGUgcHJvZHVjZWQgbGlzdCBpcyBlbXB0eSwgcmV0dXJuICd1bmRlZmluZWQnIGluc3RlYWRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgb21pdEVtcHR5PzogYm9vbGVhbjtcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgY3JlYXRpbmcgbGF6eSB1bnR5cGVkIHRva2Vuc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIExhenlBbnlWYWx1ZU9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFVzZSB0aGUgZ2l2ZW4gbmFtZSBhcyBhIGRpc3BsYXkgaGludFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBoaW50XG4gICAgICovXG4gICAgcmVhZG9ubHkgZGlzcGxheUhpbnQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogSWYgdGhlIHByb2R1Y2VkIHZhbHVlIGlzIGFuIGFycmF5IGFuZCBpdCBpcyBlbXB0eSwgcmV0dXJuICd1bmRlZmluZWQnIGluc3RlYWRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgb21pdEVtcHR5QXJyYXk/OiBib29sZWFuO1xufVxuLyoqXG4gKiBMYXppbHkgcHJvZHVjZSBhIHZhbHVlXG4gKlxuICogQ2FuIGJlIHVzZWQgdG8gcmV0dXJuIGEgc3RyaW5nLCBsaXN0IG9yIG51bWVyaWMgdmFsdWUgd2hvc2UgYWN0dWFsIHZhbHVlXG4gKiB3aWxsIG9ubHkgYmUgY2FsY3VsYXRlZCBsYXRlciwgZHVyaW5nIHN5bnRoZXNpcy5cbiAqL1xuZXhwb3J0IGNsYXNzIExhenkge1xuICAgIHB1YmxpYyBzdGF0aWMgc3RyaW5nVmFsdWUocHJvZHVjZXI6IElTdHJpbmdQcm9kdWNlciwgb3B0aW9uczogTGF6eVN0cmluZ1ZhbHVlT3B0aW9ucyA9IHt9KSB7XG4gICAgICAgIHJldHVybiBUb2tlbi5hc1N0cmluZyhuZXcgTGF6eVN0cmluZyhwcm9kdWNlciksIG9wdGlvbnMpO1xuICAgIH1cbiAgICBwdWJsaWMgc3RhdGljIG51bWJlclZhbHVlKHByb2R1Y2VyOiBJTnVtYmVyUHJvZHVjZXIpIHtcbiAgICAgICAgcmV0dXJuIFRva2VuLmFzTnVtYmVyKG5ldyBMYXp5TnVtYmVyKHByb2R1Y2VyKSk7XG4gICAgfVxuICAgIHB1YmxpYyBzdGF0aWMgbGlzdFZhbHVlKHByb2R1Y2VyOiBJTGlzdFByb2R1Y2VyLCBvcHRpb25zOiBMYXp5TGlzdFZhbHVlT3B0aW9ucyA9IHt9KSB7XG4gICAgICAgIHJldHVybiBUb2tlbi5hc0xpc3QobmV3IExhenlMaXN0KHByb2R1Y2VyLCBvcHRpb25zKSwgb3B0aW9ucyk7XG4gICAgfVxuICAgIHB1YmxpYyBzdGF0aWMgYW55VmFsdWUocHJvZHVjZXI6IElBbnlQcm9kdWNlciwgb3B0aW9uczogTGF6eUFueVZhbHVlT3B0aW9ucyA9IHt9KTogSVJlc29sdmFibGUge1xuICAgICAgICByZXR1cm4gbmV3IExhenlBbnkocHJvZHVjZXIsIG9wdGlvbnMpO1xuICAgIH1cbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKCkge1xuICAgIH1cbn1cbmFic3RyYWN0IGNsYXNzIExhenlCYXNlIGltcGxlbWVudHMgSVJlc29sdmFibGUge1xuICAgIHB1YmxpYyByZWFkb25seSBjcmVhdGlvblN0YWNrOiBzdHJpbmdbXTtcbiAgICBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgdGhpcy5jcmVhdGlvblN0YWNrID0gY2FwdHVyZVN0YWNrVHJhY2UoKTtcbiAgICB9XG4gICAgcHVibGljIGFic3RyYWN0IHJlc29sdmUoY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogYW55O1xuICAgIHB1YmxpYyB0b1N0cmluZygpIHtcbiAgICAgICAgcmV0dXJuIFRva2VuLmFzU3RyaW5nKHRoaXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUdXJuIHRoaXMgVG9rZW4gaW50byBKU09OXG4gICAgICpcbiAgICAgKiBDYWxsZWQgYXV0b21hdGljYWxseSB3aGVuIEpTT04uc3RyaW5naWZ5KCkgaXMgY2FsbGVkIG9uIGEgVG9rZW4uXG4gICAgICovXG4gICAgcHVibGljIHRvSlNPTigpOiBhbnkge1xuICAgICAgICByZXR1cm4gJzx1bnJlc29sdmVkLWxhenk+JztcbiAgICB9XG59XG5jbGFzcyBMYXp5U3RyaW5nIGV4dGVuZHMgTGF6eUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvZHVjZXI6IElTdHJpbmdQcm9kdWNlcikge1xuICAgICAgICBzdXBlcigpO1xuICAgIH1cbiAgICBwdWJsaWMgcmVzb2x2ZShjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMucHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgICB9XG59XG5jbGFzcyBMYXp5TnVtYmVyIGV4dGVuZHMgTGF6eUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvZHVjZXI6IElOdW1iZXJQcm9kdWNlcikge1xuICAgICAgICBzdXBlcigpO1xuICAgIH1cbiAgICBwdWJsaWMgcmVzb2x2ZShjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMucHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgICB9XG59XG5jbGFzcyBMYXp5TGlzdCBleHRlbmRzIExhenlCYXNlIHtcbiAgICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb2R1Y2VyOiBJTGlzdFByb2R1Y2VyLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IExhenlMaXN0VmFsdWVPcHRpb25zID0ge30pIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICB9XG4gICAgcHVibGljIHJlc29sdmUoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMucHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgICAgICAgaWYgKHJldCAhPT0gdW5kZWZpbmVkICYmIHJldC5sZW5ndGggPT09IDAgJiYgdGhpcy5vcHRpb25zLm9taXRFbXB0eSkge1xuICAgICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbn1cbmNsYXNzIExhenlBbnkgZXh0ZW5kcyBMYXp5QmFzZSB7XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9kdWNlcjogSUFueVByb2R1Y2VyLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IExhenlBbnlWYWx1ZU9wdGlvbnMgPSB7fSkge1xuICAgICAgICBzdXBlcigpO1xuICAgIH1cbiAgICBwdWJsaWMgcmVzb2x2ZShjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpIHtcbiAgICAgICAgY29uc3QgcmV0ID0gdGhpcy5wcm9kdWNlci5wcm9kdWNlKGNvbnRleHQpO1xuICAgICAgICBpZiAoQXJyYXkuaXNBcnJheShyZXQpICYmIHJldC5sZW5ndGggPT09IDAgJiYgdGhpcy5vcHRpb25zLm9taXRFbXB0eUFycmF5KSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgfVxufVxuIl19