"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stack_1 = require("./stack");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a CloudFormation
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the AWS::CloudFormation::Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define an assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}': ${reason}`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element) ? element.nestedStackResource : element;
        if (!resource) {
            throw new Error('assertion failure'); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
//# sourceMappingURL=data:application/json;base64,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