"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = scope.node.tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                scope.node.addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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