"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
// import required to be here, otherwise causes a cycle when running the generated JavaScript
// tslint:disable-next-line:ordered-imports
const cfn_element_1 = require("./cfn-element");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const deps_1 = require("./deps");
const cfn_reference_1 = require("./private/cfn-reference");
const removal_policy_1 = require("./removal-policy");
const tag_manager_1 = require("./tag-manager");
const util_1 = require("./util");
/**
 * Represents a CloudFormation resource.
 */
class CfnResource extends cfn_element_1.CfnRefElement {
    /**
     * Creates a resource construct.
     * @param cfnResourceType The CloudFormation type of this resource (e.g. AWS::DynamoDB::Table)
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("CFN") resources (such as `s3.CfnBucket`). These resources will have a
        // property for each CloudFormation property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `cfnXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         */
        this.cfnOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        if (!props.type) {
            throw new Error('The `type` property is required');
        }
        this.cfnResourceType = props.type;
        this._cfnProperties = props.properties || {};
        // if aws:cdk:enable-path-metadata is set, embed the current construct's
        // path in the CloudFormation template, so it will be possible to trace
        // back to the actual construct path.
        if (this.node.tryGetContext(cxapi.PATH_METADATA_ENABLE_CONTEXT)) {
            this.cfnOptions.metadata = {
                [cxapi.PATH_METADATA_KEY]: this.node.path,
            };
        }
    }
    /**
     * Check whether the given construct is a CfnResource
     */
    static isCfnResource(construct) {
        return construct.cfnResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     */
    applyRemovalPolicy(policy, options = {}) {
        policy = policy || options.default || removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
                break;
            case removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
                break;
            case removal_policy_1.RemovalPolicy.SNAPSHOT:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.cfnOptions.deletionPolicy = deletionPolicy;
        if (options.applyToUpdateReplacePolicy !== false) {
            this.cfnOptions.updateReplacePolicy = deletionPolicy;
        }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     * @param attributeName The name of the attribute.
     */
    getAtt(attributeName) {
        return cfn_reference_1.CfnReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized CloudFormation resource. To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * For example,
     * ```typescript
     * addOverride('Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes', ['myattribute'])
     * addOverride('Properties.GlobalSecondaryIndexes.1.ProjectionType', 'INCLUDE')
     * ```
     * would add the overrides
     * ```json
     * "Properties": {
     *   "GlobalSecondaryIndexes": [
     *     {
     *       "Projection": {
     *         "NonKeyAttributes": [ "myattribute" ]
     *         ...
     *       }
     *       ...
     *     },
     *     {
     *       "ProjectionType": "INCLUDE"
     *       ...
     *     },
     *   ]
     *   ...
     * }
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermdediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = path.split('.');
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null && typeof (curr[key]) === 'object' && !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property
     * @param value The value
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     * @param propertyPath The path to the property.
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be
     * provisioned unless the other resource has been successfully provisioned.
     *
     * This can be used for resources across stacks (or nested stack) boundaries
     * and the dependency will automatically be transferred to the relevant scope.
     */
    addDependsOn(target) {
        deps_1.addDependency(this, target, `"${this.node.path}" depends on "${target.node.path}"`);
    }
    /**
     * @returns a string representation of this resource
     */
    toString() {
        return `${super.toString()} [${this.cfnResourceType}]`;
    }
    /**
     * Called by the `addDependency` helper function in order to realize a direct
     * dependency between two resources that are directly defined in the same
     * stacks.
     *
     * Use `resource.addDependsOn` to define the dependency between two resources,
     * which also takes stack boundaries into account.
     *
     * @internal
     */
    _addResourceDependency(target) {
        this.dependsOn.add(target);
    }
    /**
     * Emits CloudFormation for this resource.
     * @internal
     */
    _toCloudFormation() {
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.logicalId]: new util_1.PostResolveToken({
                        Type: this.cfnResourceType,
                        Properties: util_1.ignoreEmpty(this.cfnProperties),
                        DependsOn: util_1.ignoreEmpty(renderDependsOn(this.dependsOn)),
                        CreationPolicy: util_1.capitalizePropertyNames(this, renderCreationPolicy(this.cfnOptions.creationPolicy)),
                        UpdatePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updatePolicy),
                        UpdateReplacePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.deletionPolicy),
                        Metadata: util_1.ignoreEmpty(this.cfnOptions.metadata),
                        Condition: this.cfnOptions.condition && this.cfnOptions.condition.logicalId,
                    }, props => {
                        const renderedProps = this.renderProperties(props.Properties || {});
                        props.Properties = renderedProps && (Object.values(renderedProps).find(v => !!v) ? renderedProps : undefined);
                        return deepMerge(props, this.rawOverrides);
                    }),
                },
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ['--- resource created at ---', ...trace].join('\n  at ');
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        // returns the set of logical ID (tokens) this resource depends on
        // sorted by construct paths to ensure test determinism
        function renderDependsOn(dependsOn) {
            return Array
                .from(dependsOn)
                .sort((x, y) => x.node.path.localeCompare(y.node.path))
                .map(r => r.logicalId);
        }
        function renderCreationPolicy(policy) {
            if (!policy) {
                return undefined;
            }
            const result = { ...policy };
            if (policy.resourceSignal && policy.resourceSignal.timeout) {
                result.resourceSignal = policy.resourceSignal;
            }
            return result;
        }
    }
    get cfnProperties() {
        const props = this._cfnProperties || {};
        if (tag_manager_1.TagManager.isTaggable(this)) {
            const tagsProp = {};
            tagsProp[this.tags.tagPropertyName] = this.tags.renderTags();
            return deepMerge(props, tagsProp);
        }
        return props;
    }
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     */
    get updatedProperites() {
        return this._cfnProperties;
    }
    validateProperties(_properties) {
        // Nothing
    }
}
exports.CfnResource = CfnResource;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof (source) !== 'object' || typeof (target) !== 'object') {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof (value) === 'object' && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
//# sourceMappingURL=data:application/json;base64,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