"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const token_1 = require("./token");
/**
 * This class contains functions for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class FromCloudFormation {
    static parseValue(cfnValue) {
        return parseCfnValueToCdkValue(cfnValue);
    }
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(value, this.getString);
    }
    static getArray(value, mapper) {
        if (!Array.isArray(value)) {
            // break the type system, and just return the given value,
            // which hopefully will be reported as invalid by the validator
            // of the property we're transforming
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        return value.map(mapper);
    }
    static getMap(value, mapper) {
        if (typeof value !== 'object') {
            // if the input is not a map (= object in JS land),
            // just return it, and let the validator of this property handle it
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        const ret = {};
        for (const [key, val] of Object.entries(value)) {
            ret[key] = mapper(val);
        }
        return ret;
    }
    static parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
}
exports.FromCloudFormation = FromCloudFormation;
function parseCfnValueToCdkValue(cfnValue) {
    // == null captures undefined as well
    if (cfnValue == null) {
        return undefined;
    }
    // if we have any late-bound values,
    // just return them
    if (token_1.isResolvableObject(cfnValue)) {
        return cfnValue;
    }
    if (Array.isArray(cfnValue)) {
        return cfnValue.map(el => parseCfnValueToCdkValue(el));
    }
    if (typeof cfnValue === 'object') {
        // an object can be either a CFN intrinsic, or an actual object
        const cfnIntrinsic = parseIfCfnIntrinsic(cfnValue);
        if (cfnIntrinsic) {
            return cfnIntrinsic;
        }
        const ret = {};
        for (const [key, val] of Object.entries(cfnValue)) {
            ret[key] = parseCfnValueToCdkValue(val);
        }
        return ret;
    }
    // in all other cases, just return the input
    return cfnValue;
}
function parseIfCfnIntrinsic(object) {
    var _a;
    const key = looksLikeCfnIntrinsic(object);
    switch (key) {
        case undefined:
            return undefined;
        case 'Ref': {
            // ToDo handle translating logical IDs
            return (_a = specialCaseRefs(object[key])) !== null && _a !== void 0 ? _a : cfn_fn_1.Fn._ref(object[key]);
        }
        case 'Fn::GetAtt': {
            // Fn::GetAtt takes a 2-element list as its argument
            const value = object[key];
            // ToDo same comment here as in Ref above
            return cfn_fn_1.Fn.getAtt((value[0]), value[1]);
        }
        case 'Fn::Join': {
            // Fn::Join takes a 2-element list as its argument,
            // where the first element is the delimiter,
            // and the second is the list of elements to join
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.join(value[0], value[1]);
        }
        case 'Fn::If': {
            // Fn::If takes a 3-element list as its argument
            // ToDo the first argument is the name of the condition,
            // so we will need to retrieve the actual object from the template
            // when we handle preserveLogicalIds=false
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionIf(value[0], value[1], value[2]);
        }
        case 'Fn::Equals': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
        }
        default:
            throw new Error(`Unsupported CloudFormation function '${key}'`);
    }
}
function looksLikeCfnIntrinsic(object) {
    const objectKeys = Object.keys(object);
    // a CFN intrinsic is always an object with a single key
    if (objectKeys.length !== 1) {
        return undefined;
    }
    const key = objectKeys[0];
    return key === 'Ref' || key.startsWith('Fn::') ? key : undefined;
}
function specialCaseRefs(value) {
    switch (value) {
        case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
        case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
        case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
        case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
        case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
        case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
        case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
        case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
        default: return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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