"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const token_1 = require("./token");
const CFN_ELEMENT_SYMBOL = Symbol.for('@aws-cdk/core.CfnElement');
/**
 * An element of a CloudFormation stack.
 */
class CfnElement extends construct_compat_1.Construct {
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, CFN_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(this.node.path)}.LogicalID`,
        });
        this.node.addMetadata(cxschema.ArtifactMetadataEntryType.LOGICAL_ID, this.logicalId, this.constructor);
    }
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(x) {
        return CFN_ELEMENT_SYMBOL in x;
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStack() {
        const trace = this.node.metadata.find(md => md.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class CfnRefElement extends CfnElement {
    /**
     * Return a string that will be resolved to a CloudFormation `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     */
    get ref() {
        return token_1.Token.asString(cfn_reference_1.CfnReference.for(this, 'Ref'));
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./private/cfn-reference");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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