"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const semver = require("semver");
const my_package_json_1 = require("./private/my-package-json");
const schema_helpers_1 = require("./private/schema-helpers");
const PACKAGE_VERSION = my_package_json_1.loadMyPackageJson().version;
/**
 * Static class with loader routines
 *
 * This class mostly exists to put the schema structs into input position
 * (taken into a function), so that the jsii-diff checker will make sure all
 * structs are only allowed to be weakened in future updates. For example,
 * it is now allowed to add new required fields, since old CDK frameworks
 * would not be emitting those fields yet.
 *
 * At the same time, we might as well validate the structure so code doesn't
 * barf on invalid disk input.
 */
class AssetManifestSchema {
    /**
     * Validate the given structured object as a valid ManifestFile schema
     */
    static validate(file) {
        const obj = file;
        if (typeof obj !== 'object' || obj === null) {
            throw new Error(`Expected object, got '${obj}`);
        }
        schema_helpers_1.expectKey(obj, 'version', schema_helpers_1.isString);
        // Current tool must be >= the version used to write the manifest
        // (disregarding MVs) which we can verify by ^-prefixing the file version.
        if (!semver.satisfies(AssetManifestSchema.currentVersion(), `^${obj.version}`)) {
            throw new Error(`Need CDK Tools >= '${obj.version}' to read this file (current is '${AssetManifestSchema.currentVersion()}')`);
        }
        schema_helpers_1.expectKey(obj, 'files', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isFileAsset)), true);
        schema_helpers_1.expectKey(obj, 'dockerImages', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isDockerImageAsset)), true);
    }
    /**
     * Take a ManifestFile as input
     *
     * The presence of this method makes sure the struct is only ever weakened
     * in future releases.
     */
    static input(file) {
        this.validate(file);
    }
    /**
     * Return the version of the schema module
     */
    static currentVersion() {
        return PACKAGE_VERSION;
    }
}
exports.AssetManifestSchema = AssetManifestSchema;
function isFileAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'path', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'packaging', schema_helpers_1.isFileAssetPackaging, true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'bucketName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'objectKey', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
function isDockerImageAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'directory', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'dockerFile', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildTarget', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildArgs', schema_helpers_1.isMapOf(schema_helpers_1.isString), true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'repositoryName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'imageTag', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
//# sourceMappingURL=data:application/json;base64,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