"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2FpdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIndhaXQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSxvQ0FBaUM7QUFFakMscURBQWlEO0FBQ2pELG1DQUFnQztBQUNoQzs7O0dBR0c7QUFDSCxNQUFhLFFBQVE7SUF1QmpCLFlBQXFDLElBQVM7UUFBVCxTQUFJLEdBQUosSUFBSSxDQUFLO0lBQUksQ0FBQztJQXRCbkQ7O09BRUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLFFBQXNCLElBQUksT0FBTyxJQUFJLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMxRzs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxTQUFpQixJQUFJLE9BQU8sSUFBSSxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDN0Y7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsSUFBWSxJQUFJLE9BQU8sSUFBSSxRQUFRLENBQUMsRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdkY7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBWSxJQUFJLE9BQU8sSUFBSSxRQUFRLENBQUMsRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFM0Y7O09BRUc7SUFDSCxJQUFXLEtBQUs7UUFDWixPQUFPLElBQUksQ0FBQyxJQUFJLENBQUM7SUFDckIsQ0FBQztDQUNKO0FBOUJELDRCQThCQztBQWdCRDs7OztHQUlHO0FBQ0gsTUFBYSxJQUFLLFNBQVEsYUFBSztJQUczQixZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQWdCO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztRQUN2QixJQUFJLENBQUMsU0FBUyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUNEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLElBQWdCO1FBQ3hCLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksV0FBVztRQUNkLE9BQU87WUFDSCxJQUFJLEVBQUUsc0JBQVMsQ0FBQyxJQUFJO1lBQ3BCLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSztZQUNsQixHQUFHLElBQUksQ0FBQyxhQUFhLEVBQUU7U0FDMUIsQ0FBQztJQUNOLENBQUM7Q0FDSjtBQTFCRCxvQkEwQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDaGFpbiB9IGZyb20gJy4uL2NoYWluJztcbmltcG9ydCB7IElDaGFpbmFibGUsIElOZXh0YWJsZSB9IGZyb20gJy4uL3R5cGVzJztcbmltcG9ydCB7IFN0YXRlVHlwZSB9IGZyb20gJy4vcHJpdmF0ZS9zdGF0ZS10eXBlJztcbmltcG9ydCB7IFN0YXRlIH0gZnJvbSAnLi9zdGF0ZSc7XG4vKipcbiAqIFJlcHJlc2VudHMgdGhlIFdhaXQgc3RhdGUgd2hpY2ggZGVsYXlzIGEgc3RhdGUgbWFjaGluZSBmcm9tIGNvbnRpbnVpbmcgZm9yIGEgc3BlY2lmaWVkIHRpbWVcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9hbWF6b24tc3RhdGVzLWxhbmd1YWdlLXdhaXQtc3RhdGUuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgV2FpdFRpbWUge1xuICAgIC8qKlxuICAgICAqIFdhaXQgYSBmaXhlZCBhbW91bnQgb2YgdGltZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGR1cmF0aW9uKGR1cmF0aW9uOiBjZGsuRHVyYXRpb24pIHsgcmV0dXJuIG5ldyBXYWl0VGltZSh7IFNlY29uZHM6IGR1cmF0aW9uLnRvU2Vjb25kcygpIH0pOyB9XG4gICAgLyoqXG4gICAgICogV2FpdCB1bnRpbCB0aGUgZ2l2ZW4gSVNPODYwMSB0aW1lc3RhbXBcbiAgICAgKlxuICAgICAqIEBleGFtcGxlIDIwMTYtMDMtMTRUMDE6NTk6MDBaXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyB0aW1lc3RhbXAodGltZXN0YW1wOiBzdHJpbmcpIHsgcmV0dXJuIG5ldyBXYWl0VGltZSh7IFRpbWVzdGFtcDogdGltZXN0YW1wIH0pOyB9XG4gICAgLyoqXG4gICAgICogV2FpdCBmb3IgYSBudW1iZXIgb2Ygc2Vjb25kcyBzdG9yZWQgaW4gdGhlIHN0YXRlIG9iamVjdC5cbiAgICAgKlxuICAgICAqIEBleGFtcGxlICQud2FpdFNlY29uZHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHNlY29uZHNQYXRoKHBhdGg6IHN0cmluZykgeyByZXR1cm4gbmV3IFdhaXRUaW1lKHsgU2Vjb25kc1BhdGg6IHBhdGggfSk7IH1cbiAgICAvKipcbiAgICAgKiBXYWl0IHVudGlsIGEgdGltZXN0YW1wIGZvdW5kIGluIHRoZSBzdGF0ZSBvYmplY3QuXG4gICAgICpcbiAgICAgKiBAZXhhbXBsZSAkLndhaXRUaW1lc3RhbXBcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHRpbWVzdGFtcFBhdGgocGF0aDogc3RyaW5nKSB7IHJldHVybiBuZXcgV2FpdFRpbWUoeyBUaW1lc3RhbXBQYXRoOiBwYXRoIH0pOyB9XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGpzb246IGFueSkgeyB9XG4gICAgLyoqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIGdldCBfanNvbigpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuanNvbjtcbiAgICB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIGEgV2FpdCBzdGF0ZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFdhaXRQcm9wcyB7XG4gICAgLyoqXG4gICAgICogQW4gb3B0aW9uYWwgZGVzY3JpcHRpb24gZm9yIHRoaXMgc3RhdGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGNvbW1lbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFdhaXQgZHVyYXRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGltZTogV2FpdFRpbWU7XG59XG4vKipcbiAqIERlZmluZSBhIFdhaXQgc3RhdGUgaW4gdGhlIHN0YXRlIG1hY2hpbmVcbiAqXG4gKiBBIFdhaXQgc3RhdGUgY2FuIGJlIHVzZWQgdG8gZGVsYXkgZXhlY3V0aW9uIG9mIHRoZSBzdGF0ZSBtYWNoaW5lIGZvciBhIHdoaWxlLlxuICovXG5leHBvcnQgY2xhc3MgV2FpdCBleHRlbmRzIFN0YXRlIGltcGxlbWVudHMgSU5leHRhYmxlIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgZW5kU3RhdGVzOiBJTmV4dGFibGVbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHRpbWU6IFdhaXRUaW1lO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogV2FpdFByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgICAgICB0aGlzLnRpbWUgPSBwcm9wcy50aW1lO1xuICAgICAgICB0aGlzLmVuZFN0YXRlcyA9IFt0aGlzXTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ29udGludWUgbm9ybWFsIGV4ZWN1dGlvbiB3aXRoIHRoZSBnaXZlbiBzdGF0ZVxuICAgICAqL1xuICAgIHB1YmxpYyBuZXh0KG5leHQ6IElDaGFpbmFibGUpOiBDaGFpbiB7XG4gICAgICAgIHN1cGVyLm1ha2VOZXh0KG5leHQuc3RhcnRTdGF0ZSk7XG4gICAgICAgIHJldHVybiBDaGFpbi5zZXF1ZW5jZSh0aGlzLCBuZXh0KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBBbWF6b24gU3RhdGVzIExhbmd1YWdlIG9iamVjdCBmb3IgdGhpcyBzdGF0ZVxuICAgICAqL1xuICAgIHB1YmxpYyB0b1N0YXRlSnNvbigpOiBvYmplY3Qge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgVHlwZTogU3RhdGVUeXBlLldBSVQsXG4gICAgICAgICAgICBDb21tZW50OiB0aGlzLmNvbW1lbnQsXG4gICAgICAgICAgICAuLi50aGlzLnRpbWUuX2pzb24sXG4gICAgICAgICAgICAuLi50aGlzLnJlbmRlck5leHRFbmQoKSxcbiAgICAgICAgfTtcbiAgICB9XG59XG4iXX0=