"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A collection of connected states
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *   be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *   machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched
     * @param graphDescription description of the state machine
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph
     *
     * Called by State.bindToGraph().
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph
     *
     * Resource changes will be bubbled up to the given graph.
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * Return a string description of this graph
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
//# sourceMappingURL=data:application/json;base64,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