"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const parallel_1 = require("./states/parallel");
/**
 * A collection of states to chain onto
 *
 * A Chain has a start and zero or more chainable ends. If there are
 * zero ends, calling next() on the Chain will fail.
 */
class Chain {
    constructor(startState, endStates, lastAdded) {
        this.lastAdded = lastAdded;
        this.id = lastAdded.id;
        this.startState = startState;
        this.endStates = endStates;
    }
    /**
     * Begin a new Chain from one chainable
     */
    static start(state) {
        return new Chain(state.startState, state.endStates, state);
    }
    /**
     * Make a Chain with the start from one chain and the ends from another
     */
    static sequence(start, next) {
        return new Chain(start.startState, next.endStates, next);
    }
    /**
     * Make a Chain with specific start and end states, and a last-added Chainable
     */
    static custom(startState, endStates, lastAdded) {
        return new Chain(startState, endStates, lastAdded);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        if (this.endStates.length === 0) {
            throw new Error(`Cannot add to chain: last state in chain (${this.lastAdded.id}) does not allow it`);
        }
        for (const endState of this.endStates) {
            endState.next(next);
        }
        return new Chain(this.startState, next.endStates, next);
    }
    /**
     * Return a single state that encompasses all states in the chain
     *
     * This can be used to add error handling to a sequence of states.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(id, props = {}) {
        return new parallel_1.Parallel(this.startState, id, props).branch(this);
    }
}
exports.Chain = Chain;
//# sourceMappingURL=data:application/json;base64,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