"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
const sagemaker_task_base_types_1 = require("./sagemaker-task-base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 *  @experimental
 */
class SagemakerTransformTask {
    constructor(props) {
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SageMaker.`);
        }
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = (props.transformInput.transformDataSource.s3DataSource.s3DataType) ? (props.transformInput) :
            Object.assign({}, props.transformInput, { transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource,
                        s3DataType: sagemaker_task_base_types_1.S3DataType.S3_PREFIX,
                    },
                },
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
    }
    bind(task) {
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(task, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
                ],
            });
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('sagemaker', 'createTransformJob', this.integrationPattern),
            parameters: this.renderParameters(),
            policyStatements: this.makePolicyStatements(task),
        };
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy) ? { BatchStrategy: this.props.batchStrategy } : {},
            ...(this.renderEnvironment(this.props.environment)),
            ...(this.props.maxConcurrentTransforms) ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {},
            ...(this.props.maxPayloadInMB) ? { MaxPayloadInMB: this.props.maxPayloadInMB } : {},
            ModelName: this.props.modelName,
            ...(this.renderTags(this.props.tags)),
            ...(this.renderTransformInput(this.transformInput)),
            TransformJobName: this.props.transformJobName,
            ...(this.renderTransformOutput(this.props.transformOutput)),
            ...(this.renderTransformResources(this.transformResources)),
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType) ? { CompressionType: input.compressionType } : {},
                ...(input.contentType) ? { ContentType: input.contentType } : {},
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    },
                },
                ...(input.splitType) ? { SplitType: input.splitType } : {},
            },
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey) ? { KmsKeyId: output.encryptionKey.keyArn } : {},
                ...(output.accept) ? { Accept: output.accept } : {},
                ...(output.assembleWith) ? { AssembleWith: output.assembleWith } : {},
            },
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: 'ml.' + resources.instanceType,
                ...(resources.volumeKmsKeyId) ? { VolumeKmsKeyId: resources.volumeKmsKeyId.keyArn } : {},
            },
        };
    }
    renderEnvironment(environment) {
        return (environment) ? { Environment: environment } : {};
    }
    renderTags(tags) {
        return (tags) ? { Tags: Object.keys(tags).map(key => ({ Key: key, Value: tags[key] })) } : {};
    }
    makePolicyStatements(task) {
        const stack = core_1.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*',
                    })],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.SagemakerTransformTask = SagemakerTransformTask;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLXRyYW5zZm9ybS10YXNrLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2FnZW1ha2VyLXRyYW5zZm9ybS10YXNrLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXdDLENBQUMsbURBQW1EO0FBQzVGLHdDQUF3QyxDQUFDLG1EQUFtRDtBQUM1RixrREFBa0QsQ0FBQyw2REFBNkQ7QUFDaEgsd0NBQXNDLENBQUMsZ0RBQWdEO0FBQ3ZGLGdFQUF3RDtBQUN4RCwyRUFBNkg7QUErRTdIOzs7O0dBSUc7QUFDSCxNQUFhLHNCQUFzQjtJQVcvQixZQUE2QixLQUE4QjtRQUE5QixVQUFLLEdBQUwsS0FBSyxDQUF5QjtRQUN2RCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlLENBQUM7UUFDcEcsTUFBTSxpQkFBaUIsR0FBRztZQUN0QixHQUFHLENBQUMseUJBQXlCLENBQUMsZUFBZTtZQUM3QyxHQUFHLENBQUMseUJBQXlCLENBQUMsSUFBSTtTQUNyQyxDQUFDO1FBQ0YsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsRUFBRTtZQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLHdDQUF3QyxJQUFJLENBQUMsa0JBQWtCLHNDQUFzQyxDQUFDLENBQUM7U0FDMUg7UUFDRCwyQ0FBMkM7UUFDM0MsSUFBSSxLQUFLLENBQUMsSUFBSSxFQUFFO1lBQ1osSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1NBQzNCO1FBQ0Qsd0ZBQXdGO1FBQ3hGLElBQUksQ0FBQyxjQUFjLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztZQUMvRyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsY0FBYyxFQUFFLEVBQUUsbUJBQW1CLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUMsbUJBQW1CLENBQUMsWUFBWTt3QkFDN0gsVUFBVSxFQUFFLHNDQUFVLENBQUMsU0FBUztxQkFDbkM7aUJBQ0o7YUFDSixDQUFDLENBQUM7UUFDUCxvREFBb0Q7UUFDcEQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSTtZQUNsRCxhQUFhLEVBQUUsQ0FBQztZQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7U0FDbkYsQ0FBQztJQUNOLENBQUM7SUFDTSxJQUFJLENBQUMsSUFBYztRQUN0QixtQ0FBbUM7UUFDbkMsSUFBSSxJQUFJLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUMxQixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7Z0JBQ3RELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztnQkFDOUQsZUFBZSxFQUFFO29CQUNiLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMkJBQTJCLENBQUM7aUJBQzFFO2FBQ0osQ0FBQyxDQUFDO1NBQ047UUFDRCxPQUFPO1lBQ0gsV0FBVyxFQUFFLG9DQUFjLENBQUMsV0FBVyxFQUFFLG9CQUFvQixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztZQUN2RixVQUFVLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ25DLGdCQUFnQixFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUM7U0FDcEQsQ0FBQztJQUNOLENBQUM7SUFDRDs7OztPQUlHO0lBQ0gsSUFBVyxJQUFJO1FBQ1gsSUFBSSxJQUFJLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUMxQixNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDN0U7UUFDRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUM7SUFDdEIsQ0FBQztJQUNPLGdCQUFnQjtRQUdwQixPQUFPO1lBQ0gsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDaEYsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ25ELEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsdUJBQXVCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzlHLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLGNBQWMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ25GLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7WUFDL0IsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNyQyxHQUFHLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUNuRCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQjtZQUM3QyxHQUFHLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDM0QsR0FBRyxDQUFDLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQztTQUM5RCxDQUFDO0lBQ04sQ0FBQztJQUNPLG9CQUFvQixDQUFDLEtBQXFCO1FBRzlDLE9BQU87WUFDSCxjQUFjLEVBQUU7Z0JBQ1osR0FBRyxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUM1RSxHQUFHLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ2hFLFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUU7d0JBQ1YsS0FBSyxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxZQUFZLENBQUMsS0FBSzt3QkFDbkQsVUFBVSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxZQUFZLENBQUMsVUFBVTtxQkFDaEU7aUJBQ0o7Z0JBQ0QsR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQzdEO1NBQ0osQ0FBQztJQUNOLENBQUM7SUFDTyxxQkFBcUIsQ0FBQyxNQUF1QjtRQUdqRCxPQUFPO1lBQ0gsZUFBZSxFQUFFO2dCQUNiLFlBQVksRUFBRSxNQUFNLENBQUMsWUFBWTtnQkFDakMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxRQUFRLEVBQUUsTUFBTSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDMUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNuRCxHQUFHLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFlBQVksRUFBRSxNQUFNLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7YUFDeEU7U0FDSixDQUFDO0lBQ04sQ0FBQztJQUNPLHdCQUF3QixDQUFDLFNBQTZCO1FBRzFELE9BQU87WUFDSCxrQkFBa0IsRUFBRTtnQkFDaEIsYUFBYSxFQUFFLFNBQVMsQ0FBQyxhQUFhO2dCQUN0QyxZQUFZLEVBQUUsS0FBSyxHQUFHLFNBQVMsQ0FBQyxZQUFZO2dCQUM1QyxHQUFHLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLGNBQWMsRUFBRSxTQUFTLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQzNGO1NBQ0osQ0FBQztJQUNOLENBQUM7SUFDTyxpQkFBaUIsQ0FBQyxXQUViO1FBR1QsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0lBQzdELENBQUM7SUFDTyxVQUFVLENBQUMsSUFFTjtRQUdULE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNsRyxDQUFDO0lBQ08sb0JBQW9CLENBQUMsSUFBYztRQUN2QyxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdCLDBFQUEwRTtRQUMxRSxNQUFNLGdCQUFnQixHQUFHO1lBQ3JCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEIsT0FBTyxFQUFFLENBQUMsOEJBQThCLEVBQUUsZ0NBQWdDLEVBQUUsNEJBQTRCLENBQUM7Z0JBQ3pHLFNBQVMsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7d0JBQ3BCLE9BQU8sRUFBRSxXQUFXO3dCQUNwQixRQUFRLEVBQUUsZUFBZTt3QkFDekIsWUFBWSxFQUFFLEdBQUc7cUJBQ3BCLENBQUMsQ0FBQzthQUNWLENBQUM7WUFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3BCLE9BQU8sRUFBRSxDQUFDLG9CQUFvQixDQUFDO2dCQUMvQixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDbkIsQ0FBQztZQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDOUIsVUFBVSxFQUFFO29CQUNSLFlBQVksRUFBRSxFQUFFLHFCQUFxQixFQUFFLHlCQUF5QixFQUFFO2lCQUNyRTthQUNKLENBQUM7U0FDTCxDQUFDO1FBQ0YsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEtBQUssR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRTtZQUNoRSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUMxQyxPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxnQkFBZ0IsRUFBRSxxQkFBcUIsQ0FBQztnQkFDdkUsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQzt3QkFDcEIsT0FBTyxFQUFFLFFBQVE7d0JBQ2pCLFFBQVEsRUFBRSxNQUFNO3dCQUNoQixZQUFZLEVBQUUscURBQXFEO3FCQUN0RSxDQUFDLENBQUM7YUFDVixDQUFDLENBQUMsQ0FBQztTQUNQO1FBQ0QsT0FBTyxnQkFBZ0IsQ0FBQztJQUM1QixDQUFDO0NBQ0o7QUExS0Qsd0RBMEtDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gXCIuLi8uLi8uLi9hd3MtZWMyXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJ1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gXCIuLi8uLi8uLi9hd3Mtc3RlcGZ1bmN0aW9uc1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnXG5pbXBvcnQgeyBTdGFjayB9IGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgZ2V0UmVzb3VyY2VBcm4gfSBmcm9tICcuLi9yZXNvdXJjZS1hcm4tc3VmZml4JztcbmltcG9ydCB7IEJhdGNoU3RyYXRlZ3ksIFMzRGF0YVR5cGUsIFRyYW5zZm9ybUlucHV0LCBUcmFuc2Zvcm1PdXRwdXQsIFRyYW5zZm9ybVJlc291cmNlcyB9IGZyb20gJy4vc2FnZW1ha2VyLXRhc2stYmFzZS10eXBlcyc7XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGFuIEFtYXpvbiBTYWdlTWFrZXIgdHJhaW5pbmcgam9iIHRhc2tcbiAqXG4gKiAgQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNhZ2VtYWtlclRyYW5zZm9ybVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUcmFpbmluZyBKb2IgTmFtZS5cbiAgICAgKi9cbiAgICByZWFkb25seSB0cmFuc2Zvcm1Kb2JOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUm9sZSBmb3IgdGhlIFRyYWluaW5nIEpvYi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQSByb2xlIGlzIGNyZWF0ZWQgd2l0aCBgQW1hem9uU2FnZU1ha2VyRnVsbEFjY2Vzc2AgbWFuYWdlZCBwb2xpY3lcbiAgICAgKi9cbiAgICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIFRoZSBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4gaW5kaWNhdGVzIGRpZmZlcmVudCB3YXlzIHRvIGNhbGwgU2FnZU1ha2VyIEFQSXMuXG4gICAgICpcbiAgICAgKiBUaGUgdmFsaWQgdmFsdWUgaXMgZWl0aGVyIEZJUkVfQU5EX0ZPUkdFVCBvciBTWU5DLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRklSRV9BTkRfRk9SR0VUXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuPzogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG4gICAgLyoqXG4gICAgICogTnVtYmVyIG9mIHJlY29yZHMgdG8gaW5jbHVkZSBpbiBhIG1pbmktYmF0Y2ggZm9yIGFuIEhUVFAgaW5mZXJlbmNlIHJlcXVlc3QuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGJhdGNoIHN0cmF0ZWd5XG4gICAgICovXG4gICAgcmVhZG9ubHkgYmF0Y2hTdHJhdGVneT86IEJhdGNoU3RyYXRlZ3k7XG4gICAgLyoqXG4gICAgICogRW52aXJvbm1lbnQgdmFyaWFibGVzIHRvIHNldCBpbiB0aGUgRG9ja2VyIGNvbnRhaW5lci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gZW52aXJvbm1lbnQgdmFyaWFibGVzXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IHN0cmluZztcbiAgICB9O1xuICAgIC8qKlxuICAgICAqIE1heGltdW0gbnVtYmVyIG9mIHBhcmFsbGVsIHJlcXVlc3RzIHRoYXQgY2FuIGJlIHNlbnQgdG8gZWFjaCBpbnN0YW5jZSBpbiBhIHRyYW5zZm9ybSBqb2IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBTYWdlTWFrZXIgY2hlY2tzIHRoZSBvcHRpb25hbCBleGVjdXRpb24tcGFyYW1ldGVycyB0byBkZXRlcm1pbmUgdGhlIHNldHRpbmdzIGZvciB5b3VyIGNob3NlbiBhbGdvcml0aG0uXG4gICAgICogSWYgdGhlIGV4ZWN1dGlvbi1wYXJhbWV0ZXJzIGVuZHBvaW50IGlzIG5vdCBlbmFibGVkLCB0aGUgZGVmYXVsdCB2YWx1ZSBpcyAxLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1heENvbmN1cnJlbnRUcmFuc2Zvcm1zPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIE1heGltdW0gYWxsb3dlZCBzaXplIG9mIHRoZSBwYXlsb2FkLCBpbiBNQi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDZcbiAgICAgKi9cbiAgICByZWFkb25seSBtYXhQYXlsb2FkSW5NQj86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoZSBtb2RlbCB0aGF0IHlvdSB3YW50IHRvIHVzZSBmb3IgdGhlIHRyYW5zZm9ybSBqb2IuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbW9kZWxOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGFncyB0byBiZSBhcHBsaWVkIHRvIHRoZSB0cmFpbiBqb2IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIHRhZ3NcbiAgICAgKi9cbiAgICByZWFkb25seSB0YWdzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfTtcbiAgICAvKipcbiAgICAgKiBEYXRhc2V0IHRvIGJlIHRyYW5zZm9ybWVkIGFuZCB0aGUgQW1hem9uIFMzIGxvY2F0aW9uIHdoZXJlIGl0IGlzIHN0b3JlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSB0cmFuc2Zvcm1JbnB1dDogVHJhbnNmb3JtSW5wdXQ7XG4gICAgLyoqXG4gICAgICogUzMgbG9jYXRpb24gd2hlcmUgeW91IHdhbnQgQW1hem9uIFNhZ2VNYWtlciB0byBzYXZlIHRoZSByZXN1bHRzIGZyb20gdGhlIHRyYW5zZm9ybSBqb2IuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdHJhbnNmb3JtT3V0cHV0OiBUcmFuc2Zvcm1PdXRwdXQ7XG4gICAgLyoqXG4gICAgICogTUwgY29tcHV0ZSBpbnN0YW5jZXMgZm9yIHRoZSB0cmFuc2Zvcm0gam9iLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSAxIGluc3RhbmNlIG9mIHR5cGUgTTQuWExhcmdlXG4gICAgICovXG4gICAgcmVhZG9ubHkgdHJhbnNmb3JtUmVzb3VyY2VzPzogVHJhbnNmb3JtUmVzb3VyY2VzO1xufVxuLyoqXG4gKiBDbGFzcyByZXByZXNlbnRpbmcgdGhlIFNhZ2VNYWtlciBDcmVhdGUgVHJhaW5pbmcgSm9iIHRhc2suXG4gKlxuICogIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIFNhZ2VtYWtlclRyYW5zZm9ybVRhc2sgaW1wbGVtZW50cyBzZm4uSVN0ZXBGdW5jdGlvbnNUYXNrIHtcbiAgICAvKipcbiAgICAgKiBEYXRhc2V0IHRvIGJlIHRyYW5zZm9ybWVkIGFuZCB0aGUgQW1hem9uIFMzIGxvY2F0aW9uIHdoZXJlIGl0IGlzIHN0b3JlZC5cbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IHRyYW5zZm9ybUlucHV0OiBUcmFuc2Zvcm1JbnB1dDtcbiAgICAvKipcbiAgICAgKiBNTCBjb21wdXRlIGluc3RhbmNlcyBmb3IgdGhlIHRyYW5zZm9ybSBqb2IuXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSB0cmFuc2Zvcm1SZXNvdXJjZXM6IFRyYW5zZm9ybVJlc291cmNlcztcbiAgICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG4gICAgcHJpdmF0ZSBfcm9sZT86IGlhbS5JUm9sZTtcbiAgICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTYWdlbWFrZXJUcmFuc2Zvcm1Qcm9wcykge1xuICAgICAgICB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9IHByb3BzLmludGVncmF0aW9uUGF0dGVybiB8fCBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVQ7XG4gICAgICAgIGNvbnN0IHN1cHBvcnRlZFBhdHRlcm5zID0gW1xuICAgICAgICAgICAgc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uRklSRV9BTkRfRk9SR0VULFxuICAgICAgICAgICAgc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uU1lOQyxcbiAgICAgICAgXTtcbiAgICAgICAgaWYgKCFzdXBwb3J0ZWRQYXR0ZXJucy5pbmNsdWRlcyh0aGlzLmludGVncmF0aW9uUGF0dGVybikpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBTZXJ2aWNlIEludGVncmF0aW9uIFBhdHRlcm46ICR7dGhpcy5pbnRlZ3JhdGlvblBhdHRlcm59IGlzIG5vdCBzdXBwb3J0ZWQgdG8gY2FsbCBTYWdlTWFrZXIuYCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gc2V0IHRoZSBzYWdlbWFrZXIgcm9sZSBvciBjcmVhdGUgbmV3IG9uZVxuICAgICAgICBpZiAocHJvcHMucm9sZSkge1xuICAgICAgICAgICAgdGhpcy5fcm9sZSA9IHByb3BzLnJvbGU7XG4gICAgICAgIH1cbiAgICAgICAgLy8gc2V0IHRoZSBTMyBEYXRhIHR5cGUgb2YgdGhlIGlucHV0IGRhdGEgY29uZmlnIG9iamVjdHMgdG8gYmUgJ1MzUHJlZml4JyBpZiBub3QgZGVmaW5lZFxuICAgICAgICB0aGlzLnRyYW5zZm9ybUlucHV0ID0gKHByb3BzLnRyYW5zZm9ybUlucHV0LnRyYW5zZm9ybURhdGFTb3VyY2UuczNEYXRhU291cmNlLnMzRGF0YVR5cGUpID8gKHByb3BzLnRyYW5zZm9ybUlucHV0KSA6XG4gICAgICAgICAgICBPYmplY3QuYXNzaWduKHt9LCBwcm9wcy50cmFuc2Zvcm1JbnB1dCwgeyB0cmFuc2Zvcm1EYXRhU291cmNlOiB7IHMzRGF0YVNvdXJjZTogeyAuLi5wcm9wcy50cmFuc2Zvcm1JbnB1dC50cmFuc2Zvcm1EYXRhU291cmNlLnMzRGF0YVNvdXJjZSxcbiAgICAgICAgICAgICAgICAgICAgICAgIHMzRGF0YVR5cGU6IFMzRGF0YVR5cGUuUzNfUFJFRklYLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgLy8gc2V0IHRoZSBkZWZhdWx0IHZhbHVlIGZvciB0aGUgdHJhbnNmb3JtIHJlc291cmNlc1xuICAgICAgICB0aGlzLnRyYW5zZm9ybVJlc291cmNlcyA9IHByb3BzLnRyYW5zZm9ybVJlc291cmNlcyB8fCB7XG4gICAgICAgICAgICBpbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICAgICAgaW5zdGFuY2VUeXBlOiBlYzIuSW5zdGFuY2VUeXBlLm9mKGVjMi5JbnN0YW5jZUNsYXNzLk00LCBlYzIuSW5zdGFuY2VTaXplLlhMQVJHRSksXG4gICAgICAgIH07XG4gICAgfVxuICAgIHB1YmxpYyBiaW5kKHRhc2s6IHNmbi5UYXNrKTogc2ZuLlN0ZXBGdW5jdGlvbnNUYXNrQ29uZmlnIHtcbiAgICAgICAgLy8gY3JlYXRlIG5ldyByb2xlIGlmIGRvZXNuJ3QgZXhpc3RcbiAgICAgICAgaWYgKHRoaXMuX3JvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhpcy5fcm9sZSA9IG5ldyBpYW0uUm9sZSh0YXNrLCAnU2FnZW1ha2VyVHJhbnNmb3JtUm9sZScsIHtcbiAgICAgICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICAgICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzJyksXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICByZXNvdXJjZUFybjogZ2V0UmVzb3VyY2VBcm4oJ3NhZ2VtYWtlcicsICdjcmVhdGVUcmFuc2Zvcm1Kb2InLCB0aGlzLmludGVncmF0aW9uUGF0dGVybiksXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB0aGlzLnJlbmRlclBhcmFtZXRlcnMoKSxcbiAgICAgICAgICAgIHBvbGljeVN0YXRlbWVudHM6IHRoaXMubWFrZVBvbGljeVN0YXRlbWVudHModGFzayksXG4gICAgICAgIH07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBleGVjdXRpb24gcm9sZSBmb3IgdGhlIFNhZ2VtYWtlciB0cmFpbmluZyBqb2IuXG4gICAgICpcbiAgICAgKiBPbmx5IGF2YWlsYWJsZSBhZnRlciB0YXNrIGhhcyBiZWVuIGFkZGVkIHRvIGEgc3RhdGUgbWFjaGluZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0IHJvbGUoKTogaWFtLklSb2xlIHtcbiAgICAgICAgaWYgKHRoaXMuX3JvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdyb2xlIG5vdCBhdmFpbGFibGUgeWV0LS11c2UgdGhlIG9iamVjdCBpbiBhIFRhc2sgZmlyc3QnKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdGhpcy5fcm9sZTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJQYXJhbWV0ZXJzKCk6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgLi4uKHRoaXMucHJvcHMuYmF0Y2hTdHJhdGVneSkgPyB7IEJhdGNoU3RyYXRlZ3k6IHRoaXMucHJvcHMuYmF0Y2hTdHJhdGVneSB9IDoge30sXG4gICAgICAgICAgICAuLi4odGhpcy5yZW5kZXJFbnZpcm9ubWVudCh0aGlzLnByb3BzLmVudmlyb25tZW50KSksXG4gICAgICAgICAgICAuLi4odGhpcy5wcm9wcy5tYXhDb25jdXJyZW50VHJhbnNmb3JtcykgPyB7IE1heENvbmN1cnJlbnRUcmFuc2Zvcm1zOiB0aGlzLnByb3BzLm1heENvbmN1cnJlbnRUcmFuc2Zvcm1zIH0gOiB7fSxcbiAgICAgICAgICAgIC4uLih0aGlzLnByb3BzLm1heFBheWxvYWRJbk1CKSA/IHsgTWF4UGF5bG9hZEluTUI6IHRoaXMucHJvcHMubWF4UGF5bG9hZEluTUIgfSA6IHt9LFxuICAgICAgICAgICAgTW9kZWxOYW1lOiB0aGlzLnByb3BzLm1vZGVsTmFtZSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlclRhZ3ModGhpcy5wcm9wcy50YWdzKSksXG4gICAgICAgICAgICAuLi4odGhpcy5yZW5kZXJUcmFuc2Zvcm1JbnB1dCh0aGlzLnRyYW5zZm9ybUlucHV0KSksXG4gICAgICAgICAgICBUcmFuc2Zvcm1Kb2JOYW1lOiB0aGlzLnByb3BzLnRyYW5zZm9ybUpvYk5hbWUsXG4gICAgICAgICAgICAuLi4odGhpcy5yZW5kZXJUcmFuc2Zvcm1PdXRwdXQodGhpcy5wcm9wcy50cmFuc2Zvcm1PdXRwdXQpKSxcbiAgICAgICAgICAgIC4uLih0aGlzLnJlbmRlclRyYW5zZm9ybVJlc291cmNlcyh0aGlzLnRyYW5zZm9ybVJlc291cmNlcykpLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlclRyYW5zZm9ybUlucHV0KGlucHV0OiBUcmFuc2Zvcm1JbnB1dCk6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgVHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgICAgICAgICAuLi4oaW5wdXQuY29tcHJlc3Npb25UeXBlKSA/IHsgQ29tcHJlc3Npb25UeXBlOiBpbnB1dC5jb21wcmVzc2lvblR5cGUgfSA6IHt9LFxuICAgICAgICAgICAgICAgIC4uLihpbnB1dC5jb250ZW50VHlwZSkgPyB7IENvbnRlbnRUeXBlOiBpbnB1dC5jb250ZW50VHlwZSB9IDoge30sXG4gICAgICAgICAgICAgICAgRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgICAgIFMzVXJpOiBpbnB1dC50cmFuc2Zvcm1EYXRhU291cmNlLnMzRGF0YVNvdXJjZS5zM1VyaSxcbiAgICAgICAgICAgICAgICAgICAgICAgIFMzRGF0YVR5cGU6IGlucHV0LnRyYW5zZm9ybURhdGFTb3VyY2UuczNEYXRhU291cmNlLnMzRGF0YVR5cGUsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAuLi4oaW5wdXQuc3BsaXRUeXBlKSA/IHsgU3BsaXRUeXBlOiBpbnB1dC5zcGxpdFR5cGUgfSA6IHt9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJUcmFuc2Zvcm1PdXRwdXQob3V0cHV0OiBUcmFuc2Zvcm1PdXRwdXQpOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgICB9IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFRyYW5zZm9ybU91dHB1dDoge1xuICAgICAgICAgICAgICAgIFMzT3V0cHV0UGF0aDogb3V0cHV0LnMzT3V0cHV0UGF0aCxcbiAgICAgICAgICAgICAgICAuLi4ob3V0cHV0LmVuY3J5cHRpb25LZXkpID8geyBLbXNLZXlJZDogb3V0cHV0LmVuY3J5cHRpb25LZXkua2V5QXJuIH0gOiB7fSxcbiAgICAgICAgICAgICAgICAuLi4ob3V0cHV0LmFjY2VwdCkgPyB7IEFjY2VwdDogb3V0cHV0LmFjY2VwdCB9IDoge30sXG4gICAgICAgICAgICAgICAgLi4uKG91dHB1dC5hc3NlbWJsZVdpdGgpID8geyBBc3NlbWJsZVdpdGg6IG91dHB1dC5hc3NlbWJsZVdpdGggfSA6IHt9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJUcmFuc2Zvcm1SZXNvdXJjZXMocmVzb3VyY2VzOiBUcmFuc2Zvcm1SZXNvdXJjZXMpOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgICB9IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFRyYW5zZm9ybVJlc291cmNlczoge1xuICAgICAgICAgICAgICAgIEluc3RhbmNlQ291bnQ6IHJlc291cmNlcy5pbnN0YW5jZUNvdW50LFxuICAgICAgICAgICAgICAgIEluc3RhbmNlVHlwZTogJ21sLicgKyByZXNvdXJjZXMuaW5zdGFuY2VUeXBlLFxuICAgICAgICAgICAgICAgIC4uLihyZXNvdXJjZXMudm9sdW1lS21zS2V5SWQpID8geyBWb2x1bWVLbXNLZXlJZDogcmVzb3VyY2VzLnZvbHVtZUttc0tleUlkLmtleUFybiB9IDoge30sXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlckVudmlyb25tZW50KGVudmlyb25tZW50OiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgICB9IHwgdW5kZWZpbmVkKToge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB7XG4gICAgICAgIHJldHVybiAoZW52aXJvbm1lbnQpID8geyBFbnZpcm9ubWVudDogZW52aXJvbm1lbnQgfSA6IHt9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlclRhZ3ModGFnczoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB8IHVuZGVmaW5lZCk6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4gKHRhZ3MpID8geyBUYWdzOiBPYmplY3Qua2V5cyh0YWdzKS5tYXAoa2V5ID0+ICh7IEtleToga2V5LCBWYWx1ZTogdGFnc1trZXldIH0pKSB9IDoge307XG4gICAgfVxuICAgIHByaXZhdGUgbWFrZVBvbGljeVN0YXRlbWVudHModGFzazogc2ZuLlRhc2spOiBpYW0uUG9saWN5U3RhdGVtZW50W10ge1xuICAgICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRhc2spO1xuICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL3NhZ2VtYWtlci1pYW0uaHRtbFxuICAgICAgICBjb25zdCBwb2xpY3lTdGF0ZW1lbnRzID0gW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnc2FnZW1ha2VyOkNyZWF0ZVRyYW5zZm9ybUpvYicsICdzYWdlbWFrZXI6RGVzY3JpYmVUcmFuc2Zvcm1Kb2InLCAnc2FnZW1ha2VyOlN0b3BUcmFuc2Zvcm1Kb2InXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFtzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmljZTogJ3NhZ2VtYWtlcicsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZTogJ3RyYW5zZm9ybS1qb2InLFxuICAgICAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiAnKicsXG4gICAgICAgICAgICAgICAgICAgIH0pXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnc2FnZW1ha2VyOkxpc3RUYWdzJ10sXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10sXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5yb2xlLnJvbGVBcm5dLFxuICAgICAgICAgICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7ICdpYW06UGFzc2VkVG9TZXJ2aWNlJzogJ3NhZ2VtYWtlci5hbWF6b25hd3MuY29tJyB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgXTtcbiAgICAgICAgaWYgKHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID09PSBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DKSB7XG4gICAgICAgICAgICBwb2xpY3lTdGF0ZW1lbnRzLnB1c2gobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnZXZlbnRzOlB1dFRhcmdldHMnLCAnZXZlbnRzOlB1dFJ1bGUnLCAnZXZlbnRzOkRlc2NyaWJlUnVsZSddLFxuICAgICAgICAgICAgICAgIHJlc291cmNlczogW3N0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgICAgICAgICAgICBzZXJ2aWNlOiAnZXZlbnRzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHJlc291cmNlOiAncnVsZScsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6ICdTdGVwRnVuY3Rpb25zR2V0RXZlbnRzRm9yU2FnZU1ha2VyVHJhbnNmb3JtSm9ic1J1bGUnLFxuICAgICAgICAgICAgICAgICAgICB9KV0sXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gICAgfVxufVxuIl19