"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
const sagemaker_task_base_types_1 = require("./sagemaker-task-base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SagemakerTrainTask {
    constructor(props) {
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SageMaker.`);
        }
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSizeInGB: 10,
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if ((!props.algorithmSpecification.algorithmName) && (!props.algorithmSpecification.trainingImage)) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = (props.algorithmSpecification.trainingInputMode) ?
            (props.algorithmSpecification) :
            ({ ...props.algorithmSpecification, trainingInputMode: sagemaker_task_base_types_1.InputMode.FILE });
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map(config => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return Object.assign({}, config, { dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: sagemaker_task_base_types_1.S3DataType.S3_PREFIX } } });
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = (props.vpcConfig.subnets) ?
                (this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds) : this.vpc.selectSubnets().subnetIds;
        }
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    bind(task) {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role = this.props.role || new iam.Role(task, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateTrainingJob: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'cloudwatch:PutMetricData',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                                'logs:CreateLogGroup',
                                'logs:DescribeLogStreams',
                                'ecr:GetAuthorizationToken',
                                ...this.props.vpcConfig
                                    ? [
                                        'ec2:CreateNetworkInterface',
                                        'ec2:CreateNetworkInterfacePermission',
                                        'ec2:DeleteNetworkInterface',
                                        'ec2:DeleteNetworkInterfacePermission',
                                        'ec2:DescribeNetworkInterfaces',
                                        'ec2:DescribeVpcs',
                                        'ec2:DescribeDhcpOptions',
                                        'ec2:DescribeSubnets',
                                        'ec2:DescribeSecurityGroups',
                                    ]
                                    : [],
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(task, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            parameters: this.renderParameters(),
            policyStatements: this.makePolicyStatements(task),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...(this.renderAlgorithmSpecification(this.algorithmSpecification)),
            ...(this.renderInputDataConfig(this.inputDataConfig)),
            ...(this.renderOutputDataConfig(this.props.outputDataConfig)),
            ...(this.renderResourceConfig(this.resourceConfig)),
            ...(this.renderStoppingCondition(this.stoppingCondition)),
            ...(this.renderHyperparameters(this.props.hyperparameters)),
            ...(this.renderTags(this.props.tags)),
            ...(this.renderVpcConfig(this.props.vpcConfig)),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage) ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {},
                ...(spec.algorithmName) ? { AlgorithmName: spec.algorithmName } : {},
                ...(spec.metricDefinitions) ?
                    { MetricDefinitions: spec.metricDefinitions
                            .map(metric => ({ Name: metric.name, Regex: metric.regex })) } : {},
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map(channel => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType) ?
                            { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType } : {},
                        ...(channel.dataSource.s3DataSource.attributeNames) ?
                            { AtttributeNames: channel.dataSource.s3DataSource.attributeNames } : {},
                    },
                },
                ...(channel.compressionType) ? { CompressionType: channel.compressionType } : {},
                ...(channel.contentType) ? { ContentType: channel.contentType } : {},
                ...(channel.inputMode) ? { InputMode: channel.inputMode } : {},
                ...(channel.recordWrapperType) ? { RecordWrapperType: channel.recordWrapperType } : {},
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey) ? { KmsKeyId: config.encryptionKey.keyArn } : {},
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSizeInGB,
                ...(config.volumeEncryptionKey) ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {},
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return (params) ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return (tags) ? { Tags: Object.keys(tags).map(key => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return (config) ? { VpcConfig: {
                SecurityGroupIds: core_1.Lazy.listValue({ produce: () => (this.securityGroups.map(sg => (sg.securityGroupId))) }),
                Subnets: this.subnets,
            } } : {};
    }
    makePolicyStatements(task) {
        const stack = core_1.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.Data.isJsonPathString(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.SagemakerTrainTask = SagemakerTrainTask;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLXRyYWluLXRhc2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzYWdlbWFrZXItdHJhaW4tdGFzay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHdDQUF3QyxDQUFDLG1EQUFtRDtBQUM1Rix3Q0FBd0MsQ0FBQyxtREFBbUQ7QUFDNUYsa0RBQWtELENBQUMsNkRBQTZEO0FBQ2hILHdDQUFzRCxDQUFDLGdEQUFnRDtBQUN2RyxnRUFBd0Q7QUFDeEQsMkVBQXFLO0FBNkVySzs7OztHQUlHO0FBQ0gsTUFBYSxrQkFBa0I7SUE0QjNCLFlBQTZCLEtBQThCO1FBQTlCLFVBQUssR0FBTCxLQUFLLENBQXlCO1FBM0IzRDs7V0FFRztRQUNhLGdCQUFXLEdBQW9CLElBQUksR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBbUJwRCxtQkFBYyxHQUF5QixFQUFFLENBQUM7UUFNdkQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxHQUFHLENBQUMseUJBQXlCLENBQUMsZUFBZSxDQUFDO1FBQ3BHLE1BQU0saUJBQWlCLEdBQUc7WUFDdEIsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7WUFDN0MsR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUk7U0FDckMsQ0FBQztRQUNGLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEVBQUU7WUFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsSUFBSSxDQUFDLGtCQUFrQixzQ0FBc0MsQ0FBQyxDQUFDO1NBQzFIO1FBQ0Qsa0RBQWtEO1FBQ2xELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSTtZQUMxQyxhQUFhLEVBQUUsQ0FBQztZQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDaEYsY0FBYyxFQUFFLEVBQUU7U0FDckIsQ0FBQztRQUNGLDRDQUE0QztRQUM1QyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixJQUFJO1lBQ2hELFVBQVUsRUFBRSxlQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztTQUNoQyxDQUFDO1FBQ0YsdURBQXVEO1FBQ3ZELElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ2hHLE1BQU0sSUFBSSxLQUFLLENBQUMsMkZBQTJGLENBQUMsQ0FBQztTQUNoSDtRQUNELDhDQUE4QztRQUM5QyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsQ0FBQyxLQUFLLENBQUMsc0JBQXNCLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO1lBQzVFLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQztZQUNoQyxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUMsc0JBQXNCLEVBQUUsaUJBQWlCLEVBQUUscUNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzdFLHdGQUF3RjtRQUN4RixJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3RELElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxVQUFVLEVBQUU7Z0JBQzVDLE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsTUFBTSxFQUFFLEVBQUUsVUFBVSxFQUFFLEVBQUUsWUFBWSxFQUFFLEVBQUUsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLFlBQVksRUFBRSxVQUFVLEVBQUUsc0NBQVUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQzthQUMvSTtpQkFDSTtnQkFDRCxPQUFPLE1BQU0sQ0FBQzthQUNqQjtRQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0gsb0RBQW9EO1FBQ3BELElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUNqQixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDO1lBQy9CLElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7Z0JBQ3RDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxTQUFTLENBQUM7U0FDeEc7SUFDTCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNILElBQVcsSUFBSTtRQUNYLElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO1NBQzdFO1FBQ0QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3RCLENBQUM7SUFDRCxJQUFXLGNBQWM7UUFDckIsSUFBSSxJQUFJLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUNwQyxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUM7U0FDbEY7UUFDRCxPQUFPLElBQUksQ0FBQyxlQUFlLENBQUM7SUFDaEMsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ksZ0JBQWdCLENBQUMsYUFBaUM7UUFDckQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUNNLElBQUksQ0FBQyxJQUFjO1FBQ3RCLDJDQUEyQztRQUMzQyxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7WUFDdkYsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO1lBQzlELGNBQWMsRUFBRTtnQkFDWixpQkFBaUIsRUFBRSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7b0JBQ3RDLFVBQVUsRUFBRTt3QkFDUixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7NEJBQ3BCLE9BQU8sRUFBRTtnQ0FDTCwwQkFBMEI7Z0NBQzFCLHNCQUFzQjtnQ0FDdEIsbUJBQW1CO2dDQUNuQixxQkFBcUI7Z0NBQ3JCLHlCQUF5QjtnQ0FDekIsMkJBQTJCO2dDQUMzQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUztvQ0FDbkIsQ0FBQyxDQUFDO3dDQUNFLDRCQUE0Qjt3Q0FDNUIsc0NBQXNDO3dDQUN0Qyw0QkFBNEI7d0NBQzVCLHNDQUFzQzt3Q0FDdEMsK0JBQStCO3dDQUMvQixrQkFBa0I7d0NBQ2xCLHlCQUF5Qjt3Q0FDekIscUJBQXFCO3dDQUNyQiw0QkFBNEI7cUNBQy9CO29DQUNELENBQUMsQ0FBQyxFQUFFOzZCQUNYOzRCQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzt5QkFDbkIsQ0FBQztxQkFDTDtpQkFDSixDQUFDO2FBQ0w7U0FDSixDQUFDLENBQUM7UUFDSCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxFQUFFO1lBQzNDLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDdEU7UUFDRCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLG1CQUFtQixFQUFFO1lBQzVFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7U0FDdEY7UUFDRCx5Q0FBeUM7UUFDekMsSUFBSSxJQUFJLENBQUMsR0FBRyxJQUFJLElBQUksQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO1lBQzlDLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtnQkFDdEUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2FBQ2hCLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxXQUFXLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3RELElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUNoRDtRQUNELE9BQU87WUFDSCxXQUFXLEVBQUUsb0NBQWMsQ0FBQyxXQUFXLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDO1lBQ3RGLFVBQVUsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDbkMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQztTQUNwRCxDQUFDO0lBQ04sQ0FBQztJQUNPLGdCQUFnQjtRQUdwQixPQUFPO1lBQ0gsZUFBZSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZTtZQUMzQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQU0sQ0FBQyxPQUFPO1lBQzVCLEdBQUcsQ0FBQyxJQUFJLENBQUMsNEJBQTRCLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUM7WUFDbkUsR0FBRyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDckQsR0FBRyxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDN0QsR0FBRyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7WUFDbkQsR0FBRyxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztZQUN6RCxHQUFHLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDM0QsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNyQyxHQUFHLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ2xELENBQUM7SUFDTixDQUFDO0lBQ08sNEJBQTRCLENBQUMsSUFBNEI7UUFHN0QsT0FBTztZQUNILHNCQUFzQixFQUFFO2dCQUNwQixpQkFBaUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCO2dCQUN6QyxHQUFHLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDeEYsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxhQUFhLEVBQUUsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNwRSxHQUFHLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQztvQkFDekIsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCOzZCQUNsQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTthQUNsRjtTQUNKLENBQUM7SUFDTixDQUFDO0lBQ08scUJBQXFCLENBQUMsTUFBaUI7UUFHM0MsT0FBTztZQUNILGVBQWUsRUFBRSxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztnQkFDcEMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxXQUFXO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1IsWUFBWSxFQUFFO3dCQUNWLEtBQUssRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUc7d0JBQ3RGLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxVQUFVO3dCQUN0RCxHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDOzRCQUN6RCxFQUFFLHNCQUFzQixFQUFFLE9BQU8sQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLHNCQUFzQixFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7d0JBQzNGLEdBQUcsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDOzRCQUNqRCxFQUFFLGVBQWUsRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtxQkFDL0U7aUJBQ0o7Z0JBQ0QsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxlQUFlLEVBQUUsT0FBTyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNoRixHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3BFLEdBQUcsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDOUQsR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQ3pGLENBQUMsQ0FBQztTQUNOLENBQUM7SUFDTixDQUFDO0lBQ08sc0JBQXNCLENBQUMsTUFBd0I7UUFHbkQsT0FBTztZQUNILGdCQUFnQixFQUFFO2dCQUNkLFlBQVksRUFBRSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUc7Z0JBQzFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsUUFBUSxFQUFFLE1BQU0sQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7YUFDN0U7U0FDSixDQUFDO0lBQ04sQ0FBQztJQUNPLG9CQUFvQixDQUFDLE1BQXNCO1FBRy9DLE9BQU87WUFDSCxjQUFjLEVBQUU7Z0JBQ1osYUFBYSxFQUFFLE1BQU0sQ0FBQyxhQUFhO2dCQUNuQyxZQUFZLEVBQUUsS0FBSyxHQUFHLE1BQU0sQ0FBQyxZQUFZO2dCQUN6QyxjQUFjLEVBQUUsTUFBTSxDQUFDLGNBQWM7Z0JBQ3JDLEdBQUcsQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsTUFBTSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQy9GO1NBQ0osQ0FBQztJQUNOLENBQUM7SUFDTyx1QkFBdUIsQ0FBQyxNQUF5QjtRQUdyRCxPQUFPO1lBQ0gsaUJBQWlCLEVBQUU7Z0JBQ2YsbUJBQW1CLEVBQUUsTUFBTSxDQUFDLFVBQVUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLFNBQVMsRUFBRTthQUMxRTtTQUNKLENBQUM7SUFDTixDQUFDO0lBQ08scUJBQXFCLENBQUMsTUFFakI7UUFHVCxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsZUFBZSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDdkQsQ0FBQztJQUNPLFVBQVUsQ0FBQyxJQUVOO1FBR1QsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0lBQ2xHLENBQUM7SUFDTyxlQUFlLENBQUMsTUFBNkI7UUFHakQsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRTtnQkFDdkIsZ0JBQWdCLEVBQUUsV0FBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQzFHLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTzthQUN4QixFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNqQixDQUFDO0lBQ08sb0JBQW9CLENBQUMsSUFBYztRQUN2QyxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdCLDBFQUEwRTtRQUMxRSxNQUFNLGdCQUFnQixHQUFHO1lBQ3JCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEIsT0FBTyxFQUFFLENBQUMsNkJBQTZCLEVBQUUsK0JBQStCLEVBQUUsMkJBQTJCLENBQUM7Z0JBQ3RHLFNBQVMsRUFBRTtvQkFDUCxLQUFLLENBQUMsU0FBUyxDQUFDO3dCQUNaLE9BQU8sRUFBRSxXQUFXO3dCQUNwQixRQUFRLEVBQUUsY0FBYzt3QkFDeEIsdUdBQXVHO3dCQUN2RyxZQUFZLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLEdBQUc7cUJBQy9HLENBQUM7aUJBQ0w7YUFDSixDQUFDO1lBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUNwQixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQztnQkFDL0IsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2FBQ25CLENBQUM7WUFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3BCLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQztnQkFDekIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQU0sQ0FBQyxPQUFPLENBQUM7Z0JBQ2hDLFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUUsRUFBRSxxQkFBcUIsRUFBRSx5QkFBeUIsRUFBRTtpQkFDckU7YUFDSixDQUFDO1NBQ0wsQ0FBQztRQUNGLElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUU7WUFDaEUsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDMUMsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCLENBQUM7Z0JBQ3ZFLFNBQVMsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7d0JBQ3BCLE9BQU8sRUFBRSxRQUFRO3dCQUNqQixRQUFRLEVBQUUsTUFBTTt3QkFDaEIsWUFBWSxFQUFFLG9EQUFvRDtxQkFDckUsQ0FBQyxDQUFDO2FBQ1YsQ0FBQyxDQUFDLENBQUM7U0FDUDtRQUNELE9BQU8sZ0JBQWdCLENBQUM7SUFDNUIsQ0FBQztDQUNKO0FBelNELGdEQXlTQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIHNmbiBmcm9tIFwiLi4vLi4vLi4vYXdzLXN0ZXBmdW5jdGlvbnNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJ1xuaW1wb3J0IHsgRHVyYXRpb24sIExhenksIFN0YWNrIH0gZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBnZXRSZXNvdXJjZUFybiB9IGZyb20gJy4uL3Jlc291cmNlLWFybi1zdWZmaXgnO1xuaW1wb3J0IHsgQWxnb3JpdGhtU3BlY2lmaWNhdGlvbiwgQ2hhbm5lbCwgSW5wdXRNb2RlLCBPdXRwdXREYXRhQ29uZmlnLCBSZXNvdXJjZUNvbmZpZywgUzNEYXRhVHlwZSwgU3RvcHBpbmdDb25kaXRpb24sIFZwY0NvbmZpZyB9IGZyb20gJy4vc2FnZW1ha2VyLXRhc2stYmFzZS10eXBlcyc7XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGFuIEFtYXpvbiBTYWdlTWFrZXIgdHJhaW5pbmcgam9iXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNhZ2VtYWtlclRyYWluVGFza1Byb3BzIHtcbiAgICAvKipcbiAgICAgKiBUcmFpbmluZyBKb2IgTmFtZS5cbiAgICAgKi9cbiAgICByZWFkb25seSB0cmFpbmluZ0pvYk5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBSb2xlIGZvciB0aGUgVHJhaW5pbmcgSm9iLiBUaGUgcm9sZSBtdXN0IGJlIGdyYW50ZWQgYWxsIG5lY2Vzc2FyeSBwZXJtaXNzaW9ucyBmb3IgdGhlIFNhZ2VNYWtlciB0cmFpbmluZyBqb2IgdG9cbiAgICAgKiBiZSBhYmxlIHRvIG9wZXJhdGUuXG4gICAgICpcbiAgICAgKiBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyX2ZyL3NhZ2VtYWtlci9sYXRlc3QvZGcvc2FnZW1ha2VyLXJvbGVzLmh0bWwjc2FnZW1ha2VyLXJvbGVzLWNyZWF0ZXRyYWluaW5nam9iLXBlcm1zXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIGEgcm9sZSB3aXRoIGFwcHJvcHJpYXRlIHBlcm1pc3Npb25zIHdpbGwgYmUgY3JlYXRlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIFRoZSBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4gaW5kaWNhdGVzIGRpZmZlcmVudCB3YXlzIHRvIGNhbGwgU2FnZU1ha2VyIEFQSXMuXG4gICAgICpcbiAgICAgKiBUaGUgdmFsaWQgdmFsdWUgaXMgZWl0aGVyIEZJUkVfQU5EX0ZPUkdFVCBvciBTWU5DLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRklSRV9BTkRfRk9SR0VUXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuPzogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG4gICAgLyoqXG4gICAgICogSWRlbnRpZmllcyB0aGUgdHJhaW5pbmcgYWxnb3JpdGhtIHRvIHVzZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBhbGdvcml0aG1TcGVjaWZpY2F0aW9uOiBBbGdvcml0aG1TcGVjaWZpY2F0aW9uO1xuICAgIC8qKlxuICAgICAqIEFsZ29yaXRobS1zcGVjaWZpYyBwYXJhbWV0ZXJzIHRoYXQgaW5mbHVlbmNlIHRoZSBxdWFsaXR5IG9mIHRoZSBtb2RlbC4gU2V0IGh5cGVycGFyYW1ldGVycyBiZWZvcmUgeW91IHN0YXJ0IHRoZSBsZWFybmluZyBwcm9jZXNzLlxuICAgICAqIEZvciBhIGxpc3Qgb2YgaHlwZXJwYXJhbWV0ZXJzIHByb3ZpZGVkIGJ5IEFtYXpvbiBTYWdlTWFrZXJcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zYWdlbWFrZXIvbGF0ZXN0L2RnL2FsZ29zLmh0bWxcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gaHlwZXJwYXJhbWV0ZXJzXG4gICAgICovXG4gICAgcmVhZG9ubHkgaHlwZXJwYXJhbWV0ZXJzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfTtcbiAgICAvKipcbiAgICAgKiAgRGVzY3JpYmVzIHRoZSB2YXJpb3VzIGRhdGFzZXRzIChlLmcuIHRyYWluLCB2YWxpZGF0aW9uLCB0ZXN0KSBhbmQgdGhlIEFtYXpvbiBTMyBsb2NhdGlvbiB3aGVyZSBzdG9yZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5wdXREYXRhQ29uZmlnOiBDaGFubmVsW107XG4gICAgLyoqXG4gICAgICogVGFncyB0byBiZSBhcHBsaWVkIHRvIHRoZSB0cmFpbiBqb2IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIHRhZ3NcbiAgICAgKi9cbiAgICByZWFkb25seSB0YWdzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfTtcbiAgICAvKipcbiAgICAgKiBJZGVudGlmaWVzIHRoZSBBbWF6b24gUzMgbG9jYXRpb24gd2hlcmUgeW91IHdhbnQgQW1hem9uIFNhZ2VNYWtlciB0byBzYXZlIHRoZSByZXN1bHRzIG9mIG1vZGVsIHRyYWluaW5nLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG91dHB1dERhdGFDb25maWc6IE91dHB1dERhdGFDb25maWc7XG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIHRoZSByZXNvdXJjZXMsIE1MIGNvbXB1dGUgaW5zdGFuY2VzLCBhbmQgTUwgc3RvcmFnZSB2b2x1bWVzIHRvIGRlcGxveSBmb3IgbW9kZWwgdHJhaW5pbmcuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIDEgaW5zdGFuY2Ugb2YgRUMyIGBNNC5YTGFyZ2VgIHdpdGggYDEwR0JgIHZvbHVtZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlc291cmNlQ29uZmlnPzogUmVzb3VyY2VDb25maWc7XG4gICAgLyoqXG4gICAgICogU2V0cyBhIHRpbWUgbGltaXQgZm9yIHRyYWluaW5nLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBtYXggcnVudGltZSBvZiAxIGhvdXJcbiAgICAgKi9cbiAgICByZWFkb25seSBzdG9wcGluZ0NvbmRpdGlvbj86IFN0b3BwaW5nQ29uZGl0aW9uO1xuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyB0aGUgVlBDIHRoYXQgeW91IHdhbnQgeW91ciB0cmFpbmluZyBqb2IgdG8gY29ubmVjdCB0by5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gVlBDXG4gICAgICovXG4gICAgcmVhZG9ubHkgdnBjQ29uZmlnPzogVnBjQ29uZmlnO1xufVxuLyoqXG4gKiBDbGFzcyByZXByZXNlbnRpbmcgdGhlIFNhZ2VNYWtlciBDcmVhdGUgVHJhaW5pbmcgSm9iIHRhc2suXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgU2FnZW1ha2VyVHJhaW5UYXNrIGltcGxlbWVudHMgaWFtLklHcmFudGFibGUsIGVjMi5JQ29ubmVjdGFibGUsIHNmbi5JU3RlcEZ1bmN0aW9uc1Rhc2sge1xuICAgIC8qKlxuICAgICAqIEFsbG93cyBzcGVjaWZ5IHNlY3VyaXR5IGdyb3VwIGNvbm5lY3Rpb25zIGZvciBpbnN0YW5jZXMgb2YgdGhpcyBmbGVldC5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoKTtcbiAgICAvKipcbiAgICAgKiBUaGUgQWxnb3JpdGhtIFNwZWNpZmljYXRpb25cbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IGFsZ29yaXRobVNwZWNpZmljYXRpb246IEFsZ29yaXRobVNwZWNpZmljYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIElucHV0IERhdGEgQ29uZmlnLlxuICAgICAqL1xuICAgIHByaXZhdGUgcmVhZG9ubHkgaW5wdXREYXRhQ29uZmlnOiBDaGFubmVsW107XG4gICAgLyoqXG4gICAgICogVGhlIHJlc291cmNlIGNvbmZpZyBmb3IgdGhlIHRhc2suXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZUNvbmZpZzogUmVzb3VyY2VDb25maWc7XG4gICAgLyoqXG4gICAgICogVGhlIHJlc291cmNlIGNvbmZpZyBmb3IgdGhlIHRhc2suXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBzdG9wcGluZ0NvbmRpdGlvbjogU3RvcHBpbmdDb25kaXRpb247XG4gICAgcHJpdmF0ZSByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcbiAgICBwcml2YXRlIHNlY3VyaXR5R3JvdXA/OiBlYzIuSVNlY3VyaXR5R3JvdXA7XG4gICAgcHJpdmF0ZSByZWFkb25seSBzZWN1cml0eUdyb3VwczogZWMyLklTZWN1cml0eUdyb3VwW10gPSBbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHN1Ym5ldHM/OiBzdHJpbmdbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm47XG4gICAgcHJpdmF0ZSBfcm9sZT86IGlhbS5JUm9sZTtcbiAgICBwcml2YXRlIF9ncmFudFByaW5jaXBhbD86IGlhbS5JUHJpbmNpcGFsO1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFNhZ2VtYWtlclRyYWluVGFza1Byb3BzKSB7XG4gICAgICAgIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID0gcHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuIHx8IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLkZJUkVfQU5EX0ZPUkdFVDtcbiAgICAgICAgY29uc3Qgc3VwcG9ydGVkUGF0dGVybnMgPSBbXG4gICAgICAgICAgICBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVQsXG4gICAgICAgICAgICBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DLFxuICAgICAgICBdO1xuICAgICAgICBpZiAoIXN1cHBvcnRlZFBhdHRlcm5zLmluY2x1ZGVzKHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIFNlcnZpY2UgSW50ZWdyYXRpb24gUGF0dGVybjogJHt0aGlzLmludGVncmF0aW9uUGF0dGVybn0gaXMgbm90IHN1cHBvcnRlZCB0byBjYWxsIFNhZ2VNYWtlci5gKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBzZXQgdGhlIGRlZmF1bHQgcmVzb3VyY2UgY29uZmlnIGlmIG5vdCBkZWZpbmVkLlxuICAgICAgICB0aGlzLnJlc291cmNlQ29uZmlnID0gcHJvcHMucmVzb3VyY2VDb25maWcgfHwge1xuICAgICAgICAgICAgaW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgICAgIGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZS5vZihlYzIuSW5zdGFuY2VDbGFzcy5NNCwgZWMyLkluc3RhbmNlU2l6ZS5YTEFSR0UpLFxuICAgICAgICAgICAgdm9sdW1lU2l6ZUluR0I6IDEwLFxuICAgICAgICB9O1xuICAgICAgICAvLyBzZXQgdGhlIHN0b3BwaW5nIGNvbmRpdGlvbiBpZiBub3QgZGVmaW5lZFxuICAgICAgICB0aGlzLnN0b3BwaW5nQ29uZGl0aW9uID0gcHJvcHMuc3RvcHBpbmdDb25kaXRpb24gfHwge1xuICAgICAgICAgICAgbWF4UnVudGltZTogRHVyYXRpb24uaG91cnMoMSksXG4gICAgICAgIH07XG4gICAgICAgIC8vIGNoZWNrIHRoYXQgZWl0aGVyIGFsZ29yaXRobSBuYW1lIG9yIGltYWdlIGlzIGRlZmluZWRcbiAgICAgICAgaWYgKCghcHJvcHMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbi5hbGdvcml0aG1OYW1lKSAmJiAoIXByb3BzLmFsZ29yaXRobVNwZWNpZmljYXRpb24udHJhaW5pbmdJbWFnZSkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTXVzdCBkZWZpbmUgZWl0aGVyIGFuIGFsZ29yaXRobSBuYW1lIG9yIHRyYWluaW5nIGltYWdlIFVSSSBpbiB0aGUgYWxnb3JpdGhtIHNwZWNpZmljYXRpb24nKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBzZXQgdGhlIGlucHV0IG1vZGUgdG8gJ0ZpbGUnIGlmIG5vdCBkZWZpbmVkXG4gICAgICAgIHRoaXMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbiA9IChwcm9wcy5hbGdvcml0aG1TcGVjaWZpY2F0aW9uLnRyYWluaW5nSW5wdXRNb2RlKSA/XG4gICAgICAgICAgICAocHJvcHMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbikgOlxuICAgICAgICAgICAgKHsgLi4ucHJvcHMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbiwgdHJhaW5pbmdJbnB1dE1vZGU6IElucHV0TW9kZS5GSUxFIH0pO1xuICAgICAgICAvLyBzZXQgdGhlIFMzIERhdGEgdHlwZSBvZiB0aGUgaW5wdXQgZGF0YSBjb25maWcgb2JqZWN0cyB0byBiZSAnUzNQcmVmaXgnIGlmIG5vdCBkZWZpbmVkXG4gICAgICAgIHRoaXMuaW5wdXREYXRhQ29uZmlnID0gcHJvcHMuaW5wdXREYXRhQ29uZmlnLm1hcChjb25maWcgPT4ge1xuICAgICAgICAgICAgaWYgKCFjb25maWcuZGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UuczNEYXRhVHlwZSkge1xuICAgICAgICAgICAgICAgIHJldHVybiBPYmplY3QuYXNzaWduKHt9LCBjb25maWcsIHsgZGF0YVNvdXJjZTogeyBzM0RhdGFTb3VyY2U6IHsgLi4uY29uZmlnLmRhdGFTb3VyY2UuczNEYXRhU291cmNlLCBzM0RhdGFUeXBlOiBTM0RhdGFUeXBlLlMzX1BSRUZJWCB9IH0gfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICByZXR1cm4gY29uZmlnO1xuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgICAgLy8gYWRkIHRoZSBzZWN1cml0eSBncm91cHMgdG8gdGhlIGNvbm5lY3Rpb25zIG9iamVjdFxuICAgICAgICBpZiAocHJvcHMudnBjQ29uZmlnKSB7XG4gICAgICAgICAgICB0aGlzLnZwYyA9IHByb3BzLnZwY0NvbmZpZy52cGM7XG4gICAgICAgICAgICB0aGlzLnN1Ym5ldHMgPSAocHJvcHMudnBjQ29uZmlnLnN1Ym5ldHMpID9cbiAgICAgICAgICAgICAgICAodGhpcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy52cGNDb25maWcuc3VibmV0cykuc3VibmV0SWRzKSA6IHRoaXMudnBjLnNlbGVjdFN1Ym5ldHMoKS5zdWJuZXRJZHM7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogVGhlIGV4ZWN1dGlvbiByb2xlIGZvciB0aGUgU2FnZW1ha2VyIHRyYWluaW5nIGpvYi5cbiAgICAgKlxuICAgICAqIE9ubHkgYXZhaWxhYmxlIGFmdGVyIHRhc2sgaGFzIGJlZW4gYWRkZWQgdG8gYSBzdGF0ZSBtYWNoaW5lLlxuICAgICAqL1xuICAgIHB1YmxpYyBnZXQgcm9sZSgpOiBpYW0uSVJvbGUge1xuICAgICAgICBpZiAodGhpcy5fcm9sZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ3JvbGUgbm90IGF2YWlsYWJsZSB5ZXQtLXVzZSB0aGUgb2JqZWN0IGluIGEgVGFzayBmaXJzdCcpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLl9yb2xlO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IGdyYW50UHJpbmNpcGFsKCk6IGlhbS5JUHJpbmNpcGFsIHtcbiAgICAgICAgaWYgKHRoaXMuX2dyYW50UHJpbmNpcGFsID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUHJpbmNpcGFsIG5vdCBhdmFpbGFibGUgeWV0LS11c2UgdGhlIG9iamVjdCBpbiBhIFRhc2sgZmlyc3QnKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdGhpcy5fZ3JhbnRQcmluY2lwYWw7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCB0aGUgc2VjdXJpdHkgZ3JvdXAgdG8gYWxsIGluc3RhbmNlcyB2aWEgdGhlIGxhdW5jaCBjb25maWd1cmF0aW9uXG4gICAgICogc2VjdXJpdHkgZ3JvdXBzIGFycmF5LlxuICAgICAqXG4gICAgICogQHBhcmFtIHNlY3VyaXR5R3JvdXA6IFRoZSBzZWN1cml0eSBncm91cCB0byBhZGRcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkU2VjdXJpdHlHcm91cChzZWN1cml0eUdyb3VwOiBlYzIuSVNlY3VyaXR5R3JvdXApOiB2b2lkIHtcbiAgICAgICAgdGhpcy5zZWN1cml0eUdyb3Vwcy5wdXNoKHNlY3VyaXR5R3JvdXApO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZCh0YXNrOiBzZm4uVGFzayk6IHNmbi5TdGVwRnVuY3Rpb25zVGFza0NvbmZpZyB7XG4gICAgICAgIC8vIHNldCB0aGUgc2FnZW1ha2VyIHJvbGUgb3IgY3JlYXRlIG5ldyBvbmVcbiAgICAgICAgdGhpcy5fZ3JhbnRQcmluY2lwYWwgPSB0aGlzLl9yb2xlID0gdGhpcy5wcm9wcy5yb2xlIHx8IG5ldyBpYW0uUm9sZSh0YXNrLCAnU2FnZW1ha2VyUm9sZScsIHtcbiAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgICAgICAgICBDcmVhdGVUcmFpbmluZ0pvYjogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdjbG91ZHdhdGNoOlB1dE1ldHJpY0RhdGEnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dHcm91cCcsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdsb2dzOkRlc2NyaWJlTG9nU3RyZWFtcycsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlY3I6R2V0QXV0aG9yaXphdGlvblRva2VuJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgLi4udGhpcy5wcm9wcy52cGNDb25maWdcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgID8gW1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2VjMjpDcmVhdGVOZXR3b3JrSW50ZXJmYWNlUGVybWlzc2lvbicsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2VjMjpEZWxldGVOZXR3b3JrSW50ZXJmYWNlJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2VQZXJtaXNzaW9uJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlYzI6RGVzY3JpYmVWcGNzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlc2NyaWJlRGhjcE9wdGlvbnMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICdlYzI6RGVzY3JpYmVTdWJuZXRzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAnZWMyOkRlc2NyaWJlU2VjdXJpdHlHcm91cHMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgOiBbXSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgICBpZiAodGhpcy5wcm9wcy5vdXRwdXREYXRhQ29uZmlnLmVuY3J5cHRpb25LZXkpIHtcbiAgICAgICAgICAgIHRoaXMucHJvcHMub3V0cHV0RGF0YUNvbmZpZy5lbmNyeXB0aW9uS2V5LmdyYW50RW5jcnlwdCh0aGlzLl9yb2xlKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy5wcm9wcy5yZXNvdXJjZUNvbmZpZyAmJiB0aGlzLnByb3BzLnJlc291cmNlQ29uZmlnLnZvbHVtZUVuY3J5cHRpb25LZXkpIHtcbiAgICAgICAgICAgIHRoaXMucHJvcHMucmVzb3VyY2VDb25maWcudm9sdW1lRW5jcnlwdGlvbktleS5ncmFudCh0aGlzLl9yb2xlLCAna21zOkNyZWF0ZUdyYW50Jyk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gY3JlYXRlIGEgc2VjdXJpdHkgZ3JvdXAgaWYgbm90IGRlZmluZWRcbiAgICAgICAgaWYgKHRoaXMudnBjICYmIHRoaXMuc2VjdXJpdHlHcm91cCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aGlzLnNlY3VyaXR5R3JvdXAgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGFzaywgJ1RyYWluSm9iU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgICAgICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB0aGlzLmNvbm5lY3Rpb25zLmFkZFNlY3VyaXR5R3JvdXAodGhpcy5zZWN1cml0eUdyb3VwKTtcbiAgICAgICAgICAgIHRoaXMuc2VjdXJpdHlHcm91cHMucHVzaCh0aGlzLnNlY3VyaXR5R3JvdXApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICByZXNvdXJjZUFybjogZ2V0UmVzb3VyY2VBcm4oJ3NhZ2VtYWtlcicsICdjcmVhdGVUcmFpbmluZ0pvYicsIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuKSxcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHRoaXMucmVuZGVyUGFyYW1ldGVycygpLFxuICAgICAgICAgICAgcG9saWN5U3RhdGVtZW50czogdGhpcy5tYWtlUG9saWN5U3RhdGVtZW50cyh0YXNrKSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJQYXJhbWV0ZXJzKCk6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgVHJhaW5pbmdKb2JOYW1lOiB0aGlzLnByb3BzLnRyYWluaW5nSm9iTmFtZSxcbiAgICAgICAgICAgIFJvbGVBcm46IHRoaXMuX3JvbGUhLnJvbGVBcm4sXG4gICAgICAgICAgICAuLi4odGhpcy5yZW5kZXJBbGdvcml0aG1TcGVjaWZpY2F0aW9uKHRoaXMuYWxnb3JpdGhtU3BlY2lmaWNhdGlvbikpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVySW5wdXREYXRhQ29uZmlnKHRoaXMuaW5wdXREYXRhQ29uZmlnKSksXG4gICAgICAgICAgICAuLi4odGhpcy5yZW5kZXJPdXRwdXREYXRhQ29uZmlnKHRoaXMucHJvcHMub3V0cHV0RGF0YUNvbmZpZykpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVyUmVzb3VyY2VDb25maWcodGhpcy5yZXNvdXJjZUNvbmZpZykpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVyU3RvcHBpbmdDb25kaXRpb24odGhpcy5zdG9wcGluZ0NvbmRpdGlvbikpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVySHlwZXJwYXJhbWV0ZXJzKHRoaXMucHJvcHMuaHlwZXJwYXJhbWV0ZXJzKSksXG4gICAgICAgICAgICAuLi4odGhpcy5yZW5kZXJUYWdzKHRoaXMucHJvcHMudGFncykpLFxuICAgICAgICAgICAgLi4uKHRoaXMucmVuZGVyVnBjQ29uZmlnKHRoaXMucHJvcHMudnBjQ29uZmlnKSksXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyQWxnb3JpdGhtU3BlY2lmaWNhdGlvbihzcGVjOiBBbGdvcml0aG1TcGVjaWZpY2F0aW9uKToge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBBbGdvcml0aG1TcGVjaWZpY2F0aW9uOiB7XG4gICAgICAgICAgICAgICAgVHJhaW5pbmdJbnB1dE1vZGU6IHNwZWMudHJhaW5pbmdJbnB1dE1vZGUsXG4gICAgICAgICAgICAgICAgLi4uKHNwZWMudHJhaW5pbmdJbWFnZSkgPyB7IFRyYWluaW5nSW1hZ2U6IHNwZWMudHJhaW5pbmdJbWFnZS5iaW5kKHRoaXMpLmltYWdlVXJpIH0gOiB7fSxcbiAgICAgICAgICAgICAgICAuLi4oc3BlYy5hbGdvcml0aG1OYW1lKSA/IHsgQWxnb3JpdGhtTmFtZTogc3BlYy5hbGdvcml0aG1OYW1lIH0gOiB7fSxcbiAgICAgICAgICAgICAgICAuLi4oc3BlYy5tZXRyaWNEZWZpbml0aW9ucykgP1xuICAgICAgICAgICAgICAgICAgICB7IE1ldHJpY0RlZmluaXRpb25zOiBzcGVjLm1ldHJpY0RlZmluaXRpb25zXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgLm1hcChtZXRyaWMgPT4gKHsgTmFtZTogbWV0cmljLm5hbWUsIFJlZ2V4OiBtZXRyaWMucmVnZXggfSkpIH0gOiB7fSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVySW5wdXREYXRhQ29uZmlnKGNvbmZpZzogQ2hhbm5lbFtdKToge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBJbnB1dERhdGFDb25maWc6IGNvbmZpZy5tYXAoY2hhbm5lbCA9PiAoe1xuICAgICAgICAgICAgICAgIENoYW5uZWxOYW1lOiBjaGFubmVsLmNoYW5uZWxOYW1lLFxuICAgICAgICAgICAgICAgIERhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgUzNEYXRhU291cmNlOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICBTM1VyaTogY2hhbm5lbC5kYXRhU291cmNlLnMzRGF0YVNvdXJjZS5zM0xvY2F0aW9uLmJpbmQodGhpcywgeyBmb3JSZWFkaW5nOiB0cnVlIH0pLnVyaSxcbiAgICAgICAgICAgICAgICAgICAgICAgIFMzRGF0YVR5cGU6IGNoYW5uZWwuZGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UuczNEYXRhVHlwZSxcbiAgICAgICAgICAgICAgICAgICAgICAgIC4uLihjaGFubmVsLmRhdGFTb3VyY2UuczNEYXRhU291cmNlLnMzRGF0YURpc3RyaWJ1dGlvblR5cGUpID9cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICB7IFMzRGF0YURpc3RyaWJ1dGlvblR5cGU6IGNoYW5uZWwuZGF0YVNvdXJjZS5zM0RhdGFTb3VyY2UuczNEYXRhRGlzdHJpYnV0aW9uVHlwZSB9IDoge30sXG4gICAgICAgICAgICAgICAgICAgICAgICAuLi4oY2hhbm5lbC5kYXRhU291cmNlLnMzRGF0YVNvdXJjZS5hdHRyaWJ1dGVOYW1lcykgP1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHsgQXR0dHJpYnV0ZU5hbWVzOiBjaGFubmVsLmRhdGFTb3VyY2UuczNEYXRhU291cmNlLmF0dHJpYnV0ZU5hbWVzIH0gOiB7fSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIC4uLihjaGFubmVsLmNvbXByZXNzaW9uVHlwZSkgPyB7IENvbXByZXNzaW9uVHlwZTogY2hhbm5lbC5jb21wcmVzc2lvblR5cGUgfSA6IHt9LFxuICAgICAgICAgICAgICAgIC4uLihjaGFubmVsLmNvbnRlbnRUeXBlKSA/IHsgQ29udGVudFR5cGU6IGNoYW5uZWwuY29udGVudFR5cGUgfSA6IHt9LFxuICAgICAgICAgICAgICAgIC4uLihjaGFubmVsLmlucHV0TW9kZSkgPyB7IElucHV0TW9kZTogY2hhbm5lbC5pbnB1dE1vZGUgfSA6IHt9LFxuICAgICAgICAgICAgICAgIC4uLihjaGFubmVsLnJlY29yZFdyYXBwZXJUeXBlKSA/IHsgUmVjb3JkV3JhcHBlclR5cGU6IGNoYW5uZWwucmVjb3JkV3JhcHBlclR5cGUgfSA6IHt9LFxuICAgICAgICAgICAgfSkpLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlck91dHB1dERhdGFDb25maWcoY29uZmlnOiBPdXRwdXREYXRhQ29uZmlnKToge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBPdXRwdXREYXRhQ29uZmlnOiB7XG4gICAgICAgICAgICAgICAgUzNPdXRwdXRQYXRoOiBjb25maWcuczNPdXRwdXRMb2NhdGlvbi5iaW5kKHRoaXMsIHsgZm9yV3JpdGluZzogdHJ1ZSB9KS51cmksXG4gICAgICAgICAgICAgICAgLi4uKGNvbmZpZy5lbmNyeXB0aW9uS2V5KSA/IHsgS21zS2V5SWQ6IGNvbmZpZy5lbmNyeXB0aW9uS2V5LmtleUFybiB9IDoge30sXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlclJlc291cmNlQ29uZmlnKGNvbmZpZzogUmVzb3VyY2VDb25maWcpOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgICB9IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFJlc291cmNlQ29uZmlnOiB7XG4gICAgICAgICAgICAgICAgSW5zdGFuY2VDb3VudDogY29uZmlnLmluc3RhbmNlQ291bnQsXG4gICAgICAgICAgICAgICAgSW5zdGFuY2VUeXBlOiAnbWwuJyArIGNvbmZpZy5pbnN0YW5jZVR5cGUsXG4gICAgICAgICAgICAgICAgVm9sdW1lU2l6ZUluR0I6IGNvbmZpZy52b2x1bWVTaXplSW5HQixcbiAgICAgICAgICAgICAgICAuLi4oY29uZmlnLnZvbHVtZUVuY3J5cHRpb25LZXkpID8geyBWb2x1bWVLbXNLZXlJZDogY29uZmlnLnZvbHVtZUVuY3J5cHRpb25LZXkua2V5QXJuIH0gOiB7fSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyU3RvcHBpbmdDb25kaXRpb24oY29uZmlnOiBTdG9wcGluZ0NvbmRpdGlvbik6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgU3RvcHBpbmdDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgICBNYXhSdW50aW1lSW5TZWNvbmRzOiBjb25maWcubWF4UnVudGltZSAmJiBjb25maWcubWF4UnVudGltZS50b1NlY29uZHMoKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVySHlwZXJwYXJhbWV0ZXJzKHBhcmFtczoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB8IHVuZGVmaW5lZCk6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4gKHBhcmFtcykgPyB7IEh5cGVyUGFyYW1ldGVyczogcGFyYW1zIH0gOiB7fTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJUYWdzKHRhZ3M6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0gfCB1bmRlZmluZWQpOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgICB9IHtcbiAgICAgICAgcmV0dXJuICh0YWdzKSA/IHsgVGFnczogT2JqZWN0LmtleXModGFncykubWFwKGtleSA9PiAoeyBLZXk6IGtleSwgVmFsdWU6IHRhZ3Nba2V5XSB9KSkgfSA6IHt9O1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlclZwY0NvbmZpZyhjb25maWc6IFZwY0NvbmZpZyB8IHVuZGVmaW5lZCk6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH0ge1xuICAgICAgICByZXR1cm4gKGNvbmZpZykgPyB7IFZwY0NvbmZpZzoge1xuICAgICAgICAgICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IExhenkubGlzdFZhbHVlKHsgcHJvZHVjZTogKCkgPT4gKHRoaXMuc2VjdXJpdHlHcm91cHMubWFwKHNnID0+IChzZy5zZWN1cml0eUdyb3VwSWQpKSkgfSksXG4gICAgICAgICAgICAgICAgU3VibmV0czogdGhpcy5zdWJuZXRzLFxuICAgICAgICAgICAgfSB9IDoge307XG4gICAgfVxuICAgIHByaXZhdGUgbWFrZVBvbGljeVN0YXRlbWVudHModGFzazogc2ZuLlRhc2spOiBpYW0uUG9saWN5U3RhdGVtZW50W10ge1xuICAgICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRhc2spO1xuICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL3NhZ2VtYWtlci1pYW0uaHRtbFxuICAgICAgICBjb25zdCBwb2xpY3lTdGF0ZW1lbnRzID0gW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnc2FnZW1ha2VyOkNyZWF0ZVRyYWluaW5nSm9iJywgJ3NhZ2VtYWtlcjpEZXNjcmliZVRyYWluaW5nSm9iJywgJ3NhZ2VtYWtlcjpTdG9wVHJhaW5pbmdKb2InXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICAgICAgc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHNlcnZpY2U6ICdzYWdlbWFrZXInLFxuICAgICAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2U6ICd0cmFpbmluZy1qb2InLFxuICAgICAgICAgICAgICAgICAgICAgICAgLy8gSWYgdGhlIGpvYiBuYW1lIGNvbWVzIGZyb20gaW5wdXQsIHdlIGNhbm5vdCB0YXJnZXQgdGhlIHBvbGljeSB0byBhIHBhcnRpY3VsYXIgQVJOIHByZWZpeCByZWxpYWJseS4uLlxuICAgICAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiBzZm4uRGF0YS5pc0pzb25QYXRoU3RyaW5nKHRoaXMucHJvcHMudHJhaW5pbmdKb2JOYW1lKSA/ICcqJyA6IGAke3RoaXMucHJvcHMudHJhaW5pbmdKb2JOYW1lfSpgLFxuICAgICAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydzYWdlbWFrZXI6TGlzdFRhZ3MnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydpYW06UGFzc1JvbGUnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLl9yb2xlIS5yb2xlQXJuXSxcbiAgICAgICAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICAgICAgICAgIFN0cmluZ0VxdWFsczogeyAnaWFtOlBhc3NlZFRvU2VydmljZSc6ICdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSksXG4gICAgICAgIF07XG4gICAgICAgIGlmICh0aGlzLmludGVncmF0aW9uUGF0dGVybiA9PT0gc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uU1lOQykge1xuICAgICAgICAgICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2V2ZW50czpQdXRUYXJnZXRzJywgJ2V2ZW50czpQdXRSdWxlJywgJ2V2ZW50czpEZXNjcmliZVJ1bGUnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFtzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmljZTogJ2V2ZW50cycsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZTogJ3J1bGUnLFxuICAgICAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiAnU3RlcEZ1bmN0aW9uc0dldEV2ZW50c0ZvclNhZ2VNYWtlclRyYWluaW5nSm9ic1J1bGUnLFxuICAgICAgICAgICAgICAgICAgICB9KV0sXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gICAgfVxufVxuIl19