"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ecs = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const run_ecs_task_base_1 = require("./run-ecs-task-base");
/**
 * Run an ECS/EC2 Task in a StepFunctions workflow
 */
class RunEcsEc2Task extends run_ecs_task_base_1.EcsRunTaskBase {
    constructor(props) {
        if (!props.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with EC2');
        }
        if (!props.cluster.hasEc2Capacity) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addXxxCapacity() on the cluster.');
        }
        if (!props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        super({
            ...props,
            parameters: {
                LaunchType: 'EC2',
                PlacementConstraints: noEmpty(flatten((props.placementConstraints || []).map(c => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten((props.placementStrategies || []).map(c => c.toJson().map(uppercaseKeys)))),
            },
        });
        if (props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking(props.cluster.vpc, undefined, props.subnets, props.securityGroup);
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            validateNoNetworkingProps(props);
            this.connections.addSecurityGroup(...props.cluster.connections.securityGroups);
        }
    }
}
exports.RunEcsEc2Task = RunEcsEc2Task;
/**
 * Validate combinations of networking arguments
 */
function validateNoNetworkingProps(props) {
    if (props.subnets !== undefined || props.securityGroup !== undefined) {
        throw new Error('vpcPlacement and securityGroup can only be used in AwsVpc networking mode');
    }
}
function uppercaseKeys(obj) {
    const ret = {};
    for (const key of Object.keys(obj)) {
        ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
    }
    return ret;
}
function flatten(xs) {
    return Array.prototype.concat([], ...xs);
}
function noEmpty(xs) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs;
}
//# sourceMappingURL=data:application/json;base64,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