"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
exports.AUTOGEN_MARKER = '$$autogen$$';
/**
 * Renders an ARN for an SSM parameter given a parameter name.
 * @param scope definition scope
 * @param parameterName the parameter name to include in the ARN
 * @param physicalName optional physical name specified by the user (to auto-detect separator)
 */
function arnForParameterName(scope, parameterName, options = {}) {
    const physicalName = options.physicalName;
    const nameToValidate = physicalName || parameterName;
    if (!core_1.Token.isUnresolved(nameToValidate) && nameToValidate.includes('/') && !nameToValidate.startsWith('/')) {
        throw new Error(`Parameter names must be fully qualified (if they include "/" they must also begin with a "/"): ${nameToValidate}`);
    }
    return core_1.Stack.of(scope).formatArn({
        service: 'ssm',
        resource: 'parameter',
        sep: isSimpleName() ? '/' : '',
        resourceName: parameterName,
    });
    /**
     * Determines the ARN separator for this parameter: if we have a concrete
     * parameter name (or explicitly defined physical name), we will parse them
     * and decide whether a "/" is needed or not. Otherwise, users will have to
     * explicitly specify `simpleName` when they import the ARN.
     */
    function isSimpleName() {
        // look for a concrete name as a hint for determining the separator
        const concreteName = !core_1.Token.isUnresolved(parameterName) ? parameterName : physicalName;
        if (!concreteName || core_1.Token.isUnresolved(concreteName)) {
            if (options.simpleName === undefined) {
                throw new Error('Unable to determine ARN separator for SSM parameter since the parameter name is an unresolved token. Use "fromAttributes" and specify "simpleName" explicitly');
            }
            return options.simpleName;
        }
        const result = !concreteName.startsWith('/');
        // if users explicitly specify the separator and it conflicts with the one we need, it's an error.
        if (options.simpleName !== undefined && options.simpleName !== result) {
            if (concreteName === exports.AUTOGEN_MARKER) {
                throw new Error('If "parameterName" is not explicitly defined, "simpleName" must be "true" or undefined since auto-generated parameter names always have simple names');
            }
            throw new Error(`Parameter name "${concreteName}" is ${result ? 'a simple name' : 'not a simple name'}, but "simpleName" was explicitly set to ${options.simpleName}. Either omit it or set it to ${result}`);
        }
        return result;
    }
}
exports.arnForParameterName = arnForParameterName;
//# sourceMappingURL=data:application/json;base64,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