"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ssm = require("./ssm.generated");
const util_1 = require("./util");
/**
 * Basic features shared across all types of SSM Parameters.
 */
class ParameterBase extends core_1.Resource {
    grantRead(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'ssm:DescribeParameters',
                'ssm:GetParameters',
                'ssm:GetParameter',
                'ssm:GetParameterHistory',
            ],
            resourceArns: [this.parameterArn],
        });
    }
    grantWrite(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['ssm:PutParameter'],
            resourceArns: [this.parameterArn],
        });
    }
}
/**
 * SSM parameter type
 */
var ParameterType;
(function (ParameterType) {
    /**
     * String
     */
    ParameterType["STRING"] = "String";
    /**
     * Secure String
     * Parameter Store uses an AWS Key Management Service (KMS) customer master key (CMK) to encrypt the parameter value.
     */
    ParameterType["SECURE_STRING"] = "SecureString";
    /**
     * String List
     */
    ParameterType["STRING_LIST"] = "StringList";
    /**
     * An Amazon EC2 image ID, such as ami-0ff8a91507f77f867
     */
    ParameterType["AWS_EC2_IMAGE_ID"] = "AWS::EC2::Image::Id";
})(ParameterType = exports.ParameterType || (exports.ParameterType = {}));
/**
 * SSM parameter tier
 */
var ParameterTier;
(function (ParameterTier) {
    /**
     * String
     */
    ParameterTier["ADVANCED"] = "Advanced";
    /**
     * String
     */
    ParameterTier["INTELLIGENT_TIERING"] = "Intelligent-Tiering";
    /**
     * String
     */
    ParameterTier["STANDARD"] = "Standard";
})(ParameterTier = exports.ParameterTier || (exports.ParameterTier = {}));
/**
 * Creates a new String SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.allowedPattern) {
            _assertValidValue(props.stringValue, props.allowedPattern);
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: props.type || ParameterType.STRING,
            value: props.stringValue,
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringValue = resource.attrValue;
    }
    /**
     * Imports an external string parameter by name.
     */
    static fromStringParameterName(scope, id, stringParameterName) {
        return this.fromStringParameterAttributes(scope, id, { parameterName: stringParameterName });
    }
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope, id, attrs) {
        if (!attrs.parameterName) {
            throw new Error('parameterName cannot be an empty string');
        }
        const type = attrs.type || ParameterType.STRING;
        const stringValue = attrs.version
            ? new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, `${attrs.parameterName}:${attrs.version}`).toString()
            : new core_1.CfnParameter(scope, `${id}.Parameter`, { type: `AWS::SSM::Parameter::Value<${type}>`, default: attrs.parameterName }).valueAsString;
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = type;
                this.stringValue = stringValue;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a secure string parameter from the SSM parameter store.
     */
    static fromSecureStringParameterAttributes(scope, id, attrs) {
        const stringValue = new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM_SECURE, `${attrs.parameterName}:${attrs.version}`).toString();
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = ParameterType.SECURE_STRING;
                this.stringValue = stringValue;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope, parameterName) {
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SSM_PARAMETER_PROVIDER,
            props: { parameterName },
            dummyValue: `dummy-value-for-${parameterName}`,
        }).value;
        return value;
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForStringParameter(scope, parameterName, version) {
        return StringParameter.valueForTypedStringParameter(scope, parameterName, ParameterType.STRING, version);
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param type The type of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForTypedStringParameter(scope, parameterName, type = ParameterType.STRING, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromStringParameterAttributes(stack, id, { parameterName, version, type }).stringValue;
    }
    /**
     * Returns a token that will resolve (during deployment)
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter
     * @param version The parameter version (required for secure strings)
     */
    static valueForSecureStringParameter(scope, parameterName, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromSecureStringParameterAttributes(stack, id, { parameterName, version }).stringValue;
    }
}
exports.StringParameter = StringParameter;
/**
 * Creates a new StringList SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringListParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.stringListValue.find(str => !core_1.Token.isUnresolved(str) && str.indexOf(',') !== -1)) {
            throw new Error('Values of a StringList SSM Parameter cannot contain the \',\' character. Use a string parameter instead.');
        }
        if (props.allowedPattern && !core_1.Token.isUnresolved(props.stringListValue)) {
            props.stringListValue.forEach(str => _assertValidValue(str, props.allowedPattern));
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: ParameterType.STRING_LIST,
            value: props.stringListValue.join(','),
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringListValue = core_1.Fn.split(',', resource.attrValue);
    }
    /**
     * Imports an external parameter of type string list.
     * Returns a token and should not be parsed.
     */
    static fromStringListParameterName(scope, id, stringListParameterName) {
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = stringListParameterName;
                this.parameterArn = util_1.arnForParameterName(this, this.parameterName);
                this.parameterType = ParameterType.STRING_LIST;
                this.stringListValue = core_1.Fn.split(',', new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, stringListParameterName).toString());
            }
        }
        return new Import(scope, id);
    }
}
exports.StringListParameter = StringListParameter;
/**
 * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
 *
 * @param value          the value to be validated.
 * @param allowedPattern the regular expression to use for validation.
 *
 * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
 *         ``cdk.unresolved``).
 */
function _assertValidValue(value, allowedPattern) {
    if (core_1.Token.isUnresolved(value) || core_1.Token.isUnresolved(allowedPattern)) {
        // Unable to perform validations against unresolved tokens
        return;
    }
    if (!new RegExp(allowedPattern).test(value)) {
        throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
    }
}
function makeIdentityForImportedValue(parameterName) {
    return `SsmParameterValue:${parameterName}:C96584B6-F00A-464E-AD19-53AFF4B05118`;
}
//# sourceMappingURL=data:application/json;base64,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