"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1)
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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