"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => ({ ...acc, [k]: v.conditions }), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
            redrivePolicy: this.buildDeadLetterConfig(this.deadLetterQueue),
        });
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue;
        deadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [deadLetterQueue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new aws_iam_1.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': props.topic.topicArn },
            },
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                deadLetterTargetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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