"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sns = require("../../aws-sns"); // Automatically re-written from '@aws-cdk/aws-sns'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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