"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy,
            },
            ruleSetName: props.ruleSet.receiptRuleSetName,
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16',
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID,
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        },
                    }),
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet,
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log('SES Notification:\n', JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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